%
% This file is part of AtomVM.
%
% Copyright 2018-2023 Fred Dushin <fred@dushin.net>
%
% Licensed under the Apache License, Version 2.0 (the "License");
% you may not use this file except in compliance with the License.
% You may obtain a copy of the License at
%
%    http://www.apache.org/licenses/LICENSE-2.0
%
% Unless required by applicable law or agreed to in writing, software
% distributed under the License is distributed on an "AS IS" BASIS,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
% See the License for the specific language governing permissions and
% limitations under the License.
%
% SPDX-License-Identifier: Apache-2.0 OR LGPL-2.1-or-later
%

%%-----------------------------------------------------------------------------
%% @doc AtomVM-specific APIs
%%
%% This module contains functions that are specific to the AtomVM platform.
%% @end
%%-----------------------------------------------------------------------------
-module(atomvm).

-export([
    platform/0,
    random/0,
    rand_bytes/1,
    read_priv/2,
    add_avm_pack_binary/2,
    add_avm_pack_file/2,
    close_avm_pack/2
]).

-type platform_name() ::
    generic_unix
    | emscripten
    | esp32
    | pico
    | stm32.

-type avm_path() :: string() | binary().

%%-----------------------------------------------------------------------------
%% @returns The platform name.
%% @doc     Return the platform moniker.
%%          You may use this function to uniquely identify the platform
%%          type on which your application is running.
%% @end
%%-----------------------------------------------------------------------------
-spec platform() -> platform_name().
platform() ->
    erlang:nif_error(undefined).

%%-----------------------------------------------------------------------------
%% @returns random 32-bit integer.
%% @doc     Returns a random 32-bit integer value.
%%          This function will use a cryptographically strong RNG if available.
%%          Otherwise, the random value is generated using a PRNG.
%% @end
%%-----------------------------------------------------------------------------
-spec random() -> integer().
random() ->
    erlang:nif_error(undefined).

%%-----------------------------------------------------------------------------
%% @param   Len non-negative integer
%% @returns Binary containing random sequence of bytes of length Len.
%% @doc     Returns a binary containing random sequence of bytes of length Len.
%%          Supplying a negative value will result in a badarg error.
%%          This function will use a cryptographically strong RNG if available.
%%          Otherwise, the random value is generated using a PRNG.
%% @end
%%-----------------------------------------------------------------------------
-spec rand_bytes(Len :: non_neg_integer()) -> binary().
rand_bytes(_Len) ->
    erlang:nif_error(undefined).

%%-----------------------------------------------------------------------------
%% @param   App application name.
%% @param   Path path to the resource.
%% @returns Binary containing the resource content.
%% @doc     This function allows to fetch priv/ resources content.
%% @end
%%-----------------------------------------------------------------------------
-spec read_priv(App :: atom(), Path :: list()) -> binary().
read_priv(_App, _Path) ->
    erlang:nif_error(undefined).

%%-----------------------------------------------------------------------------
%% @param   AVMData AVM data.
%% @param   Options Options, as a property list.
%% @returns `ok'
%% @doc     Add code from an AVM binary to your application.
%%
%%          This function will add the data in the `AVMData' parameter to
%%          your application.  The data is assumed to be valid AVM data (e.g, as
%%          generated by packbeam tooling).
%%
%%          Failure to properly load AVM data is result in a runtime `error'
%% @end
%%-----------------------------------------------------------------------------
-spec add_avm_pack_binary(AVMData :: binary(), Options :: [{name, Name :: atom()}]) -> ok.
add_avm_pack_binary(_AVMData, _Options) ->
    erlang:nif_error(undefined).

%%-----------------------------------------------------------------------------
%% @param   AVMPath Path to AVM data.
%% @param   Options Options, as a property list.
%% @returns `ok'
%% @doc     Add code from an AVM binary to your application.
%%
%%          This function will add the data located in the `AVMPath' parameter to
%%          your application.  The data is assumed to be valid AVM data (e.g, as
%%          generated by packbeam tooling).
%%
%%          On `generic_unix' platforms, the `AVMPath' may be a valid file system
%%          path to an AVM file.
%%
%%          On `esp32' platforms, the `AVMPath' should be the name of an ESP32
%%          flash partition, prefixed with the string
%%          `/dev/partition/by-name/'.  Thus, for example, if you specify
%%          `/dev/partition/by-name/main2.app' as the `AVMPath', the ESP32
%%          flash should contain a data partition with the
%%          name `main2.app'
%%
%%          Failure to properly load AVM path is result in a runtime `error'
%% @end
%%-----------------------------------------------------------------------------
-spec add_avm_pack_file(AVMPath :: avm_path(), Options :: []) -> ok.
add_avm_pack_file(_AVMPath, _Options) ->
    erlang:nif_error(undefined).

%%-----------------------------------------------------------------------------
%% @param   name the AVM name.
%% @param   Options Options, as a property list.
%% @returns `ok | error'
%% @doc     Close previously opened AVM binary from your application.
%%
%%          This function will close the data referenced by the `Name' parameter from
%%          your application.  The `Name' parameter must reference previously
%%          opened AVM data.
%%
%%          Failure to close AVM data is result in a runtime `error'
%% @end
%%-----------------------------------------------------------------------------
-spec close_avm_pack(Name :: atom(), Options :: []) -> ok | error.
close_avm_pack(_Name, _Options) ->
    erlang:nif_error(undefined).
