!! Copyright (C) 2004-2013 M. Oliveira, F. Nogueira
!! Copyright (C) 2016 M. Oliveira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module xc_m
  use iso_c_binding
  use global_m
  use oct_parser_m
  use messages_m
  use units_m
  use io_m
  use output_m
  use mesh_m
  use xc_f03_lib_m
  use functionals_m
  use ps_io_m
  use states_batch_m
  implicit none


                    !---Interfaces---!

  interface assignment (=)
     module procedure xc_copy
  end interface


                    !---Derived Data Types---!

  type xc_t
    private
    integer :: nspin
    type(functional_t) :: functls(2)
    integer :: ids(2)
    integer :: corrections

    ! Approximation for the kinetic energy density
    integer :: ked_approximation
    type(functional_t) :: ked_functl

    !Non-linear core-corrections
    !This is not initialized in xc_init but in xc_nlcc_init
    integer  :: nlcc_scheme
    real(R8) :: rc
    integer  :: np
    real(R8), pointer :: rho_core(:)
    real(R8), pointer :: grad_core(:)
    real(R8), pointer :: lapl_core(:)
    real(R8), pointer :: tau_core(:)
  end type xc_t


                    !---Global Variables---!

  !Input parameters for XCFunctionals block
  integer, parameter :: XC_DERIV_METHOD   = 1000, &
                        XC_PARAMETERS     = 1001, &
                        XC_DENS_THRESHOLD = 1002
  
  !Core-correcton scheme
  integer, parameter :: CC_NONE  = 0, &
                        CC_TM    = 1, &
                        CC_FHI   = 2

  !Corrections to the xc functionals
  !Note: this is a flag, so the values should be powers of 2
  integer, parameter :: XC_CORRECTION_NONE  =  0, &
                        XC_CORRECTION_ADSIC =  1, &
                        XC_CORRECTION_RHOXC =  2

  !Exact kinetic energy density
  integer, parameter :: XC_KED_EXACT = 0


                    !---Public/Private Statements---!

  private
  public :: xc_t, &
            xc_null, &
            xc_init, &
            xc_nlcc_init, &
            assignment(=), &
            xc_end, &
            xc_model_output_info, &
            xc_model_save, &
            xc_model_load, &
            xc_potential, &
            xc_exx_mixing, &
            xc_evaluate_ked_approximation, & 
            xc_output, &
            xc_ps_io_set, &
            XC_DERIV_METHOD, &
            XC_PARAMETERS, &
            XC_DENS_THRESHOLD, &
            CC_NONE, &
            CC_TM, &
            CC_FHI, &
            XC_CORRECTION_NONE,  &
            XC_CORRECTION_ADSIC, &
            XC_CORRECTION_RHOXC

contains

  !-----------------------------------------------------------------------
  !> Nullifies and sets to zero all the components of xc_model.           
  !-----------------------------------------------------------------------
  subroutine xc_null(xc_model)
    type(xc_t), intent(out) :: xc_model

    integer :: i

    call push_sub("xc_null")

    xc_model%nspin = 0
    do i = 1, 2
      call functional_null(xc_model%functls(i))
    end do
    xc_model%ids = 0
    xc_model%corrections = 0
    xc_model%ked_approximation = 0
    call functional_null(xc_model%ked_functl)

    xc_model%nlcc_scheme = CC_NONE
    xc_model%rc     = M_ZERO
    xc_model%np     = 0
    nullify(xc_model%rho_core)
    nullify(xc_model%grad_core)
    nullify(xc_model%lapl_core)
    nullify(xc_model%tau_core)

    call pop_sub()
  end subroutine xc_null

  !-----------------------------------------------------------------------
  !> Initializes exchange-correlation model data. The exchange-correlation
  !> models are read from the input file.                                 
  !-----------------------------------------------------------------------
  subroutine xc_init(nspin, xc_model)
    integer,    intent(in)    :: nspin
    type(xc_t), intent(inout) :: xc_model
    
    type xc_input_data
      integer  :: id
      integer  :: deriv_method
      integer  :: n_parameters
      real(R8), allocatable :: parameters(:)
      real(R8) :: dens_threshold
    end type xc_input_data
    
    type(c_ptr) :: blk
    integer :: ifunc, icol, ierr, n_funcs, n_cols, flag, ipar
    type(xc_input_data), allocatable :: func_input(:)
    
    call push_sub("xc_init")
    
    xc_model%nspin = nspin

    !Open the xc functionals block and get the number of lines
    ierr = oct_parse_f90_block("XCFunctionals", blk)
    if (ierr /= 0) then
      ! No XCFunctionals block: use default functionals     
      n_funcs = 2
      allocate(func_input(n_funcs))

      ! LDA exchange
      func_input(1)%id =  XC_LDA_X
      func_input(1)%deriv_method = XC_DERIV_NUMERICAL
      func_input(1)%n_parameters = 0

      ! Perdew-Wang LDA correlation
      func_input(2)%id =  XC_LDA_C_PW
      func_input(2)%deriv_method = XC_DERIV_NUMERICAL
      func_input(2)%n_parameters = 0

    else
      n_funcs = oct_parse_f90_block_n(blk)
      if (n_funcs > 2) then
        message(1) = "More than two functionals specified in the XCFunctionals block."
        call write_fatal(1)
      end if

      allocate(func_input(2))

      ! Defaults
      func_input%id = 0
      func_input%deriv_method = XC_DERIV_NUMERICAL
      func_input%n_parameters = 0
      func_input%dens_threshold = M_ZERO

      do ifunc = 1, n_funcs
        n_cols = oct_parse_f90_block_cols(blk, ifunc-1)
        ierr = oct_parse_f90_block_int(blk, ifunc-1, 0, func_input(ifunc)%id)
        if (ierr /= 0) then
          write(message(1),'("Unable to read functional id from line ",I2," and column 1 of the XCFunctionals block.")') ifunc
          call write_fatal(1)
        end if

        icol = 2
        do while (icol <= n_cols)          
          ierr = oct_parse_f90_block_int(blk, ifunc-1, icol-1, flag)
          if (ierr /= 0) then
            write(message(1),'("Unable to read an integer from line ",I2," and column ",I2," of the XCFunctionals block.")') &
              ifunc, icol
            call write_fatal(1)
          end if

          select case (flag)
          case (XC_DERIV_METHOD)
            ierr = oct_parse_f90_block_int(blk, ifunc-1, icol, func_input(ifunc)%deriv_method)
            if (ierr /= 0) then
              write(message(1),'("Unable to read an integer from line ",I2," and column ",I2," of the XCFunctionals block.")') &
                ifunc, icol + 1
              call write_fatal(1)
            end if
            icol = icol + 2

          case (XC_PARAMETERS)
            func_input(ifunc)%n_parameters = n_cols - icol
            allocate(func_input(ifunc)%parameters(func_input(ifunc)%n_parameters))
            do ipar = 1, func_input(ifunc)%n_parameters
              ierr = oct_parse_f90_block_double(blk, ifunc-1, icol+ipar-1, func_input(ifunc)%parameters(ipar))
              if (ierr /= 0) then
                write(message(1),'("Unable to read a real from line ",I2," and column ",I2," of the XCFunctionals block.")') &
                  ifunc, icol + ipar
                call write_fatal(1)
              end if
            end do
            icol = n_cols + 1            

          case (XC_DENS_THRESHOLD)
            ierr = oct_parse_f90_block_double(blk, ifunc-1, icol, func_input(ifunc)%dens_threshold)
            if (ierr /= 0) then
              write(message(1),'("Unable to read an real from line ",I2," and column ",I2," of the XCFunctionals block.")') &
                ifunc, icol + 1
              call write_fatal(1)
            end if
            icol = icol + 2
            
          case default
            write(message(1),'("Unknown flag at line ",I2," and column ",I2," of the XCFunctionals block.")') ifunc, icol
            call write_fatal(1)

          end select
        end do
 
      end do
      
      !Close the XCFunctionals block
      call oct_parse_f90_block_end(blk)
    end if

    ! Initialize functionals
    do ifunc = 1, 2
      xc_model%ids(ifunc) = func_input(ifunc)%id
      call functional_init(nspin, func_input(ifunc)%id, func_input(ifunc)%deriv_method, func_input(ifunc)%dens_threshold, &
                           func_input(ifunc)%n_parameters, func_input(ifunc)%parameters, xc_model%functls(ifunc))
    end do

    ! Deallocate memory
    do ifunc = 1, 2
      if (allocated(func_input(ifunc)%parameters)) then
        deallocate(func_input(ifunc)%parameters)
      end if
    end do
    deallocate(func_input)

    
    !We cannot have two hybrid functionals at once:
    if (functional_exx_mixing(xc_model%functls(1)) /= M_ZERO .and. functional_exx_mixing(xc_model%functls(2)) /= M_ZERO) then
      message(1) = "Only one functional can be an hybrid functional."
      call write_fatal(1)
    end if
    
    !Read exchange-correlation corrections
    xc_model%corrections = oct_parse_f90_int('XCCorrections', XC_CORRECTION_NONE)
    
    !rhoxc correction should be applied only when we have an exchange functional
    if (iand(xc_model%corrections, XC_CORRECTION_RHOXC) /= 0 .and. &
      .not. ( functional_kind(xc_model%functls(1)) == XC_EXCHANGE_CORRELATION .or. &
              functional_kind(xc_model%functls(1)) == XC_EXCHANGE .or. &
              functional_kind(xc_model%functls(2)) == XC_EXCHANGE_CORRELATION .or. &
              functional_kind(xc_model%functls(2)) == XC_EXCHANGE)) then
       message(1) = "rhoxc correction should be applied to an exchange functional."
      call write_fatal(1)
    end if
 
    !Approximations for kinetic energy density
    xc_model%ked_approximation = oct_parse_f90_int('KEDFunctional', XC_KED_EXACT)
    select case (xc_model%ked_approximation)
    case (XC_KED_EXACT)
    case default
      call functional_init(nspin, xc_model%ked_approximation, XC_DERIV_NUMERICAL, 0.0_r8, 0, (/0.0_r8/), xc_model%ked_functl)
    end select

    call pop_sub()
  end subroutine xc_init

  !-----------------------------------------------------------------------!
  !> Initializes the non-local core-corrections part of the xc model.
  !> The following schemes are available:
  !>
  !>  * 4th order even polynomial used in atom code by Jose Luis Martin (cc_scheme = CC_TM)
  !>  * 6th order polynomial used in fhi98PP by Martin Fuchs (cc_scheme = CC_FHI)
  !-----------------------------------------------------------------------!
  subroutine xc_nlcc_init(xc_model, m, rc, cc_scheme, core_density)
    type(xc_t),   intent(inout) :: xc_model                           !< exchange-correlation model
    type(mesh_t), intent(in)    :: m                                  !< mesh
    real(R8),     intent(inout) :: rc                                 !< core cut-off radius
    integer,      intent(in)    :: cc_scheme                          !< core-correction scheme to used
    real(R8),     intent(in)    :: core_density(m%np, xc_model%nspin) !< the all electron core density

    integer :: i, i_rc
    logical :: monotonous
    real(R8) :: cd_rc, cdp_rc, cdpp_rc, cdppp_rc, a, b, c, d, e, f, g, dummy, dummy1
    real(R8), allocatable :: cd(:)

    xc_model%nlcc_scheme = cc_scheme
    xc_model%rc   = rc
    xc_model%np   = m%np
    allocate(xc_model%rho_core(m%np),  xc_model%grad_core(m%np), &
             xc_model%lapl_core(m%np), xc_model%tau_core(m%np))

    !Get first, second, and third derivatives of the core density
    allocate(cd(m%np))
    cd = sum(core_density,2)
    cd_rc   = mesh_extrapolate(m, cd, rc)
    cdp_rc  = mesh_extrapolate(m, mesh_derivative(m, cd), rc)
    cdpp_rc = mesh_extrapolate(m, mesh_derivative2(m, cd), rc)    
    cdppp_rc = mesh_extrapolate(m, mesh_derivative3(m, cd), rc)

    select case (xc_model%nlcc_scheme)
    case (CC_TM)
      !Get parameters
      c = (rc*(cdpp_rc/cd_rc - (cdp_rc/cd_rc)**2) - cdp_rc/cd_rc)/(M_EIGHT*rc**3)
      b = (cdp_rc/cd_rc - M_FOUR*c*rc**3)/(M_TWO*rc)
      a = log(cd_rc) - b*rc**2 - c*rc**4

      !Compute partial core-density    
      do i = 1, m%np
        if (m%r(i) < rc) then
          xc_model%rho_core(i) = exp(a + b*m%r(i)**2 + c*m%r(i)**4)
        else
          xc_model%rho_core(i) = cd(i)
        end if
        if ( abs(xc_model%rho_core(i)) < M_EPSILON ) xc_model%rho_core(i)=M_ZERO
      end do

    case (CC_FHI)
      ! Insert a 4*Pi factor in the charge density and its derivatives
      cd_rc    = cd_rc*M_FOUR*M_PI
      cdp_rc   = cdp_rc*M_FOUR*M_PI
      cdpp_rc  = cdpp_rc*M_FOUR*M_PI
      cdppp_rc = cdppp_rc*M_FOUR*M_PI

      ! Find i_rc
      do i = 1, m%np
        if (m%r(i) > rc) exit
      end do
      i_rc = i

      ! Find the 5 non-zero coefficients of the polynomial
      ! Start with a = cd_rc
      a = cd_rc
      do
        ! Solve linear part
        d = M_TWENTY*(cd_rc - a)/rc**3 - M_TEN*cdp_rc/rc**2 + &
             M_TWO*cdpp_rc/rc - cdppp_rc/M_SIX
        e = (90.0_R8*(a - cd_rc) + M_FIFTY*rc*cdp_rc - M_ELEVEN*rc**2*cdpp_rc + &
             rc**3*cdppp_rc)/(M_TWO*rc**4)
        f = (72.0_R8*(cd_rc - a) - 42.0_R8*rc*cdp_rc + M_TEN*rc**2*cdpp_rc - &
             rc**3*cdppp_rc)/(M_TWO*rc**5)
        g = (M_SIXTY*(a - cd_rc) + 36.0_R8*rc*cdp_rc - M_NINE*rc**2*cdpp_rc + &
             rc**3*cdppp_rc)/(M_SIX*rc**6)

        ! Test monotonous decay
        dummy1 = a + d*m%r(i_rc)**3 + e*m%r(i_rc)**4 + &
                 f*m%r(i_rc)**5 + g*m%r(i_rc)**6
        monotonous = .true.
        do i = i_rc, 2, -1
          dummy = dummy1
          dummy1 = a + d*m%r(i-1)**3 + e*m%r(i-1)**4 + &
                   f*m%r(i-1)**5 + g*m%r(i-1)**6
          if (dummy1 < dummy) then
            monotonous = .false.
            exit
          end if
        end do
        if (monotonous) exit
        a = a*1.25_R8
      end do

      !Compute partial core-density    
      do i = 1, m%np
        if (m%r(i) < rc) then
          xc_model%rho_core(i) = (a + d*m%r(i)**3 + e*m%r(i)**4 + &
                                     f*m%r(i)**5 + g*m%r(i)**6)/M_FOUR/M_PI
        else
          xc_model%rho_core(i) = cd(i)
        end if
        if ( abs(xc_model%rho_core(i)) < M_EPSILON ) xc_model%rho_core(i) = M_ZERO
      end do

    end select

    !Higher order derivatives of the core density
    xc_model%grad_core = mesh_gradient(m, xc_model%rho_core)
    xc_model%lapl_core = mesh_laplacian(m, xc_model%rho_core)
    xc_model%tau_core = xc_model%grad_core**2/xc_model%rho_core/M_FOUR
    where(xc_model%rho_core <= 1e-30)
      xc_model%rho_core = M_ZERO
      xc_model%grad_core = M_ZERO
      xc_model%lapl_core = M_ZERO
      xc_model%tau_core = M_ZERO
    end where

    !Output info
    write(info_unit("pp"),*)
    message(1) = ""
    message(2) = "Core Corrections:"
    select case (cc_scheme)
    case (CC_TM)
      message(3) = "  Scheme: Troullier-Martins scheme" 
      write(message(4),'(2x,a,f12.6)') "rc    :", rc
      write(message(5),'(2x,a)') "Polynomial coefficients:"
      write(message(6),'(4x,a,f16.10)') "a =", a + log(M_FOUR*M_PI)
      write(message(7),'(4x,a,f16.10)') "b =", b
      write(message(8),'(4x,a,f16.10)') "c =", c
      write(message(9),'(2x,a)') "Matching at rc:"
      write(message(10),'(4x,a,f15.9,a,f15.9)') "cd(rc)   |  f0(rc)  =", &
           cd_rc,   "   | ", exp(a + b*rc**2 + c*rc**4)
      write(message(11),'(4x,a,f15.9,a,f15.9)') "cd'(rc)  |  f1(rc)  =", &
           cdp_rc,  "   | ", (M_TWO*b*rc + M_FOUR*c*rc**3)*exp(a + b*rc**2 + c*rc**4)
      write(message(12),'(4x,a,f15.9,a,f15.9)') "cd''(rc) |  f2(rc)  =", &
           cdpp_rc, "   | ", M_TWO*exp(a + b*rc**2 + c*rc**4)*(b + M_TWO*(b**2 + 3*c)*rc**2 + &
           M_EIGHT*b*c*rc**4 + M_EIGHT*c**2*rc**6)
      write(message(13),'(2x,a,f16.9)') "Integrated core charge: ", &
           M_FOUR*M_PI*mesh_integrate(m, xc_model%rho_core)
      call write_info(13,20)
      call write_info(13,unit=info_unit("pp"))

    case (CC_FHI)
      message(3) = "  Scheme: FHI scheme" 
      write(message(4),'(2x,a,f12.6)') "rc    :", rc
      write(message(5),'(2x,a)') "Polynomial coefficients:"
      write(message(6),'(4x,a,f16.10)') "a =", a
      write(message(7),'(4x,a,f16.10)') "b =", M_ZERO
      write(message(8),'(4x,a,f16.10)') "c =", M_ZERO
      write(message(9),'(4x,a,f16.10)') "d =", d
      write(message(10),'(4x,a,f16.10)') "e =", e
      write(message(11),'(4x,a,f16.10)') "f =", f
      write(message(12),'(4x,a,f16.10)') "g =", g
      write(message(13),'(2x,a)') "Matching at rc:"
      write(message(14),'(4x,a,f15.9,a,f15.9)') "cd(rc)    |  f0(rc)  =", &
           cd_rc,    "   | ", a + d*rc**3 + e*rc**4 + f*rc**5 + g*rc**6
      write(message(15),'(4x,a,f15.9,a,f15.9)') "cd'(rc)   |  f1(rc)  =", &
           cdp_rc,   "   | ", M_THREE*d*rc**2 + M_FOUR*e*rc**3 + M_FIVE*f*rc**4 + M_SIX*g*rc**5
      write(message(16),'(4x,a,f15.9,a,f15.9)') "cd''(rc)  |  f2(rc)  =", &
           cdpp_rc,  "   | ", M_SIX*d*rc + M_TWELVE*e*rc**2 + M_TWENTY*f*rc**3 + M_THIRTY*g*rc**4
      write(message(17),'(4x,a,f15.9,a,f15.9)') "cd'''(rc) |  f3(rc)  =", &
           cdppp_rc, "   | ", M_SIX*d + 24.0_r8*e*rc + M_SIXTY*f*rc**2 + 120.0_r8*g*rc**3
      write(message(18),'(2x,a,f15.9)') "Integrated core charge: ", &
           M_FOUR*M_PI*mesh_integrate(m, xc_model%rho_core)
      call write_info(18,20)
      call write_info(18,unit=info_unit("pp"))
    end select

  end subroutine xc_nlcc_init

  !-----------------------------------------------------------------------
  !> Copies xc model xc_in to xc model xc_out.                            
  !-----------------------------------------------------------------------
  subroutine xc_copy(xc_out, xc_in)
    type(xc_t), intent(inout) :: xc_out
    type(xc_t), intent(in)    :: xc_in

    integer :: i

    call push_sub("xc_copy")

    call xc_end(xc_out)

    xc_out%nspin = xc_in%nspin
    do i = 1, 2
      xc_out%functls(i) = xc_in%functls(i)
    end do
    xc_out%ids = xc_in%ids
    xc_out%corrections = xc_in%corrections
    xc_out%ked_approximation = xc_in%ked_approximation
    xc_out%ked_functl = xc_in%ked_functl

    xc_out%nlcc_scheme  = xc_in%nlcc_scheme
    xc_out%rc    = xc_in%rc
    xc_out%np    = xc_in%np
    if (associated(xc_in%rho_core)) then
      allocate(xc_out%rho_core(xc_out%np))
      xc_out%rho_core = xc_in%rho_core
    end if
    if (associated(xc_in%grad_core)) then
      allocate(xc_out%grad_core(xc_out%np))
      xc_out%grad_core = xc_in%grad_core
    end if
    if (associated(xc_in%lapl_core)) then
      allocate(xc_out%lapl_core(xc_out%np))
      xc_out%lapl_core = xc_in%lapl_core
    end if
    if (associated(xc_in%tau_core)) then
      allocate(xc_out%tau_core(xc_out%np))
      xc_out%tau_core = xc_in%tau_core
    end if

    call pop_sub()
  end subroutine xc_copy

  !-----------------------------------------------------------------------
  !> Frees all memory associated to the xc_model.                         
  !-----------------------------------------------------------------------
  subroutine xc_end(xc_model)
    type(xc_t), intent(inout) :: xc_model

    integer :: i

    call push_sub("xc_end")

    xc_model%nspin = 0
    do i = 1, 2
      call functional_end(xc_model%functls(i))
    end do
    xc_model%ids = 0
    xc_model%corrections = 0

    xc_model%ked_approximation = 0
    call functional_end(xc_model%ked_functl)

    xc_model%nlcc_scheme = CC_NONE
    xc_model%rc   = M_ZERO
    xc_model%np   = 0    
    if (associated(xc_model%rho_core)) deallocate(xc_model%rho_core)
    if (associated(xc_model%grad_core)) deallocate(xc_model%grad_core)
    if (associated(xc_model%lapl_core)) deallocate(xc_model%lapl_core)
    if (associated(xc_model%tau_core)) deallocate(xc_model%tau_core)

    call pop_sub()
  end subroutine xc_end

  !-----------------------------------------------------------------------
  !> Writes the exchange-correlation model data to a file.                
  !-----------------------------------------------------------------------
  subroutine xc_model_save(unit, xc_model)
    integer,    intent(in) :: unit
    type(xc_t), intent(in) :: xc_model

    integer :: i

    call push_sub("xc_model_save")

    write(unit) xc_model%nspin
    do i = 1, 2
      call functional_save(unit, xc_model%functls(i))
    end do
    write(unit) xc_model%ids, xc_model%corrections, xc_model%ked_approximation
    call functional_save(unit, xc_model%ked_functl)

    write(unit) xc_model%nlcc_scheme
    if (xc_model%nlcc_scheme /= CC_NONE) then
      write(unit) xc_model%rc, xc_model%np
      write(unit) (xc_model%rho_core(i), i=1,xc_model%np)
      write(unit) (xc_model%grad_core(i), i=1,xc_model%np)
      write(unit) (xc_model%lapl_core(i), i=1,xc_model%np)
      write(unit) (xc_model%tau_core(i), i=1,xc_model%np)
    end if

    call pop_sub()
  end subroutine xc_model_save

  !-----------------------------------------------------------------------
  !> Reads the exchange-correlation model data from a file.               
  !-----------------------------------------------------------------------
  subroutine xc_model_load(unit, xc_model)
    integer,    intent(in) :: unit
    type(xc_t), intent(inout) :: xc_model

    integer :: i

    call push_sub("xc_model_load")

    read(unit) xc_model%nspin
    do i = 1, 2
      call functional_load(unit, xc_model%functls(i))
    end do
    read(unit) xc_model%ids, xc_model%corrections, xc_model%ked_approximation
    call functional_load(unit, xc_model%ked_functl)

    read(unit) xc_model%nlcc_scheme
    if (xc_model%nlcc_scheme /= CC_NONE) then
      read(unit) xc_model%rc, xc_model%np
      allocate(xc_model%rho_core(xc_model%np), xc_model%grad_core(xc_model%np), &
               xc_model%lapl_core(xc_model%np), xc_model%tau_core(xc_model%np))
      read(unit) (xc_model%rho_core(i), i=1,xc_model%np)
      read(unit) (xc_model%grad_core(i), i=1,xc_model%np)
      read(unit) (xc_model%lapl_core(i), i=1,xc_model%np)
      read(unit) (xc_model%tau_core(i), i=1,xc_model%np)
    end if

    call pop_sub()
  end subroutine xc_model_load

  !-----------------------------------------------------------------------
  !> Given a set of states, computes the corresponding                    
  !> exchange-correlation potentials and energies.                        
  !-----------------------------------------------------------------------
  subroutine xc_potential(xc_model, m, states, nspin, vxc, exc, vxctau)
    type(xc_t),                     intent(inout) :: xc_model            !< exchange-correlation model
    type(mesh_t),                   intent(in)    :: m                   !< mesh
    integer,                        intent(in)    :: nspin               !< number of spin channels
    type(states_batch_t),           intent(in)    :: states              !< the states
    real(R8),             optional, intent(out)   :: vxc(m%np, nspin)    !< exchange-correlation potential
    real(R8),             optional, intent(out)   :: exc                 !< exchange-correlation energy
    real(R8),             optional, intent(out)   :: vxctau(m%np, nspin) !< extra term arising from MGGA functionals

    integer  :: i
    real(R8) :: eaux, ip(nspin)
    real(R8), allocatable :: e(:), v(:,:), vaux(:,:), vtau(:,:)
    real(R8), allocatable :: rho(:,:), rho_grad(:,:), rho_lapl(:,:), tau(:,:)
    real(R8), allocatable :: dv(:,:)

    call push_sub("xc_potential")

    !Allocate potential and energy work arrays
    allocate(v(m%np, xc_model%nspin), vaux(m%np, nspin), e(m%np))
    allocate(vtau(m%np, xc_model%nspin))
    e = M_ZERO; eaux = M_ZERO; v = M_ZERO; vaux = M_ZERO; vtau = M_ZERO
    if (present(exc))    exc    = M_ZERO
    if (present(vxc))    vxc    = M_ZERO
    if (present(vxctau)) vxctau = M_ZERO

    !Get total electronic density and kinetic energy density
    allocate(rho(m%np, nspin), rho_grad(m%np, nspin), rho_lapl(m%np, nspin), tau(m%np, nspin))
    rho = states_batch_density(states, nspin, m)
    rho_grad = states_batch_density_grad(states, nspin, m)
    rho_lapl = states_batch_density_lapl(states, nspin, m)
    if (xc_model%ked_approximation == XC_KED_EXACT) then
      tau = states_batch_tau(states, nspin, m)
    else
      call functional_get_tau(xc_model%ked_functl, m, rho, rho_grad, rho_lapl, tau)
    end if
    if (xc_model%nlcc_scheme /= CC_NONE) then
      do i = 1, nspin
        rho(:, i)      = rho(:, i)      + xc_model%rho_core/real(nspin,R8)
        rho_grad(:, i) = rho_grad(:, i) + xc_model%grad_core/real(nspin,R8)
        rho_lapl(:, i) = rho_lapl(:, i) + xc_model%lapl_core/real(nspin,R8)
        tau(:, i)      = tau(:, i)      + xc_model%tau_core/real(nspin,R8)
      end do
    end if

    !Get ionization potential
    ip = states_batch_ip(states, nspin)
 
    !Get energy and potential
    do i = 1, 2
      call functional_get_vxc(xc_model%functls(i), m, rho, rho_grad, rho_lapl, &
                              tau, ip, v, e, vtau)
      if (present(exc)) then
        exc = exc + M_FOUR*M_PI*mesh_integrate(m, e)
      end if
      if (present(vxc)) then
        vxc = vxc + v
      end if
      if (present(vxctau)) then
        vxctau = vxctau + vtau
      end if

      if (iand(xc_model%corrections, XC_CORRECTION_RHOXC) /= 0 .and. present(vxc)) then
        if ( functional_kind(xc_model%functls(i)) == XC_EXCHANGE .or. &
             functional_kind(xc_model%functls(i)) == XC_EXCHANGE_CORRELATION) then
          allocate(dv(m%np, nspin))
          call functional_rhoxc(xc_model%functls(i), m, nspin, rho, rho_grad, rho_lapl, tau, ip, dv)
          vxc = vxc + dv
          deallocate(dv)
        end if
      end if
    end do

    if (iand(xc_model%corrections, XC_CORRECTION_ADSIC) /= 0) then
      call functional_adsic(xc_model%functls(1:2), m, nspin, rho, rho_grad, &
           rho_lapl, tau, ip, vaux, eaux)
      if (present(vxc)) then
        vxc = vxc - vaux
      end if
      if (present(exc)) then
        exc = exc - eaux
      end if

    end if

    !Deallocate arrays
    deallocate(rho, rho_grad, rho_lapl, tau, e, v, vaux)

    call pop_sub()
  end subroutine xc_potential

  function xc_exx_mixing(xc_model)
    type(xc_t), intent(in) :: xc_model
    real(R8) :: xc_exx_mixing

    ! Only one of the functionals is allowed to have a mixing different from zero.
    ! It is thus OK to sum the mixing coefficient of both functionals. This avoids
    ! having to figure out which functional is an hybrid.
    xc_exx_mixing = functional_exx_mixing(xc_model%functls(1)) + &
                    functional_exx_mixing(xc_model%functls(2))

  end function xc_exx_mixing

  !-----------------------------------------------------------------------
  !> Returns the name of the exchange-correlation functionals.            
  !-----------------------------------------------------------------------
  subroutine xc_model_output_info(xc_model, unit, verbose_limit)
    type(xc_t),           intent(in) :: xc_model
    integer,    optional, intent(in) :: unit
    integer,    optional, intent(in) :: verbose_limit

    integer :: i, n_message

    message(1) = ""
    message(2) = "Exchange-Correlation model:"
    call write_info(2, verbose_limit=verbose_limit, unit=unit)
    
    do i = 1, 2
      call functional_output_info(xc_model%functls(i), unit, verbose_limit)
    end do

    n_message = 0    
    if (iand(xc_model%corrections, XC_CORRECTION_ADSIC) /= 0) then
      n_message = n_message + 1
      write(message(n_message),'(2X,"Self-interaction correction: Averaged-Density SIC")')
    end if
    if (iand(xc_model%corrections, XC_CORRECTION_RHOXC) /= 0) then
      n_message = n_message + 1
      write(message(n_message),'(2X,"Exchange-correlation density correction")')
    end if

    if (xc_model%ked_approximation /= XC_KED_EXACT) then
      n_message = n_message + 1
      write(message(n_message),'(2X,"KED approximation: ",A)') trim(functional_name(xc_model%ked_functl))
    end if
    
    call write_info(n_message, verbose_limit=verbose_limit, unit=unit)

  end subroutine xc_model_output_info

  !-----------------------------------------------------------------------
  !>
  !-----------------------------------------------------------------------
  subroutine xc_evaluate_ked_approximation(xc_model, m, states, nspin, unit)
    type(xc_t),           intent(in) :: xc_model !< exchange-correlation model
    type(mesh_t),         intent(in) :: m        !< mesh
    type(states_batch_t), intent(in) :: states   !< the states
    integer,              intent(in) :: nspin    !< number of spin channels
    integer,              intent(in) :: unit     !< 

    real(R8) :: ek_ks, ek_app
    real(R8), allocatable :: rho(:,:), rho_grad(:,:), rho_lapl(:,:), tau_app(:,:), tau_ks(:,:)

    call push_sub("xc_evaluate_ked_approximation")

    if (xc_model%ked_approximation == XC_KED_EXACT) then
      !Nothing do be done!
      call pop_sub()
      return
    end if

    !Compute Kohn-Sham densities
    allocate(rho(m%np, nspin), rho_grad(m%np, nspin), rho_lapl(m%np, nspin))
    allocate(tau_ks(m%np, nspin), tau_app(m%np, nspin))
    rho = states_batch_density(states, nspin, m)
    rho_grad = states_batch_density_grad(states, nspin, m)
    rho_lapl = states_batch_density_lapl(states, nspin, m)
    tau_ks = states_batch_tau(states, nspin, m)

    !Compute approximate KED
    call functional_get_tau(xc_model%ked_functl, m, rho, rho_grad, rho_lapl, tau_app)

    !Get kinetic energy
    ek_ks  = M_TWO*M_PI*mesh_integrate(m, sum(tau_ks, dim=2))
    ek_app = M_TWO*M_PI*mesh_integrate(m, sum(tau_app, dim=2))

    !Output
    write(message(1),'(2X,"KED Quality Factor: ",F10.3)') M_TWO*M_PI*mesh_integrate(m, sum(abs(tau_ks - tau_app),dim=2))/ek_ks
    if (any(tau_app < M_ZERO)) then
      message(2) = "  KED Negative Values: Yes"
    else
      message(2) = "  KED Negative Values: No"
    end if
    write(message(3),'(2X,"Kinetic Energy [",A,"]:",3X,"Kohn-Sham",5X,"Approximate",3X,"Error (%)")') trim(units_out%energy%abbrev)
    write(message(4),'(2X,17X,F14.6,2X,F14.6,3X,F6.2)') ek_ks, ek_app, (ek_app - ek_ks)/ek_ks*100
    call write_info(4,20)
    call write_info(4,unit=unit)

    deallocate(rho, rho_grad, rho_lapl, tau_app, tau_ks)

    call pop_sub()
  end subroutine xc_evaluate_ked_approximation

  !-----------------------------------------------------------------------
  !> Given a set of states, outputs the the corresponding                 
  !> exchange-correlation potentials and energies.                        
  !-----------------------------------------------------------------------
  subroutine xc_output(xc_model, m, states, nspin, dir)
    type(xc_t),           intent(in) :: xc_model !< exchange-correlation model
    type(mesh_t),         intent(in) :: m        !< mesh
    type(states_batch_t), intent(in) :: states   !< the states
    integer,              intent(in) :: nspin    !< number of spin channels
    character(len=*),     intent(in) :: dir      !< 

    integer  :: if, p, i, unit
    character(20) :: filename
    character(3) :: spin
    character(2) :: type
    type(xc_t) :: xc_tmp
    real(R8), allocatable :: e(:), v(:,:), tau(:,:), vtau(:,:)

    call push_sub("xc_output")

    !Create a temporary xc model
    call xc_null(xc_tmp)
    xc_tmp = xc_model
    do if = 1, 2
      call functional_end(xc_tmp%functls(if))
      call functional_init(nspin, 0, XC_DERIV_NUMERICAL, 0.0_r8, 0, (/0.0_r8/), xc_tmp%functls(if))
    end do

    !Allocate memory
    allocate(v(m%np, nspin), e(m%np), vtau(m%np, nspin))

    !Loop over the functionals
    do if = 1, 2
      if (functional_kind(xc_model%functls(if)) < 0) cycle

      !Get the potential
      xc_tmp%functls(1) = xc_model%functls(if)
      call xc_potential(xc_tmp, m, states, nspin, vxc=v, vxctau=vtau)

      do p = 1, nspin
        !Get the filename
        select case (functional_kind(xc_model%functls(if)))
        case (XC_EXCHANGE)
          type = "x "
        case (XC_CORRELATION)
          type = "c "
        case (XC_EXCHANGE_CORRELATION)
          type = "xc"
        end select
        if (nspin == 2) then
          if (p == 1) then
            spin = "_dn"
          elseif (p == 2) then
            spin = "_up"
          end if
        else
          spin = ""
        end if
        filename = "v_"//trim(type)//trim(spin)

        !Open file
        call io_open(unit, file=trim(dir)//"/"//trim(filename))

        !Write header
        write(unit,'("# ")')
        write(unit,'("# Energy units: ",A)') trim(units_out%energy%name)
        write(unit,'("# Length units: ",A)') trim(units_out%length%name)
        write(unit,'("#")')
        write(unit,'("# ",36("-"))')
        write(unit,'("# |",8X,"r",7X,"|",7X,"v(r)",6X,"|")')
        write(unit,'("# ",36("-"))')

        !Ouput
        do i = 1, m%np
          write(unit,'(4(3X,ES15.8E2))') m%r(i)/units_out%length%factor, &
                                         v(i, p)/units_out%energy%factor
        end do

        close(unit)
        
        !MGGA term
        if (functional_family(xc_model%functls(if)) == XC_FAMILY_MGGA) then
          filename = "vtau_"//trim(type)//trim(spin)

          !Open file
          call io_open(unit, file=trim(dir)//"/"//trim(filename))
          
          !Write header
          write(unit,'("# ")')
          write(unit,'("# Energy units: ",A)') trim(units_out%energy%name)
          write(unit,'("# Length units: ",A)') trim(units_out%length%name)
          write(unit,'("#")')
          write(unit,'("# ",36("-"))')
          write(unit,'("# |",8X,"r",7X,"|",7X,"v(r)",6X,"|")')
          write(unit,'("# ",36("-"))')

          !Ouput
          do i = 1, m%np
            write(unit,'(4(3X,ES15.8E2))') m%r(i)/units_out%length%factor, &
                 vtau(i, p)/units_out%energy%factor
          end do

          close(unit)
        end if

      end do
    end do

    !Deallocate memory
    deallocate(v, e, vtau)
    call xc_end(xc_tmp)

    !NLCC
    if (xc_model%nlcc_scheme /= CC_NONE) then
      call io_open(unit, file=trim(dir)//"/core_correction")

      write(unit,'("#")')
      write(unit,'("# Radial core correction density and derivatives.")')
      write(unit,'("# Length units: ",A)') trim(units_out%length%name)
      write(unit,'("#")')

      write(unit,'("# ",88("-"))')
      write(unit,'("# |",7X,"r",7X,"|",6X,"n(r)",7X,"|",4X,"grad_n(r)",4X,"|",4X,"lapl_n(r)",4X,"|",5X,"tau(r)",6X,"|")')
      write(unit,'("# ",88("-"))')
      do i = 1, m%np
        write(unit,'(3X,ES14.8E2,3X,ES15.8E2,3X,ES15.8E2,3X,ES15.8E2,3X,ES15.8E2)') &
             m%r(i)/units_out%length%factor, &
             xc_model%rho_core(i)*units_out%length%factor, &
             xc_model%grad_core(i)*units_out%length%factor**2, &
             xc_model%lapl_core(i)*units_out%length%factor**3, &
             xc_model%tau_core(i)*units_out%length%factor/units_out%energy%factor
      end do

      close(unit)

    end if

    if (xc_model%ked_approximation /= XC_KED_EXACT) then
      
      allocate(tau(m%np, nspin))
      call functional_get_tau(xc_model%ked_functl, m, &
           states_batch_density(states, nspin, m), &
           states_batch_density_grad(states, nspin, m), &
           states_batch_density_lapl(states, nspin, m), tau)

      do p = 1, nspin
        if (nspin == 1) then
          filename = trim(dir)//"/app_tau"
        else
          if (p == 1) then
            filename = trim(dir)//"/app_tau_dn"
          elseif (p == 2) then
            filename = trim(dir)//"/app_tau_up"
          end if
        end if
        call io_open(unit, file=trim(filename))

        write(unit,'("#")')
        write(unit,'("# Radial approximated kinetic energy density.")')
        write(unit,'("# Length units: ",A)') trim(units_out%length%name)
        write(unit,'("# Energy units: ",A)') trim(units_out%energy%name)
        write(unit,'("#")')

        write(unit,'("# ",35("-"))')
        write(unit,'("# |",7X,"r",7X,"|",5X,"tau(r)",6X,"|")')
        write(unit,'("# ",35("-"))')
        do i = 1, m%np
          write(unit,'(3X,ES14.8E2,3X,ES15.8E2)') m%r(i)/units_out%length%factor, &
                                 tau(i, p)*units_out%length%factor/units_out%energy%factor
        end do

        close(unit)
      end do
      
      deallocate(tau)
    end if

    call pop_sub()
  end subroutine xc_output

  !-----------------------------------------------------------------------
  !> Pass the information about the exchange-correlation model to the     
  !> ps_io module.                                                        
  !-----------------------------------------------------------------------
  subroutine xc_ps_io_set(xc_model)
    type(xc_t), intent(in) :: xc_model

    call push_sub("xc_ps_io_set")

    call ps_io_set_xc(xc_model%ids)
    
    if (xc_model%nlcc_scheme /= CC_NONE) then
      call ps_io_set_nlcc(xc_model%nlcc_scheme, xc_model%rc, xc_model%np, xc_model%rho_core, xc_model%tau_core)
    end if

    call pop_sub()
  end subroutine xc_ps_io_set

end module xc_m
