!! Copyright (C) 2004-2012 M. Oliveira, F. Nogueira
!! Copyright (C) 2011-2012 T. Cerqueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module quantum_numbers_m
  use global_m
  use messages_m
  implicit none


                    !---Interfaces---!

  interface operator (==)
    module procedure qn_equal
  end interface


                    !---Derived Data Types---!

  type qn_t
    integer  :: n
    integer  :: l
    real(R8) :: j
    real(R8) :: s
    real(R8) :: m
    real(R8) :: sg
    integer  :: k
  end type qn_t


                    !---Global Variables---!

  type(qn_t), parameter :: QN_NULL = qn_t(0, 0, M_ZERO, M_ZERO, M_ZERO, M_ZERO, 0)


                    !---Public/Private Statements---!

  private
  public :: qn_t, &
            QN_NULL, &
            qn_init, &
            qn_number_of_nodes, &
            qn_update_number_of_nodes, &
            qn_label, &
            qn_max_occ, &
            operator(==), &
            qn_wf_dim, &
            qn_equal_fold


contains

  !-----------------------------------------------------------------------
  !> Initialize a set of quantum numbers.                                  
  !-----------------------------------------------------------------------
  function qn_init(n, l, s, j, m, sg)
    integer,            intent(in) :: n
    integer,            intent(in) :: l
    real(R8),           intent(in) :: s
    real(R8), optional, intent(in) :: j
    real(R8), optional, intent(in) :: m
    real(R8), optional, intent(in) :: sg

    type(qn_t) :: qn_init

    call push_sub("qn_init")

    ASSERT(.not. (present(j) .and. present(m)))

    !Set quantum numbers
    qn_init%n = n
    qn_init%l = l
    qn_init%s = s
    if (present(j)) then
      qn_init%j = j
    else
      qn_init%j = M_ZERO
    end if
    if (present(m)) then
      qn_init%m = m
    else
      qn_init%m = M_ZERO
    end if
    if (present(sg)) then
      qn_init%sg = sg
    else
      qn_init%sg = M_ZERO
    end if
    if (qn_init%j /= M_ZERO) then
      qn_init%k = -int(2*(j - real(l,R8))*(j + M_HALF))
    elseif (qn_init%sg == -M_HALF) then
      qn_init%k = - (l + 1)
    elseif (qn_init%sg == M_HALF) then
      qn_init%k = l
    else
      qn_init%k = 0
    end if

    !Check consistency, unless n=0
    if (((l < 0) .or. ( l > n - 1)) .and. n /= 0) then
      message(1) = "Bad angular quantum number in qn_init."
      write(message(2), '("n = ",I2,", l = ",I2)') n, l
      call write_fatal(2)
    end if
    if (qn_init%j /= M_ZERO) then
      if(((j /= l - M_HALF) .and. (j /= l + M_HALF)) .and. n /= 0) then
        message(1) = "Bad total angular quantum number in qn_init."
        write(message(2), '("n = ",I2,", l = ",I2,", j = ",F4.1)') n, l, j
        call write_fatal(2)
      end if
    end if
    if (abs(qn_init%m) == l + M_HALF .and. qn_init%sg /= M_ZERO .and. n /= 0) then
      message(1) = "Bad sigma quantum number in qn_init."
      write(message(2), '("n = ",I2,", m = ",F4.1,", sg = ",F4.1)') n, qn_init%m, qn_init%sg
      call write_fatal(2)
    end if

    call pop_sub()
  end function qn_init

  !-----------------------------------------------------------------------
  !> True if all components of qn_a and qn_b are equal; false otherwise.  
  !-----------------------------------------------------------------------
  elemental function qn_equal(qn_a, qn_b)
    type(qn_t), intent(in) :: qn_a, qn_b
    logical :: qn_equal

    qn_equal = qn_a%n == qn_b%n .and. qn_a%l == qn_b%l .and. &
               qn_a%j == qn_b%j .and. qn_a%s == qn_b%s .and. &
               qn_a%m == qn_b%m .and. qn_a%sg == qn_b%sg

  end function qn_equal

  !-----------------------------------------------------------------------
  !> Returns how many nodes the wavefuntion should have.                  
  !-----------------------------------------------------------------------
  elemental function qn_number_of_nodes(qn)
    type(qn_t), intent(in) :: qn 
    integer :: qn_number_of_nodes

    qn_number_of_nodes = qn%n - qn%l - 1

  end function qn_number_of_nodes

  !-----------------------------------------------------------------------
  !> Update the n quantum number so the wavefunction has a specific number
  !> of nodes.
  !-----------------------------------------------------------------------
  subroutine qn_update_number_of_nodes(qn, number_of_nodes)
    type(qn_t), intent(inout) :: qn 
    integer,    intent(in)    :: number_of_nodes

    qn%n = number_of_nodes + qn%l + 1

  end subroutine qn_update_number_of_nodes
  
  !-----------------------------------------------------------------------
  !> Returns a label that identifies the set of quantum numbers.          
  !-----------------------------------------------------------------------
  function qn_label(qn, full)
    type(qn_t), intent(in) :: qn
    logical,    intent(in), optional :: full

    character(len=10) :: qn_label
    character(len=1)  :: sgn

    select case (qn%l)
    case (0)
      write(qn_label,'(I1,A)') qn%n, "s"
    case (1)
      write(qn_label,'(I1,A)') qn%n, "p"
    case (2)
      write(qn_label,'(I1,A)') qn%n, "d"
    case (3)
      write(qn_label,'(I1,A)') qn%n, "f"
    case default
      write(qn_label,'(I1,A)') qn%n, "x"
    end select

    if (qn%j /= M_ZERO) then
      write(qn_label(3:5),'(I1,".5")') int(qn%j - M_HALF)
    elseif (qn%m /= M_ZERO) then
      sgn = '-'
      if (qn%m > 0) sgn = '+'
      write(qn_label(3:6),'(A1,I1,".5")') sgn, int(abs(qn%m) - M_HALF)
    end if
   
    if (present(full)) then
      if (full) then
        if (qn%s == M_HALF .or. qn%sg == M_HALF) then
          qn_label = trim(qn_label)//"_up"
        elseif (qn%s == -M_HALF .or. qn%sg == -M_HALF) then
          qn_label = trim(qn_label)//"_dn"
        end if
      end if
    end if

  end function qn_label

  !-----------------------------------------------------------------------
  !> Returns the maximum occupancy for a given set of quantum numbers.    
  !-----------------------------------------------------------------------
  elemental function qn_max_occ(qn)
    type(qn_t), intent(in) :: qn
    real(R8) :: qn_max_occ
  
    if (qn%j == M_ZERO .and. qn%m == M_ZERO) then
      if (qn%s == M_ZERO) then
        qn_max_occ = M_TWO*(M_TWO*qn%l + M_ONE)
      else
        qn_max_occ = M_TWO*qn%l + M_ONE
      end if
    else if (qn%m /= M_ZERO) then
      qn_max_occ = M_ONE
    else
      qn_max_occ = M_TWO*qn%j + M_ONE
    end if

  end function qn_max_occ

  !-----------------------------------------------------------------------
  !> Returns the dimension of the wavefunction for a given set of quantum 
  !> numbers.                                                             
  !-----------------------------------------------------------------------
  elemental function qn_wf_dim(qn)
    type(qn_t), intent(in) :: qn
    integer :: qn_wf_dim
    
    if (qn%j /= M_ZERO) then
      qn_wf_dim = 2
    else if (qn%m /= M_ZERO) then
      if (abs(qn%m) == qn%l + M_HALF) then
        qn_wf_dim = 2
      else
        qn_wf_dim = 4
      end if
    else
      qn_wf_dim = 1
    end if

  end function qn_wf_dim

  !-----------------------------------------------------------------------
  !> Returns true if qn_a and qn_b belong to the same fold. This is the   
  !> case if all the quantum numbers  that appear on the wave-equation of 
  !> sets a and b are the same, i.e., they only differ in the number of   
  !> nodes quantum number.                                                
  !-----------------------------------------------------------------------
  elemental function qn_equal_fold(qn_a, qn_b, polarized)
    type(qn_t),           intent(in) :: qn_a, qn_b
    logical,    optional, intent(in) :: polarized
    logical :: qn_equal_fold

    qn_equal_fold = ( (qn_a%l == qn_b%l) .and. &
                      (qn_a%j == qn_b%j) .and. &
                      (qn_a%s == qn_b%s) .and. &
                      (qn_a%m == qn_b%m) )

    if (present(polarized)) then
      qn_equal_fold = qn_equal_fold .and. (polarized .or. qn_a%sg == qn_b%sg)
    end if

  end function qn_equal_fold
  
end module quantum_numbers_m
