!! Copyright (C) 2004-2010 M. Oliveira, F. Nogueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: potentials.F90 778 2013-07-11 15:49:39Z micael $

#include "global.h"

module potentials_m
  use global_m
  use messages_m
  use mesh_m
  use mixing_m
  use quantum_numbers_m
  use loc_potentials_m
  use sl_potentials_m
  use kb_projectors_m
  implicit none


                    !---Interfaces---!

  interface assignment (=)
     module procedure potential_copy
  end interface

  interface potential_init
     module procedure potential_init_ae, potential_init_ps, &
                      potential_init_ps_scr, potential_init_ps_full, &
                      potential_init_kb
   end interface


                    !---Derived Data Types---!

  type potential_t
    private
    integer      :: type
    type(mesh_t) :: m

    type(loc_potential_t), pointer :: vl
    type(sl_potential_t),  pointer :: vsl
    type(kb_projectors_t), pointer :: kb

    !Screening  will be treated as a local potential
    logical :: screened
    logical :: polarized
    integer :: nspin
    type(loc_potential_t), pointer :: vhxc(:)
    type(loc_potential_t), pointer :: vxctau(:)
  end type potential_t


                    !---Global Variables---!

  integer, parameter :: LOC = 1, &
                        SL  = 2, &
                        KBP = 3


                    !---Public/Private Statements---!

  private
  public :: potential_t, &
            potential_null, &
            potential_init, &
            assignment(=), &
            potential_end, &
            potential_save, &
            potential_load, &
            classical_turning_point, &
            v, &
            dvdr, &
            d2vdr2, &
            bxc, &
            vtau, &
            dvtaudr, &
            bxc_integral, &
            potential_debug, &
            potential_update_vhxc, &
            potential_mix, &
            potential_nuclear_charge, &
            potential_is_polarized, &
            potential_min, &
            potential_max, &
            potential_rmin, &
            potential_rmax, &
            potential_kb_energy, &
            potential_output, &
            potential_ps_io_set


contains

  subroutine potential_null(potential)
    !-----------------------------------------------------------------------!
    ! Nullifies and sets to zero all the components of the potential.       !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(out) :: potential

    call push_sub("potential_null")

    potential%type = 0
    call mesh_null(potential%m)

    nullify(potential%vl)
    nullify(potential%vsl)
    nullify(potential%kb)

    potential%screened = .false.
    potential%polarized = .false.
    potential%nspin = 0
    nullify(potential%vhxc)
    nullify(potential%vxctau)

    call pop_sub()
  end subroutine potential_null

  subroutine potential_init_ae(potential, m, nspin, vhxc, vxctau, z)
    !-----------------------------------------------------------------------!
    ! Initialize the potential.                                             !
    !                                                                       !
    !  potential - potential to be initialized                              !
    !  m         - mesh                                                     !
    !  nspin     - number of spin channels                                  !
    !  vhxc      - Hartree and exchange-correlation potential               !
    !  vxctau    - extra exchange-correlation term from MGGA                !
    !  z         - nuclear charge                                           !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: potential
    type(mesh_t),      intent(in)    :: m
    integer,           intent(in)    :: nspin
    real(R8),          intent(in)    :: vhxc(m%np, nspin)
    real(R8),          intent(in)    :: vxctau(m%np, nspin)
    real(R8),          intent(in)    :: z

    call push_sub("potential_init_ae")

    ASSERT(potential%type == 0)

    potential%type = LOC
    potential%m = m

    allocate(potential%vl)
    call loc_potential_null(potential%vl)
    call loc_potential_init(potential%vl, m, z=z)

    potential%screened = .true.
    call potential_init_vhxc(potential, nspin, vhxc, vxctau)

    call pop_sub()
  end subroutine potential_init_ae

  subroutine potential_init_ps(potential, m, v)
    !-----------------------------------------------------------------------!
    ! Initialize a single pseudo-potential without screening.               !
    !                                                                       !
    !  potential - potential to be initialized                              !
    !  m         - mesh                                                     !
    !  v         - values of the potential on the mesh                      !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: potential
    type(mesh_t),      intent(in)    :: m
    real(R8),          intent(in)    :: v(m%np)

    call push_sub("potential_init_ps")

    ASSERT(potential%type == 0)

    potential%type = LOC
    potential%m = m

    allocate(potential%vl)
    call loc_potential_null(potential%vl)
    call loc_potential_init(potential%vl, m, v=v)

    potential%screened = .false.

    call pop_sub()
  end subroutine potential_init_ps

  subroutine potential_init_ps_scr(potential, m, nspin, vhxc, vxctau, v)
    !-----------------------------------------------------------------------!
    ! Initialize a single pseudo-potential with screening.                  !
    !                                                                       !
    !  potential - potential to be initialized                              !
    !  m         - mesh                                                     !
    !  nspin     - number of spin channels                                  !
    !  vhxc      - Hartree and exchange-correlation potential               !
    !  vxctau    - extra exchange-correlation term from MGGA                !
    !  v         - values of the potential on the mesh                      !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: potential
    type(mesh_t),      intent(in)    :: m
    integer,           intent(in)    :: nspin
    real(R8),          intent(in)    :: vhxc(m%np, nspin)
    real(R8),          intent(in)    :: vxctau(m%np, nspin)
    real(R8),          intent(in)    :: v(m%np)

    call push_sub("potential_init_ps_scr")

    ASSERT(potential%type == 0)

    potential%type = LOC
    potential%m = m

    allocate(potential%vl)
    call loc_potential_null(potential%vl)
    call loc_potential_init(potential%vl, m, v=v)

    potential%screened = .true.
    call potential_init_vhxc(potential, nspin, vhxc, vxctau)

    call pop_sub()
  end subroutine potential_init_ps_scr

  subroutine potential_init_ps_full(potential, m, nspin, vhxc, vxctau, nc, qn, v)
    !-----------------------------------------------------------------------!
    ! Initialize a full set of pseudo-potentials with screening.            !
    !                                                                       !
    !  potential - potential to be initialized                              !
    !  m         - mesh                                                     !
    !  nspin     - number of spin channels                                  !
    !  vhxc      - Hartree and exchange-correlation potential               !
    !  vxctau    - extra exchange-correlation term from MGGA                !
    !  nc        - number of components                                     !
    !  qn        - quantum numbers of each component                        !
    !  v         - values of the potentials on the mesh for each component  !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: potential
    type(mesh_t),      intent(in)    :: m
    integer,           intent(in)    :: nspin
    real(R8),          intent(in)    :: vhxc(m%np, nspin)
    real(R8),          intent(in)    :: vxctau(m%np, nspin)
    integer,           intent(in)    :: nc
    type(qn_t),        intent(in)    :: qn(nc)
    real(R8),          intent(in)    :: v(m%np, nc)

    call push_sub("potential_init_ps_full")

    ASSERT(potential%type == 0)

    potential%type = SL
    potential%m = m

    allocate(potential%vsl)
    call sl_potential_null(potential%vsl)
    call sl_potential_init(potential%vsl, m, nc, qn, v)

    potential%screened = .true.
    call potential_init_vhxc(potential, nspin, vhxc, vxctau)

    call pop_sub()
  end subroutine potential_init_ps_full

  subroutine potential_init_kb(potential, m, l_local, v_local, nc, qn, e, proj_f)
    !-----------------------------------------------------------------------!
    ! Initialize a full set of pseudo-potentials with screening.            !
    !                                                                       !
    !  potential - potential to be initialized                              !
    !  m         - mesh                                                     !
    !  nspin     - number of spin channels                                  !
    !  l_local   - local potential angular momentum                         !
    !  v_local   - the local part of the potential                          !
    !  nc        - the number of KB projectors                              !
    !  qn        - quantum numbers of each component                        !
    !  e         - the KB energies                                          !
    !  proj_f    - values of the projectors on the mesh                     !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: potential
    type(mesh_t),      intent(in)    :: m
    integer,           intent(in)    :: l_local
    type(potential_t), intent(in)    :: v_local
    integer,           intent(in)    :: nc
    type(qn_t),        intent(in)    :: qn(nc)
    real(R8),          intent(in)    :: e(nc)
    real(R8),          intent(in)    :: proj_f(m%np, nc)

    integer :: is

    call push_sub("potential_init_kb")

    ASSERT(potential%type == 0)
    ASSERT(v_local%type == LOC)

    potential%type = KBP
    potential%m = m

    allocate(potential%kb)
    call kb_projectors_null(potential%kb)
    call kb_projectors_init(potential%kb, l_local, v_local%vl, m, nc, qn, e, proj_f)

    potential%screened = v_local%screened
    if (v_local%screened) then
      potential%nspin = v_local%nspin
      allocate(potential%vhxc(potential%nspin), potential%vxctau(potential%nspin))
      do is = 1, potential%nspin
        call loc_potential_null(potential%vhxc(is))
        call loc_potential_null(potential%vxctau(is))
        potential%vhxc(is) = v_local%vhxc(is)
        potential%vxctau(is) = v_local%vxctau(is)
      end do
    end if

    call pop_sub()
  end subroutine potential_init_kb

  subroutine potential_init_vhxc(potential, nspin, vhxc, vxctau)
    !-----------------------------------------------------------------------!
    ! Initialize the Hartree and exchange-correlation part of the potential.!
    !                                                                       !
    !  potential - potential to be initialized                              !
    !  nspin     - number of spin channels                                  !
    !  vhxc      - Hartree and exchange-correlation potential               !
    !  vxctau    - extra exchange-correlation term from MGGA                !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: potential
    integer,           intent(in)    :: nspin
    real(R8),          intent(in)    :: vhxc(potential%m%np, nspin)
    real(R8),          intent(in)    :: vxctau(potential%m%np, nspin)

    integer :: is
    real(R8) :: ra, rb, bxc_int

    call push_sub("potential_init_vhxc")

    ASSERT(potential%type /= 0)

    potential%nspin = nspin
    allocate(potential%vhxc(nspin), potential%vxctau(nspin))
    do is = 1, nspin
      call loc_potential_null(potential%vhxc(is))
      call loc_potential_null(potential%vxctau(is))
      call loc_potential_init(potential%vhxc(is), potential%m, v=vhxc(:, is))
      call loc_potential_init(potential%vxctau(is), potential%m, v=vxctau(:, is))
    end do

    if (nspin == 2) then
      ra = potential_rmin(potential)
      rb = potential_rmax(potential)
      bxc_int = M_HALF*(loc_integral(potential%vhxc(2), ra, rb) - &
                        loc_integral(potential%vhxc(1), ra, rb))
      potential%polarized = abs(bxc_int) > 1e-10
    else
      potential%polarized = .false.
    end if

    call pop_sub()
  end subroutine potential_init_vhxc

  subroutine potential_copy(potential_a, potential_b)
    !-----------------------------------------------------------------------!
    ! Copies the potential potential_a to potential potential_b.            !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: potential_a
    type(potential_t), intent(in)    :: potential_b

    integer :: is

    call push_sub("potential_copy")

    ASSERT(potential_b%type /= 0)

    call potential_end(potential_a)

    potential_a%type = potential_b%type
    potential_a%m = potential_b%m

    select case (potential_b%type)
    case (LOC)
      allocate(potential_a%vl)
      call loc_potential_null(potential_a%vl)
      potential_a%vl = potential_b%vl
    case (SL)
      allocate(potential_a%vsl)
      call sl_potential_null(potential_a%vsl)
      call sl_potential_copy(potential_a%m, potential_a%vsl, potential_b%vsl)
    case (KBP)
      allocate(potential_a%kb)
      call kb_projectors_null(potential_a%kb)
      call kb_projectors_copy(potential_a%m, potential_a%kb, potential_b%kb)
    end select

    potential_a%screened = potential_b%screened
    potential_a%polarized = potential_b%polarized
    potential_a%nspin = potential_b%nspin
    if (potential_b%screened) then
      allocate(potential_a%vhxc(potential_a%nspin), potential_a%vxctau(potential_a%nspin))
      do is = 1, potential_a%nspin
        call loc_potential_null(potential_a%vhxc(is))
        call loc_potential_null(potential_a%vxctau(is))
        potential_a%vhxc(is) = potential_b%vhxc(is)
        potential_a%vxctau(is) = potential_b%vxctau(is)
      end do
    end if

    call pop_sub()
  end subroutine potential_copy

  subroutine potential_end(potential)
    !-----------------------------------------------------------------------!
    ! Frees all the memory associated to the potential.                     !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: potential

    integer :: is

    call push_sub("potential_end")

    potential%type = 0
    call mesh_end(potential%m)

    if (associated(potential%vl)) then
      call loc_potential_end(potential%vl)
      deallocate(potential%vl)
    end if

    if (associated(potential%vsl)) then
      call sl_potential_end(potential%vsl)
      deallocate(potential%vsl)
    end if

    if (associated(potential%kb)) then
      call kb_projectors_end(potential%kb)
      deallocate(potential%kb)
    end if

    if (associated(potential%vhxc)) then
      do is = 1, potential%nspin
        call loc_potential_end(potential%vhxc(is))
        call loc_potential_end(potential%vxctau(is))
      end do
      deallocate(potential%vhxc, potential%vxctau)
    end if
    potential%nspin = 0
    potential%screened = .false.

    call pop_sub()
  end subroutine potential_end

  subroutine potential_save(unit, potential)
    !-----------------------------------------------------------------------!
    ! Writes the potential to a file.                                       !
    !                                                                       !
    !  uni       - file unit number                                         !
    !  potential - potential to be written                                  !
    !-----------------------------------------------------------------------!
    integer,           intent(in) :: unit
    type(potential_t), intent(in) :: potential

    integer :: is

    call push_sub("potential_save")

    ASSERT(potential%type /= 0)

    write(unit) potential%type
    call mesh_save(unit, potential%m)

    select case (potential%type)
    case (LOC)
      call loc_potential_save(unit, potential%m, potential%vl)
    case (SL)
      call sl_potential_save(unit, potential%m, potential%vsl)
    case (KBP)
      call kb_projectors_save(unit, potential%m, potential%kb)
    end select

    write(unit) potential%screened
    if (potential%screened) then
      write(unit) potential%nspin
      do is = 1, potential%nspin
        call loc_potential_save(unit, potential%m, potential%vhxc(is))
        call loc_potential_save(unit, potential%m, potential%vxctau(is))
      end do
    end if

    call pop_sub()
  end subroutine potential_save

  subroutine potential_load(unit, potential)
    !-----------------------------------------------------------------------!
    ! Reads the potential from a file.                                      !
    !                                                                       !
    !  unit      - file unit number                                         !
    !  potential - potential to be read                                     !
    !-----------------------------------------------------------------------!
    integer,           intent(in)    :: unit
    type(potential_t), intent(inout) :: potential

    integer :: is

    call push_sub("potential_load")

    ASSERT(potential%type == 0)

    read(unit) potential%type
    call mesh_load(unit, potential%m)

    select case (potential%type)
    case (LOC)
      allocate(potential%vl)
      call loc_potential_null(potential%vl)
      call loc_potential_load(unit, potential%m, potential%vl)
    case (SL)
      allocate(potential%vsl)
      call sl_potential_null(potential%vsl)
      call sl_potential_load(unit, potential%m, potential%vsl)
    case (KBP)
      allocate(potential%kb)
      call kb_projectors_null(potential%kb)
      call kb_projectors_load(unit, potential%m, potential%kb)
    end select

    read(unit) potential%screened
    if (potential%screened) then
      read(unit) potential%nspin
      allocate(potential%vhxc(potential%nspin), potential%vxctau(potential%nspin))
      do is = 1, potential%nspin
        call loc_potential_null(potential%vhxc(is))
        call loc_potential_null(potential%vxctau(is))
        call loc_potential_load(unit, potential%m, potential%vhxc(is))
        call loc_potential_load(unit, potential%m, potential%vxctau(is))
      end do
    end if

    call pop_sub()
  end subroutine potential_load

  function classical_turning_point(potential, e, qn)
    !-----------------------------------------------------------------------!
    ! Returns the closest mesh point to the point at which the potential is !
    ! equal to a given energy.                                              !
    !                                                                       !
    !  m         - mesh                                                     !
    !  potential - potential                                                !
    !  e         - energy                                                   !
    !  qn        - set of quantum numbers                                   !

    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8),          intent(in) :: e
    type(qn_t),        intent(in) :: qn
    real(R8) :: classical_turning_point

    integer  :: i
    real(R8) :: d

    call push_sub("classical_turning_point")

    ASSERT(potential%type /= 0)

    if (e == M_ZERO) then
      classical_turning_point = potential%m%r(potential%m%np - 1)
    else
      d = v(potential, potential%m%r(potential%m%np), qn) - e
      do i = potential%m%np - 1, 1, -1
        if ((v(potential, potential%m%r(i), qn) - e)*d <= M_ZERO) then
          classical_turning_point = potential%m%r(i)
          exit
        end if
        if (i == 1) then
          classical_turning_point = potential%m%r(potential%m%np - 1)
        end if
      end do
    end if

    call pop_sub()
  end function classical_turning_point

  function v(potential, r, qn, unscreened)
    !-----------------------------------------------------------------------!
    ! Returns the value of the potential felt by an electron of at radius r.!
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8),          intent(in) :: r
    type(qn_t),        intent(in) :: qn
    logical,           intent(in), optional :: unscreened
    real(R8) :: v

    logical :: unscreened_

    ASSERT(potential%type /= 0)

    if( r <= potential%m%r(potential%m%np)) then
      select case (potential%type)
      case (LOC)
        v = loc_v(potential%vl, r)
      case (SL)
        v = sl_v(potential%vsl, r, qn)
      case (KBP)
        v = kb_v(potential%kb, r)
      end select

      if (potential%screened) then
        unscreened_ = .false.
        if (present(unscreened)) unscreened_ = unscreened
        if (.not. unscreened_) then          
          if (qn%s == M_ZERO .and. qn%m /= M_ZERO) then
            v = v + M_HALF*(loc_v(potential%vhxc(1), r) + loc_v(potential%vhxc(2), r))
          else
            v = v + loc_v(potential%vhxc(int(qn%s + 1.5_r8)), r)
          end if
        end if
      end if

    else
      v = M_ZERO
    end if

  end function v

  function dvdr(potential, r, qn)
    !-----------------------------------------------------------------------!
    ! Returns the value of the first derivative of the potential felt by an !
    ! electron at radius r.                                                 !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8),          intent(in) :: r
    type(qn_t),        intent(in) :: qn
    real(R8) :: dvdr

    ASSERT(potential%type /= 0)

    if( r <= potential%m%r(potential%m%np)) then
      select case (potential%type)
      case (LOC)
        dvdr = loc_dvdr(potential%vl, r)
      case (SL)
        dvdr = sl_dvdr(potential%vsl, r, qn)
      case (KBP)
        dvdr = kb_dvdr(potential%kb, r)
      end select

      if (potential%screened) then
        if (qn%s == M_ZERO .and. qn%m /= M_ZERO) then
          dvdr = dvdr + M_HALF*(loc_dvdr(potential%vhxc(1), r) + loc_dvdr(potential%vhxc(2), r))
        else
          dvdr = dvdr + loc_dvdr(potential%vhxc(int(qn%s + 1.5_r8)), r)
        end if
      end if

    else
      dvdr = M_ZERO
    end if

  end function dvdr

  function d2vdr2(potential, r, qn)
    !-----------------------------------------------------------------------!
    ! Returns the value of the second derivative of the potential felt by   !
    ! an electron at radius r.                                              !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8),          intent(in) :: r
    type(qn_t),        intent(in) :: qn
    real(R8) :: d2vdr2

    ASSERT(potential%type /= 0)

    if( r <= potential%m%r(potential%m%np)) then
      select case (potential%type)
      case (LOC)
        d2vdr2 = loc_d2vdr2(potential%vl, r)
      case (SL)
        d2vdr2 = sl_d2vdr2(potential%vsl, r, qn)
      case (KBP)
        d2vdr2 = kb_d2vdr2(potential%kb, r)
      end select

      if (potential%screened) then
        if (qn%s == M_ZERO .and. qn%m /= M_ZERO) then
          d2vdr2 = d2vdr2 + M_HALF*(loc_d2vdr2(potential%vhxc(1), r) + loc_d2vdr2(potential%vhxc(2), r))
        else
          d2vdr2 = d2vdr2 + loc_d2vdr2(potential%vhxc(int(qn%s + 1.5_r8)), r)
        end if
      end if

    else
      d2vdr2 = M_ZERO
    end if

  end function d2vdr2

  function bxc(potential, r)
    !-----------------------------------------------------------------------!
    ! Returns the value of the potential felt by an electron of at radius r.!
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8),          intent(in) :: r
    real(R8) :: bxc

    ASSERT(potential%type /= 0)
    ASSERT(potential%screened)

    if(potential%nspin == 2 .and. r <= potential%m%r(potential%m%np)) then
      bxc = M_HALF*(loc_v(potential%vhxc(2), r) - loc_v(potential%vhxc(1), r))
    else
      bxc = M_ZERO
    end if

  end function bxc

  function vtau(potential, r, qn)
    !-----------------------------------------------------------------------!
    ! Returns the value of the extra exchange-correlation term from MGGA    !
    ! at radius r.                                                          !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8),          intent(in) :: r
    type(qn_t),        intent(in) :: qn
    real(R8) :: vtau

    ASSERT(potential%type /= 0)

    if( r <= potential%m%r(potential%m%np) .and. potential%screened) then
      vtau = loc_v(potential%vxctau(int(qn%s + 1.5_r8)), r)
    else
      vtau = M_ZERO
    end if

  end function vtau

  function dvtaudr(potential, r, qn)
    !-----------------------------------------------------------------------!
    ! Returns the value of the first derivative of the extra                !
    ! exchange-correlation term from MGGA at radius r.                      !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8),          intent(in) :: r
    type(qn_t),        intent(in) :: qn
    real(R8) :: dvtaudr

    ASSERT(potential%type /= 0)

    if( r <= potential%m%r(potential%m%np) .and. potential%screened) then
      dvtaudr = loc_dvdr(potential%vxctau(int(qn%s + 1.5_r8)), r)
    else
      dvtaudr = M_ZERO
    end if

  end function dvtaudr

  function bxc_integral(potential, ra, rb)
    !-----------------------------------------------------------------------!
    ! Returns the value of the integral of bxc between ra and rb.           !
    !-----------------------------------------------------------------------!
    type(potential_t),  intent(in) :: potential
    real(R8), optional, intent(in) :: ra, rb
    real(R8) :: bxc_integral

    real(R8) :: ra_, rb_

    ASSERT(potential%type /= 0)
    ASSERT(potential%screened)

    if (present(ra)) then
      ra_ = ra
      ra_ = max(potential_rmin(potential), ra_)
      ra_ = min(potential_rmax(potential), ra_)
    else
      ra_ = potential_rmin(potential)
    end if

    if (present(rb)) then
      rb_ = rb
      rb_ = max(potential_rmin(potential), rb_)
      rb_ = min(potential_rmax(potential), rb_)
    else
      rb_ = potential_rmax(potential)
    end if

    if (potential%nspin == 2) then
      bxc_integral = M_HALF*(loc_integral(potential%vhxc(2), ra_, rb_) - &
                             loc_integral(potential%vhxc(1), ra_, rb_))
    else
      bxc_integral = M_ZERO
    end if

  end function bxc_integral

  subroutine potential_debug(potential)
    !-----------------------------------------------------------------------!
    ! Prints debug information to the "debug_info" directory.               !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential

    call push_sub("potential_debug")

    ASSERT(potential%type /= 0)

    call potential_output(potential, "debug_info")

    call pop_sub()
  end subroutine potential_debug

  subroutine potential_update_vhxc(potential, vhxc, vxctau)
    !-----------------------------------------------------------------------!
    ! Update the value of the Hartree and exchange-correlation part of the  !
    ! potential.                                                            !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: potential
    real(R8),          intent(in)    :: vhxc(:,:)
    real(R8),          intent(in)    :: vxctau(:,:)

    integer :: is

    call push_sub("potential_update_vhxc")

    ASSERT(potential%type /= 0 .and. potential%screened)

    do is = 1, potential%nspin
      call loc_potential_end(potential%vhxc(is))
      call loc_potential_end(potential%vxctau(is))
    end do
    deallocate(potential%vhxc, potential%vxctau)

    call potential_init_vhxc(potential, potential%nspin, vhxc, vxctau)

    call pop_sub()
  end subroutine potential_update_vhxc

  subroutine potential_mix(pot_in, pot_out, nit, mix)
    !-----------------------------------------------------------------------!
    ! Mix input and output all-electron potentials. This subroutine should  !
    ! be used during an self-consistent field cycle and is only valid for   !
    ! all-electron potentials. On exit the input potential.                 !
    ! is replaced by the mixed potential.                                   !
    !                                                                       !
    !  m        - mesh                                                      !
    !  pot_in   - input potential/new potential                             !
    !  pot_out  - output potential                                          !
    !  nit      - number of iteration of the SCF cycle                      !
    !  mix_info - information about the mixing procedure                    !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(inout) :: pot_in
    integer,           intent(in)    :: nit
    type(potential_t), intent(in)    :: pot_out
    type(mixing_t),    intent(inout) :: mix

    integer  :: is, i, np, nspin
    real(R8), allocatable :: vhxc_in(:), vhxc_out(:), vhxc_new(:), vhxc(:,:), vxctau(:,:)
    
    call push_sub("potential_mix")

    ASSERT(pot_in%type == pot_out%type)
    ASSERT(pot_in%screened .and. pot_out%screened)

    np = pot_in%m%np
    nspin = pot_in%nspin

    !Put all the components of vhxc in a single vector
    allocate(vhxc_in(np*pot_in%nspin*2))
    allocate(vhxc_out(np*pot_in%nspin*2))
    allocate(vhxc_new(np*pot_in%nspin*2))
    do is = 1, pot_in%nspin
      do i = 1, np
        vhxc_in(np*(is - 1) + i)  = loc_v(pot_in%vhxc(is),  pot_in%m%r(i))
        vhxc_out(np*(is - 1) + i) = loc_v(pot_out%vhxc(is), pot_in%m%r(i))
        vhxc_in(np*(is + nspin - 1) + i)  = loc_v(pot_in%vxctau(is),  pot_in%m%r(i))
        vhxc_out(np*(is + nspin - 1) + i) = loc_v(pot_out%vxctau(is), pot_in%m%r(i))
      end do
    end do
  
    !Mix
    call mixing(mix, nit, vhxc_in, vhxc_out, vhxc_new)

    !Put the components of vhxc_new back to the original ordering
    allocate(vhxc(np, pot_in%nspin), vxctau(np, pot_in%nspin))
    do is = 1, pot_in%nspin
      do i = 1, np
        vhxc(i, is) = vhxc_new(np*(is - 1) + i)
        vxctau(i, is) = vhxc_new(np*(is + nspin - 1) + i)
      end do
    end do

    !Replace vhxc part of the input potential by the new vhxc
    call potential_update_vhxc(pot_in, vhxc, vxctau)
    deallocate(vhxc_in, vhxc_out, vhxc_new, vhxc, vxctau)

    call pop_sub()
  end subroutine potential_mix

  function potential_nuclear_charge(potential)
    !-----------------------------------------------------------------------!
    ! Returns the value of the nuclear charge of an all-electron potential. !
    ! Returns zero for other types of potentials.                           !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8) :: potential_nuclear_charge

    call push_sub("potential_nuclear_charge")

    ASSERT(potential%type /= 0)

    potential_nuclear_charge = M_ZERO
    if (potential%type == LOC) then 
      potential_nuclear_charge = loc_potential_nuclear_charge(potential%vl)
    end if

    call pop_sub()
  end function potential_nuclear_charge

function potential_is_polarized(potential)
    !-----------------------------------------------------------------------!
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    logical :: potential_is_polarized

    call push_sub("potential_is_polarized")

    ASSERT(potential%type /= 0)

    potential_is_polarized = potential%polarized

    call pop_sub()
  end function potential_is_polarized

  function potential_min(potential, qn)
    !-----------------------------------------------------------------------!
    ! Returns the minimum value of the potential.                           !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    type(qn_t),        intent(in) :: qn
    real(R8) :: potential_min
 
    integer :: i
    real(R8) :: vv
 
    call push_sub("potential_min")

    ASSERT(potential%type /= 0)

    if (potential%type == LOC) then
      if (loc_potential_is_ae(potential%vl)) then
        message(1) = "Trying to find the minimun of an all-electron potential."
        call write_fatal(1)
      end if
    end if

    potential_min = M_ZERO
    do i = 1, potential%m%np
      vv = v(potential, potential%m%r(i), qn)
      if (vv < potential_min) potential_min = vv
    end do

    call pop_sub()
  end function potential_min

  function potential_max(potential, qn)
    !-----------------------------------------------------------------------!
    ! Returns the maximum value of the potential.                           !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    type(qn_t),        intent(in) :: qn
    real(R8) :: potential_max
 
    integer :: i
    real(R8) :: vv
 
    call push_sub("potential_max")

    ASSERT(potential%type /= 0)

    potential_max = v(potential, potential%m%r(1), qn)
    do i = 2, potential%m%np
      vv = v(potential, potential%m%r(i), qn)
      if (vv > potential_max) potential_max = vv
    end do

    call pop_sub()
  end function potential_max

  function potential_rmin(potential)
    !-----------------------------------------------------------------------!
    ! Closest point to the origin that can be safely used.                  !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8) :: potential_rmin

    call push_sub("potential_rmin")

    ASSERT(potential%type /= 0)

    potential_rmin = potential%m%r(1)

    call pop_sub()
  end function potential_rmin

  function potential_rmax(potential)
    !-----------------------------------------------------------------------!
    ! Larger point that can be safely used.                                 !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    real(R8) :: potential_rmax

    call push_sub("potential_rmax")

    ASSERT(potential%type /= 0)

    potential_rmax = potential%m%r(potential%m%np)

    call pop_sub()
  end function potential_rmax

  function potential_kb_energy(potential, qn)
    !-----------------------------------------------------------------------!
    ! Returns the value of the KB energy.                                   !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    type(qn_t),        intent(in) :: qn
    real(R8) :: potential_kb_energy

    call push_sub("potential_kb_energy")

    ASSERT(potential%type == KBP)

    potential_kb_energy = kb_projectors_energy(potential%kb, qn)

    call pop_sub()
  end function potential_kb_energy

  subroutine potential_output(potential, dir)
    !-----------------------------------------------------------------------!
    ! Writes the potential to a file in a format suitable for plotting.     !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential
    character(len=*),     intent(in) :: dir

    call push_sub("potential_output")

    ASSERT(potential%type /= 0)

    select case (potential%type)
    case (LOC)
      call loc_potential_output(potential%vl, potential%m, trim(dir)//"/v_ext")
    case (SL)
      call sl_potential_output(potential%vsl, potential%m, dir)
    case (KBP)
      call kb_projectors_output(potential%kb, potential%m, dir)
    end select

    if (potential%screened .and. potential%type /= KBP) then
      if (potential%nspin == 1) then
        call loc_potential_output(potential%vhxc(1), potential%m, trim(dir)//"/v_hxc")
      else
        call loc_potential_output(potential%vhxc(1), potential%m, trim(dir)//"/v_hxc_dn")
        call loc_potential_output(potential%vhxc(2), potential%m, trim(dir)//"/v_hxc_up")
      end if
    end if

    call pop_sub()
  end subroutine potential_output

  subroutine potential_ps_io_set(potential)
    !-----------------------------------------------------------------------!
    ! Pass the information about the pseudopotentials to the ps_io module.  !
    !-----------------------------------------------------------------------!
    type(potential_t), intent(in) :: potential

    call push_sub("potential_ps_io_set")

    ASSERT(potential%type == SL .or. potential%type == KBP)

    select case (potential%type)
    case (SL)
      call sl_potential_ps_io_set(potential%vsl, potential%m)
    case (KBP)
      call kb_projectors_ps_io_set(potential%kb, potential%m)
    end select

    call pop_sub()
  end subroutine potential_ps_io_set

end module potentials_m
