!! Copyright (C) 2008-2011 M. Oliveira, F. Nogueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: functionals.F90 818 2014-01-14 17:42:02Z micael $

#include "global.h"

module functionals_m
  use global_m
  use oct_parser_m
  use messages_m
  use mesh_m
  use states_m
  use hartree_m
  use xc_f90_types_m
  use xc_f90_lib_m
  implicit none


                    !---Interfaces---!

  interface assignment (=)
    module procedure functional_copy
  end interface


                    !---Derived Data Types---!

  type functional_t
    private
    integer :: family  ! LDA, GGA, etc.
    integer :: id      ! identifier
    integer :: nspin   ! XC_UNPOLARIZED | XC_POLARIZED

    type(xc_f90_pointer_t) :: conf ! the pointer used to call the library
    type(xc_f90_pointer_t) :: info ! information about the functional

    integer  :: irel
    real(R8) :: xalpha
  end type functional_t


                    !---Global Variables---!

  !Some functionals not available in libxc
  integer, parameter :: XC_OEP_X_EXX  = 600, &
                        XC_OEP_XC_SIC = 601, &
                        XC_MGGA_K_GE2 = 599

  !Some version of libxc break backward compatibility...
#if LIBXC_VERSION == 100
  integer, parameter :: XC_KINETIC = 3, &
                        XC_GGA_X_LB = XC_GGA_XC_LB
#endif


                    !---Public/Private Statements---!

  private
  public :: functional_t, &
            functional_null, &
            functional_init, &
            assignment(=), &
            functional_get_vxc, &
            functional_get_tau, &
            functional_adsic, &
            functional_rhoxc, &
            functional_name, &
            functional_kind, &
            functional_family, &
            functional_save, &
            functional_load, &
            functional_end,  &
            XC_OEP_X_EXX, &
            XC_OEP_XC_SIC, &
            XC_MGGA_K_GE2

contains

  subroutine functional_null(functl)
    !-----------------------------------------------------------------------!
    ! Nullifies and sets to zero all the components of the functional.      !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(out) :: functl

    call push_sub("functional_null")

    functl%family = 0
    functl%id     = 0
    functl%nspin  = 0
    functl%irel   = 0
    functl%xalpha = M_ZERO

    call pop_sub()
  end subroutine functional_null

  subroutine functional_init(nspin, id, irel, functl)
    !-----------------------------------------------------------------------!
    ! Initializes a functional.                                             !
    !-----------------------------------------------------------------------!
    integer,            intent(in)    :: nspin
    integer,            intent(in)    :: id, irel
    type(functional_t), intent(inout) :: functl

    call push_sub("functional_init")

    functl%id = id
    functl%nspin = nspin
    functl%irel = irel

    if(functl%id /= 0) then
      ! get the family of the functional
      if (id == XC_OEP_X_EXX) then
        functl%family = XC_FAMILY_OEP
      elseif (id == XC_MGGA_K_GE2) then
        functl%family = XC_FAMILY_MGGA
      else
        functl%family = xc_f90_family_from_id(functl%id)
      end if

      if (functl%family == XC_FAMILY_UNKNOWN) then
        write(message(1), '(a,i3,a)') "'", functl%id,"' is not a known functional!"
        message(2) = "Please check the manual for a list of possible values."
        call write_fatal(2)
      end if

    end if

    !Extra variables
    if (functl%family == XC_FAMILY_LDA .and. functl%id == XC_LDA_C_XALPHA) then
      call oct_parse_float('Xalpha', M_ONE, functl%xalpha)
    end if

    !Initialize
    if(functl%id /= 0 .and. functl%id /= XC_MGGA_K_GE2) then
      call functional_libxc_init(functl)
    end if

    call pop_sub()
  end subroutine functional_init

  subroutine functional_libxc_init(functl)
    !-----------------------------------------------------------------------!
    ! Initialize the libxc objects of the functional.                       !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(inout) :: functl

    if (functl%family /= XC_FAMILY_OEP) then
      call xc_f90_func_init(functl%conf, functl%info, functl%id, functl%nspin)

      if (functl%id == XC_LDA_C_XALPHA) then
        call xc_f90_lda_c_xalpha_set_par(functl%conf, functl%xalpha)
      end if

      if (functl%id == XC_LDA_X) then
#if LIBXC_VERSION == 200
        call xc_f90_lda_x_set_par(functl%conf, M_FOUR/M_THREE, functl%irel, M_ZERO)
#else
        call xc_f90_lda_x_set_par(functl%conf, functl%irel)
#endif
      end if
    end if

  end subroutine functional_libxc_init

  subroutine functional_copy(functl_out, functl_in)
    !-----------------------------------------------------------------------!
    ! Copies the functional functl_in to functl_out.                        !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(inout) :: functl_out
    type(functional_t), intent(in)    :: functl_in

    call push_sub("functional_copy")

    call functional_end(functl_out)

    functl_out%family = functl_in%family
    functl_out%id     = functl_in%id
    functl_out%nspin  = functl_in%nspin

    functl_out%irel = functl_in%irel
    functl_out%xalpha = functl_in%xalpha

    if(functl_out%id /= 0 .and. functl_out%id /= XC_MGGA_K_GE2) then
      call functional_libxc_init(functl_out)
    end if

    call pop_sub()
  end subroutine functional_copy

  subroutine functional_end(functl)
    !-----------------------------------------------------------------------!
    ! Frees all memory associated to the functional.                        !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(inout) :: functl

    call push_sub("functional_end")

    select case (functl%family)
    case (XC_FAMILY_LDA, XC_FAMILY_GGA, XC_FAMILY_MGGA)
      if (functl%id /= XC_MGGA_K_GE2) then
        call xc_f90_func_end(functl%conf)
      end if
    end select

    functl%family = 0
    functl%id     = 0
    functl%nspin  = 0

    functl%irel   = 0
    functl%xalpha = M_ZERO

    call pop_sub()
  end subroutine functional_end

  function functional_name(functl)
    !-----------------------------------------------------------------------!
    ! Returns the name of the functional.                                   !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(in) :: functl
    character(120) :: functional_name

    select case (functl%id)
    case (0)
      functional_name = "None"
    case (XC_OEP_X_EXX)
      functional_name = "Exact Exchange"
    case (XC_MGGA_K_GE2)
      functional_name = "Second-order gradient expansion of the kinetic energy density"
    case default
      call xc_f90_info_name(functl%info, functional_name)
    end select

  end function functional_name

  function functional_kind(functl)
    !-----------------------------------------------------------------------!
    ! Returns the kind of functional we have                                !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(in) :: functl
    integer :: functional_kind

    select case (functl%id)
    case (0)
      functional_kind = -1
    case (XC_OEP_X_EXX)
      functional_kind = XC_EXCHANGE
    case (XC_OEP_XC_SIC)
      functional_kind = XC_EXCHANGE_CORRELATION
    case (XC_MGGA_K_GE2)
      functional_kind = XC_KINETIC
    case default
      functional_kind = xc_f90_info_kind(functl%info)
    end select

  end function functional_kind

  elemental function functional_family(functl)
    !-----------------------------------------------------------------------!
    ! Returns the family of the functional                                  !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(in) :: functl
    integer :: functional_family

    functional_family = functl%family

  end function functional_family

  subroutine functional_save(unit, functl)
    !-----------------------------------------------------------------------!
    ! Writes the functional data to a file.                                 !
    !-----------------------------------------------------------------------!
    integer,            intent(in) :: unit
    type(functional_t), intent(in) :: functl

    call push_sub("functl_save")

    write(unit) functl%family
    write(unit) functl%id
    write(unit) functl%nspin

    write(unit) functl%irel
    write(unit) functl%xalpha

    call pop_sub()
  end subroutine functional_save

  subroutine functional_load(unit, functl)
    !-----------------------------------------------------------------------!
    ! Reads the exchange-correlation model data from a file.                !
    !-----------------------------------------------------------------------!
    integer,            intent(in)    :: unit
    type(functional_t), intent(inout) :: functl

    call push_sub("functl_load")

    read(unit) functl%family
    read(unit) functl%id
    read(unit) functl%nspin

    read(unit) functl%irel
    read(unit) functl%xalpha

    if(functl%id /= 0 .and. functl%id /= XC_MGGA_K_GE2) then
      call functional_libxc_init(functl)
    end if

    call pop_sub()
  end subroutine functional_load

  subroutine functional_get_vxc(functl, m, rho, rho_grad, rho_lapl, tau, ip, v, e, vtau)
    !-----------------------------------------------------------------------!
    ! Given a density, computes the corresponding exchange/correlation      !
    ! potentials and energies.                                              !
    !                                                                       !
    !  functl   - functional                                                !
    !  m        - mesh                                                      !
    !  rho      - electronic radial density                                 !
    !  rho_grad - gradient of the electronic radial density                 !
    !  rho_lapl - laplacian of the electronic radial density                !
    !  tau      - radial kinetic energy density                             !
    !  ip       - ionization potential                                      !
    !  v        - potential                                                 !
    !  e        - energy per-volume                                         !
    !  vtau     - extra term arising from MGGA potential                    !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(inout) :: functl
    type(mesh_t),       intent(in)    :: m
    real(R8),           intent(in)    :: rho(m%np, functl%nspin)
    real(R8),           intent(in)    :: rho_grad(m%np, functl%nspin)
    real(R8),           intent(in)    :: rho_lapl(m%np, functl%nspin)
    real(R8),           intent(in)    :: tau(m%np, functl%nspin)
    real(R8),           intent(in)    :: ip(functl%nspin)
    real(R8),           intent(out)   :: v(m%np, functl%nspin), e(m%np)
    real(R8),           intent(out)   :: vtau(m%np, functl%nspin)

    integer  :: i, is, nspin
    real(R8) :: sigma(3), vsigma(3), a, b, c
    real(R8), parameter   :: alpha = -0.012_r8, beta = 1.023_r8
    real(R8), allocatable :: d(:), dedd(:,:), ldedd(:)
    real(R8), allocatable :: lgd(:), dedgd(:,:)
    real(R8), allocatable :: ltau(:), dedtau(:)
    real(R8), allocatable :: lld(:), dedld(:,:), ldedld(:)

    call push_sub("functional_get_vxc")

    ASSERT(size(v, dim=2) == functl%nspin)
    ASSERT(functional_kind(functl) /= XC_KINETIC)

    if (functl%family == 0) then
      v = M_ZERO ; e = M_ZERO ; vtau = M_ZERO
      call pop_sub()
      return
    end if

    nspin = functl%nspin

    !Allocate work arrays
    allocate(d(nspin), dedd(m%np, nspin), ldedd(nspin))
    d = M_ZERO; dedd = M_ZERO; ldedd = M_ZERO
    if (functl%family == XC_FAMILY_GGA .or. functl%family == XC_FAMILY_MGGA) then
      allocate(lgd(nspin), dedgd(m%np, nspin))
      lgd = M_ZERO; dedgd = M_ZERO
    end if
    if (functl%family == XC_FAMILY_MGGA) then
      allocate(lld(nspin), ldedld(nspin), dedld(m%np, nspin))
      lld = M_ZERO; ldedld = M_ZERO; dedld = M_ZERO
      allocate(ltau(nspin), dedtau(nspin))
      ltau = M_ZERO; dedtau = M_ZERO
    end if

    if (functl%id == XC_MGGA_X_TB09) then
      if (maxval(ip) == M_ZERO) then
        c = M_ONE
      else
        a = M_ZERO
        do
          c = alpha + beta*sqrt(M_TWO*sqrt(M_TWO*(maxval(ip) + a)))
          b = (M_THREE*c - M_TWO)/M_PI*sqrt(M_FIVE/M_SIX*(maxval(ip) + a))
          if (abs(a - b) < 1.0e-8) exit
          a = b
        end do
      end if
      call xc_f90_mgga_x_tb09_set_par(functl%conf, c)
    end if

    !Space loop
    do i = 1, m%np
      ! make a local copy with the correct memory order
      d(1:nspin) = rho(i, 1:nspin)
      if (functl%family == XC_FAMILY_GGA .or. functl%family == XC_FAMILY_MGGA) then
        lgd(1:nspin) = rho_grad(i, 1:nspin)

        sigma = M_ZERO
        sigma(1) = lgd(1)**2
        if(nspin == 2) then
          sigma(2) = lgd(1)*lgd(2)
          sigma(3) = lgd(2)**2
        end if
      end if
      if (functl%family == XC_FAMILY_MGGA) then
#if LIBXC_VERSION >= 200
        ltau(1:nspin) = tau(i, 1:nspin)/M_TWO
#else
        ltau(1:nspin) = tau(i, 1:nspin)
#endif     
        lld(1:nspin) = rho_lapl(i, 1:nspin)
      end if

      if (iand(xc_f90_info_flags(functl%info), XC_FLAGS_HAVE_EXC) .ne. 0) then
 
        select case(functl%family)
        case(XC_FAMILY_LDA)
          call xc_f90_lda_exc_vxc(functl%conf, 1, d(1), e(i), ldedd(1))
        case(XC_FAMILY_GGA)
          call xc_f90_gga_exc_vxc(functl%conf, 1, d(1), sigma(1), e(i), ldedd(1), vsigma(1))
        case(XC_FAMILY_MGGA)
          call xc_f90_mgga_exc_vxc(functl%conf, 1, d(1), sigma(1), lld(1), ltau(1), &
               e(i), ldedd(1), vsigma(1), ldedld(1), dedtau(1))
        end select

      else !Just get the potential

        select case(functl%family)
        case(XC_FAMILY_LDA)
          call xc_f90_lda_vxc(functl%conf, 1, d(1), ldedd(1))
        case(XC_FAMILY_GGA)
          call xc_f90_gga_vxc(functl%conf, 1, d(1), sigma(1), ldedd(1), vsigma(1))
        case(XC_FAMILY_MGGA)
          call xc_f90_mgga_vxc(functl%conf, 1, d(1), sigma(1), lld(1), ltau(1), &
               ldedd(1), vsigma(1), ldedld(1), dedtau(1))
        end select
        e(i) = M_ZERO

      end if

      e(i) = e(i)*sum(d)
      dedd(i, :) = ldedd(:)
      if (functl%family == XC_FAMILY_GGA .or. functl%family == XC_FAMILY_MGGA) then
        if (nspin == 1) then
          dedgd(i, 1) = M_TWO*vsigma(1)*lgd(1)
        else
          dedgd(i, 1) = M_TWO*vsigma(1)*lgd(1) + vsigma(2)*lgd(2)
          dedgd(i, 2) = M_TWO*vsigma(3)*lgd(2) + vsigma(2)*lgd(1)
        end if
      end if
      if(functl%family == XC_FAMILY_MGGA) then
        dedld(i, 1:nspin) = dedld(i, 1:nspin) + ldedld(1:nspin)
#if LIBXC_VERSION >= 200
        vtau(i, 1:nspin) = vtau(i, 1:nspin) + dedtau(1:nspin)/M_TWO
#else
        vtau(i, 1:nspin) = vtau(i, 1:nspin) + dedtau(1:nspin)
#endif     
      else
        vtau(i, 1:nspin) = M_ZERO
      end if
    end do

    !Compute potentials
    v = dedd
    if (functl%family == XC_FAMILY_GGA .or. functl%family == XC_FAMILY_MGGA) then
      do is = 1, nspin
        v(:, is) = v(:, is) - mesh_divergence(m, dedgd(:, is))
      end do
    end if
    if (functl%family == XC_FAMILY_MGGA) then
      do is = 1, nspin
        v(:, is) = v(:, is) + mesh_laplacian(m, dedld(:, is))
      end do
    end if

    ! If LB94, we can calculate an approximation to the energy from
    ! Levy-Perdew relation PRA 32, 2010 (1985)
    if (functl%id == XC_GGA_X_LB) then
      do is = 1, nspin
        e = e - rho(:, is)*m%r*mesh_gradient(m, v(:, is))
      end do
    end if

    !Shift potentials that do not go to zero at infinity
    do is = 1, nspin
      select case (functl%id)
      case (XC_MGGA_X_BJ06)
        a = sqrt(M_FIVE/M_SIX*ip(is))/M_PI
      case (XC_MGGA_X_TB09)
        a = (M_THREE*c - M_TWO)*sqrt(M_FIVE/M_SIX*ip(is))/M_PI
      case default
        a = M_ZERO
      end select
      do i = m%np, 1, -1
        if (v(i, is) /= M_ZERO) then
          v(1:i, is) = v(1:i, is) - a
          exit
        end if
      end do
    end do

    !Deallocate arrays
    deallocate(d, dedd, ldedd)
    if (functl%family == XC_FAMILY_GGA .or. functl%family == XC_FAMILY_MGGA) then
      deallocate(lgd, dedgd)
    end if
    if (functl%family == XC_FAMILY_MGGA) then
      deallocate(lld, dedld, ldedld)
      deallocate(ltau, dedtau)
    end if

    call pop_sub()
  end subroutine functional_get_vxc

  subroutine functional_get_tau(functl, m, rho, rho_grad, rho_lapl, tau)
    !-----------------------------------------------------------------------!
    ! Computes the approximated kinetic energy density.                     !
    !                                                                       !
    !  functl   - functional                                                !
    !  m        - mesh                                                      !
    !  rho      - electronic radial density                                 !
    !  rho_grad - gradient of the electronic radial density                 !
    !  rho_lapl - laplacian of the electronic radial density                !
    !  tau      - radial kinetic energy density                             !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(in)  :: functl
    type(mesh_t),       intent(in)  :: m
    real(R8),           intent(in)  :: rho(m%np, functl%nspin)
    real(R8),           intent(in)  :: rho_grad(m%np, functl%nspin)
    real(R8),           intent(in)  :: rho_lapl(m%np, functl%nspin)
    real(R8),           intent(out) :: tau(m%np, functl%nspin)

    integer  :: i, is, nspin
    real(R8) :: sigma(3)
    real(R8), allocatable :: d(:), lgd(:), ltau(:), lld(:)

    call push_sub("functional_get_tau")

    ASSERT(functional_kind(functl) == XC_KINETIC)

    if (functl%id == XC_MGGA_K_GE2) then
      where (rho <= 1e-30)
        tau = M_ZERO
      elsewhere        
        tau = M_THREE/M_FIVE*(M_THREE*M_PI**2)**(M_TWO/M_THREE)*rho**(M_FIVE/M_THREE) + &
             rho_lapl/M_THREE + rho_grad**2/rho/36.0_r8
      end where
      call pop_sub()
      return
    end if

    nspin = functl%nspin

    !Allocate work arrays
    allocate(d(nspin), ltau(nspin))
    d = M_ZERO; ltau = M_ZERO
    if (functl%family == XC_FAMILY_GGA .or. functl%family == XC_FAMILY_MGGA) then
      allocate(lgd(nspin))
      lgd = M_ZERO
    end if
    if (functl%family == XC_FAMILY_MGGA) then
      allocate(lld(nspin))
      lld = M_ZERO
    end if

    !Spin loop
    do is = 1, nspin
      !Space loop
      do i = 1, m%np
        ! make a local copy with the correct memory order
        d = M_ZERO
        d(is) = rho(i, is)

        if (functl%family == XC_FAMILY_GGA .or. functl%family == XC_FAMILY_MGGA) then
          lgd = M_ZERO
          lgd(is) = rho_grad(i, is)

          sigma = M_ZERO
          sigma(1) = lgd(1)**2
          if(nspin == 2) then
            sigma(2) = lgd(1)*lgd(2)
            sigma(3) = lgd(2)**2
          end if
        end if
        if (functl%family == XC_FAMILY_MGGA) then
          lld = M_ZERO
          lld(is) = rho_lapl(i, is)
        end if

        select case(functl%family)
        case(XC_FAMILY_LDA)
          call xc_f90_lda_exc(functl%conf, 1, d(1), ltau(1))
        case(XC_FAMILY_GGA)
          call xc_f90_gga_exc(functl%conf, 1, d(1), sigma(1), ltau(1))
        end select

#if LIBXC_VERSION >= 200
    tau(i, is) = M_TWO*ltau(1)*d(is)
#else
    tau(i, is) = ltau(1)*d(is)
#endif
      end do
    end do

    !Deallocate arrays
    deallocate(d, ltau)
    if (functl%family == XC_FAMILY_GGA .or. functl%family == XC_FAMILY_MGGA) deallocate(lgd)
    if (functl%family == XC_FAMILY_MGGA) deallocate(lld)

    call pop_sub()
  end subroutine functional_get_tau

  subroutine functional_adsic(functls, m, nspin, rho, rho_grad, rho_lapl, tau, ip, vxc, exc)
    !-----------------------------------------------------------------------!
    ! Computes the ADSIC corrections                                        !
    !                                                                       !
    !  functls  - the functionals that should be corrected                  !
    !  m        - mesh                                                      !
    !  nspin    - number of spin channels                                   !
    !  rho      - electronic radial density                                 !
    !  rho_grad - gradient of the electronic radial density                 !
    !  rho_lapl - laplacian of the electronic radial density                !
    !  tau      - radial kinetic energy density                             !
    !  charge   - electronic charge                                         !
    !  ip       - ionization potential                                      !
    !  vxc      - ADSIC correction to the potential                         !
    !  exc      - ADSIC correction to the energy                            !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(inout) :: functls(2)
    type(mesh_t),       intent(in)    :: m
    integer,            intent(in)    :: nspin
    real(R8),           intent(in)    :: rho(m%np, nspin)
    real(R8),           intent(in)    :: rho_grad(m%np, nspin)
    real(R8),           intent(in)    :: rho_lapl(m%np, nspin)
    real(R8),           intent(in)    :: tau(m%np, nspin)
    real(R8),           intent(in)    :: ip(nspin)
    real(R8),           intent(out)   :: vxc(m%np, nspin)
    real(R8),           intent(out)   :: exc

    integer  :: i, is
    real(R8), allocatable :: e(:), v(:,:), vtau(:,:), charge(:)
    real(R8), allocatable :: srho(:,:), sgrad(:,:), slapl(:,:), stau(:,:)

    call push_sub("functional_adsic")

    !Allocate potential and energy work arrays
    allocate(v(m%np, nspin), e(m%np), vtau(m%np, nspin), charge(nspin))
    e = M_ZERO; v = M_ZERO; vtau = M_ZERO
    exc = M_ZERO; vxc = M_ZERO
		
    !Get charge
    do is = 1, nspin
      charge(is) = M_FOUR*M_PI*mesh_integrate(m, rho(:,is))
    end do

    if (sum(charge) /= M_ZERO) then
      !Get correction for Hartree potential and energy
      call hartree_potential(m, sum(rho,dim=2), sum(charge), vh = vxc, eh = exc)
      vxc = vxc/sum(charge)
      exc = exc/sum(charge)

      allocate(srho(m%np, nspin), sgrad(m%np, nspin), slapl(m%np, nspin), stau(m%np, nspin))
      do is = 1, nspin
        if (charge(is) == M_ZERO) cycle
        srho(:, is) = rho(:, is)/charge(is)
        sgrad(:, is) = rho_grad(:, is)/charge(is)
        slapl(:, is) = rho_lapl(:, is)/charge(is)
        stau (:, is) = tau(:, is)/charge(is)
        if (nspin == 2) then
          srho (:, nspin-is+1) = M_ZERO
          sgrad(:, nspin-is+1) = M_ZERO
          slapl(:, nspin-is+1) = M_ZERO
          stau (:, nspin-is+1) = M_ZERO
        end if
        !Get xc energy and potential for the average density
        do i = 1, 2
          call functional_get_vxc(functls(i), m, srho, sgrad, slapl, stau, ip, v, e, vtau)
          exc = exc + charge(is)*M_FOUR*M_PI*mesh_integrate(m, e)
          vxc = vxc + v
        end do
      end do

      deallocate(srho, sgrad, slapl, stau)
    end if

    !Deallocates arrays
    deallocate(e, v, vtau, charge)

    call pop_sub()
  end subroutine functional_adsic

  subroutine functional_rhoxc(functl, m, nspin, rho, rho_grad, rho_lapl, tau, ip, dvxc)
    !-----------------------------------------------------------------------!
    ! Computes the rhoxc correction                                         !
    !                                                                       !
    !  functl   - the functional that should be corrected                   !
    !  m        - mesh                                                      !
    !  nspin    - number of spin channels                                   !
    !  rho      - electronic radial density                                 !
    !  rho_grad - gradient of the electronic radial density                 !
    !  rho_lapl - laplacian of the electronic radial density                !
    !  tau      - radial kinetic energy density                             !
    !  ip       - ionization potential                                      !
    !  dvxc     - correction to the potential                               !
    !-----------------------------------------------------------------------!
    type(functional_t), intent(inout) :: functl
    type(mesh_t),       intent(in)    :: m
    integer,            intent(in)    :: nspin
    real(R8),           intent(in)    :: rho(m%np, nspin)
    real(R8),           intent(in)    :: rho_grad(m%np, nspin)
    real(R8),           intent(in)    :: rho_lapl(m%np, nspin)
    real(R8),           intent(in)    :: tau(m%np, nspin)
    real(R8),           intent(in)    :: ip(nspin)
    real(R8),           intent(out)   :: dvxc(m%np, nspin)

    integer  :: i
    real(R8) :: qxc
    real(R8), allocatable :: nxcb(:), nxc(:), vxc(:,:)
    real(R8), allocatable :: urho(:,:), ugrad(:,:), ulapl(:,:), utau(:,:)
    real(R8), allocatable :: uvxc(:,:), uexc(:), uvxctau(:,:)

    call push_sub("functional_rhoxc")

    !Get spin-unpolarized potential
    allocate(urho(m%np, nspin), ugrad(m%np, nspin), ulapl(m%np, nspin), utau(m%np, nspin))
    allocate(uvxc(m%np, nspin), uexc(m%np), uvxctau(m%np, nspin))
    do i = 1, m%np
      urho(i, :)  = sum(rho(i, 1:nspin))/nspin
      ugrad(i, :) = sum(rho_grad(i, 1:nspin))/nspin
      ulapl(i, :) = sum(rho_lapl(i, 1:nspin))/nspin
      utau(i, :)  = sum(tau(i, 1:nspin))/nspin
    end do
    call functional_get_vxc(functl, m, urho, ugrad, ulapl, utau, ip, uvxc, uexc, uvxctau)

    !Allocate xc density and potential
    allocate(nxcb(m%np), nxc(m%np), vxc(m%np, nspin))

    !Get original xc density
    nxc = -M_ONE/(M_FOUR*M_PI)*mesh_laplacian(m, uvxc)

    !Remove density from the tail such that the charge is as close as possible to -1
    nxcb = M_FOUR*M_PI*mesh_primitive(m, nxc)
    do i = m%np, 2, -1
      if ( (abs(nxcb(i-1)) > M_ONE .and. abs(nxcb(i)) <= M_ONE) .or. &
           (nxcb(i-1) - nxcb(i) > 1e-4) ) then
        nxc(i:m%np) = M_ZERO
        exit
      end if
    end do

    !Normalize xc density to -1
    qxc = M_FOUR*M_PI*mesh_integrate(m, nxc)
    if (qxc /= M_ZERO) then
      nxc = -nxc/qxc
    end if

    !Solve the Poisson equation for the xc density to get the correction to the potential
    call hartree_potential(m, nxc, -M_ONE, vh=vxc)
    dvxc = vxc - uvxc

    !Deallocates arrays
    deallocate(nxcb, nxc, vxc)
    deallocate(urho, ugrad, ulapl, utau)
    deallocate(uvxc, uexc, uvxctau)

    call pop_sub()
  end subroutine functional_rhoxc

end module functionals_m
