% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simex.R
\name{simex}
\alias{simex}
\alias{print.simex}
\alias{summary.simex}
\alias{print.summary.simex}
\alias{plot.simex}
\alias{predict.simex}
\alias{refit}
\alias{refit.simex}
\title{Measurement error in models using SIMEX}
\usage{
simex(model, SIMEXvariable, measurement.error, lambda = c(0.5, 1, 1.5,
  2), B = 100, fitting.method = "quadratic",
  jackknife.estimation = "quadratic", asymptotic = TRUE)

\method{plot}{simex}(x, xlab = expression((1 + lambda)),
  ylab = colnames(b)[-1], ask = FALSE, show = rep(TRUE, NCOL(b) - 1),
  ...)

\method{predict}{simex}(object, newdata, ...)

\method{print}{simex}(x, digits = max(3, getOption("digits") - 3), ...)

\method{print}{summary.simex}(x, digits = max(3, getOption("digits") -
  3), ...)

\method{refit}{simex}(object, fitting.method = "quadratic",
  jackknife.estimation = "quadratic", asymptotic = TRUE, ...)

\method{summary}{simex}(object, ...)
}
\arguments{
\item{model}{the naive model}

\item{SIMEXvariable}{character or vector of characters containing the names of the variables with measurement error}

\item{measurement.error}{given standard deviations of measurement errors. In
case of homoskedastic measurement error it is a matrix with dimension
1x\code{length(SIMEXvariable)}. In case of heteroskedastic error for at least one
SIMEXvariable it is a matrix of dimension nx}

\item{lambda}{vector of lambdas for which the simulation step should be done (without 0)}

\item{B}{number of iterations for each lambda}

\item{fitting.method}{fitting method for the extrapolation. \code{linear}, \code{quadratic},
\code{nonlinear} are implemented. (first 4 letters are enough)}

\item{jackknife.estimation}{specifying the extrapolation method for jackknife
variance estimation. Can be set to \code{FALSE} if it should not be performed}

\item{asymptotic}{logical, indicating if asymptotic variance estimation should
be done, in the naive model the option \code{x = TRUE} has to be set}

\item{x}{object of class 'simex'}

\item{xlab}{optional name for the X-Axis}

\item{ylab}{vector containing the names for the Y-Axis}

\item{ask}{logical. If \code{TRUE}, the user is asked for input, before a new figure is drawn}

\item{show}{vector of logicals indicating for wich variables a plot should be produced}

\item{\dots}{arguments passed to other functions}

\item{object}{of class 'simex'}

\item{newdata}{optionally, a data frame in which to look for
variables with which to predict. If omitted, the fitted linear predictors are used}

\item{digits}{number of digits to be printed}
}
\value{
An object of class 'simex' which contains:

\item{coefficients}{the corrected coefficients of the SIMEX model,}
\item{SIMEX.estimates}{the estimates for every lambda,}
\item{model}{the naive model,}
\item{measurement.error}{the known error standard deviations,}
\item{B}{the number of iterations,}
\item{extrapolation}{the model object of the extrapolation step,}
\item{fitting.method}{the fitting method used in the extrapolation step,}
\item{residuals}{the residuals of the main model,}
\item{fitted.values}{the fitted values of the main model,}
\item{call}{the function call,}
\item{variance.jackknife}{the jackknife variance estimate,}
\item{extrapolation.variance}{the model object of the variance extrapolation,}
\item{variance.jackknife.lambda}{the data set for the extrapolation,}
\item{variance.asymptotic}{the asymptotic variance estimates,}
\item{theta}{the estimates for every B and lambda,}
...
}
\description{
Implementation of the SIMEX algorithm for measurement error models according to Cook and Stefanski
}
\details{
Nonlinear is implemented as described in Cook and Stefanski, but is numerically
instable. It is not advisable to use this feature. If a nonlinear extrapolation
is desired please use the \code{refit()} method.

Asymptotic is only implemented for naive models of class \code{lm} or \code{glm} with homoscedastic measurement error.

\code{refit()} refits the object with a different extrapolation function.
}
\section{Methods (by generic)}{
\itemize{
\item \code{plot}: Plot the simulation and extrapolation step

\item \code{predict}: Predict using simex correction

\item \code{print}: Print simex nicely

\item \code{print}: Print summary nicely

\item \code{refit}: Refits the model with a different extrapolation function

\item \code{summary}: Summary of simulation and extrapolation
}}

\examples{
## Seed
set.seed(49494)

## simulating the measurement error standard deviations
sd_me <- 0.3
sd_me2 <- 0.4
temp <- runif(100, min = 0, max = 0.6)
sd_me_het1 <- sort(temp)
temp2 <- rnorm(100, sd = 0.1)
sd_me_het2 <- abs(sd_me_het1 + temp2)

## simulating the independent variables x (real and with measurement error):

x_real <- rnorm(100)
x_real2 <- rpois(100, lambda = 2)
x_real3 <- -4*x_real + runif(100, min = -10, max = 10)  # correlated to x_real

x_measured <- x_real + sd_me * rnorm(100)
x_measured2 <- x_real2 + sd_me2 * rnorm(100)
x_het1 <- x_real + sd_me_het1 * rnorm(100)
x_het2 <- x_real3 + sd_me_het2 * rnorm(100)

## calculating dependent variable y:
y <- x_real + rnorm(100, sd = 0.05)
y2 <- x_real + 2*x_real2 + rnorm(100, sd = 0.08)
y3 <- x_real + 2*x_real3 + rnorm(100, sd = 0.08)

### one variable with homoscedastic measurement error
(model_real <- lm(y ~ x_real))

(model_naiv <- lm(y ~ x_measured, x = TRUE))

(model_simex <- simex(model_naiv, SIMEXvariable = "x_measured", measurement.error = sd_me))
plot(model_simex)

### two variables with homoscedastic measurement errors
(model_real2 <- lm(y2 ~ x_real + x_real2))
(model_naiv2 <- lm(y2 ~ x_measured + x_measured2, x = TRUE))
(model_simex2 <- simex(model_naiv2, SIMEXvariable = c("x_measured", "x_measured2"),
         measurement.error = cbind(sd_me, sd_me2)))

plot(model_simex2)

\dontrun{
### one variable with increasing heteroscedastic measurement error
model_real

(mod_naiv1 <- lm(y ~ x_het1, x = TRUE))
(mod_simex1 <- simex(mod_naiv1, SIMEXvariable = "x_het1",
                measurement.error = sd_me_het1, asymptotic = FALSE))

plot(mod_simex1)

### two correlated variables with heteroscedastic measurement errors
(model_real3 <- lm(y3 ~ x_real + x_real3))
(mod_naiv2 <- lm(y3 ~ x_het1 + x_het2, x = TRUE))
(mod_simex2 <- simex(mod_naiv2, SIMEXvariable = c("x_het1", "x_het2"),
              measurement.error = cbind(sd_me_het1, sd_me_het2), asymptotic = FALSE))

plot(mod_simex2)

### two variables, one with homoscedastic, one with heteroscedastic measurement error
model_real2
(mod_naiv3 <- lm(y2 ~ x_measured + x_het2, x = TRUE))
(mod_simex3 <- simex(mod_naiv3, SIMEXvariable = c("x_measured", "x_het2"),
                    measurement.error = cbind(sd_me, sd_me_het2), asymptotic = FALSE))

### glm: two variables, one with homoscedastic, one with heteroscedastic measurement error
t <- x_real + 2*x_real2 + rnorm(100, sd = 0.01)
g <- 1 / (1 + exp(t))
u <- runif(100)
ybin <- as.numeric(u < g)

(logit_real <- glm(ybin ~ x_real + x_real2, family = binomial))
(logit_naiv <- glm(ybin ~ x_measured + x_het2, x = TRUE, family = binomial))
(logit_simex <- simex(logit_naiv, SIMEXvariable = c("x_measured", "x_het2"),
                    measurement.error = cbind(sd_me, sd_me_het2), asymptotic = FALSE))

summary(logit_simex)
print(logit_simex)
plot(logit_simex)

### polr: two variables, one with homoscedastic, one with heteroscedastic measurement error

if(require("MASS")) {# Requires MASS
yerr <- jitter(y, amount=1)
yfactor <- cut(yerr, 3, ordered_result=TRUE)

(polr_real <- polr(yfactor ~ x_real + x_real2))
(polr_naiv <- polr(yfactor ~ x_measured + x_het2, Hess = TRUE))
(polr_simex <- simex(polr_naiv, SIMEXvariable = c("x_measured", "x_het2"),
                    measurement.error = cbind(sd_me, sd_me_het2), asymptotic = FALSE))

summary(polr_simex)
print(polr_simex)
plot(polr_simex)
}
}

}
\references{
Cook, J.R. and Stefanski, L.A. (1994) Simulation-extrapolation estimation in
parametric measurement error models. \emph{Journal of the American Statistical
Association}, \bold{89}, 1314 -- 1328

Carroll, R.J., Küchenhoff, H., Lombard, F. and Stefanski L.A. (1996) Asymptotics
for the SIMEX estimator in nonlinear measurement error models. \emph{Journal
  of the American Statistical Association}, \bold{91}, 242 -- 250

Carrol, R.J., Ruppert, D., Stefanski, L.A. and Crainiceanu, C. (2006).
\emph{Measurement error in nonlinear models: A modern perspective.}, Second
Edition. London: Chapman and Hall.

Lederer, W. and Küchenhoff, H. (2006) A short introduction to the SIMEX and MCSIMEX. \emph{R News}, \bold{6(4)}, 26--31
}
\seealso{
\code{\link[simex]{mcsimex}} for discrete data with misclassification,
 \code{\link[stats]{lm}}, \code{\link[stats]{glm}}
}
\author{
Wolfgang Lederer,\email{wolfgang.lederer@gmail.com}

Heidi Seibold,\email{heidi.bold@gmail.com}
}
\keyword{models}
