% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcsimex.R
\name{mcsimex}
\alias{mcsimex}
\alias{print.mcsimex}
\alias{summary.mcsimex}
\alias{print.summary.mcsimex}
\alias{plot.mcsimex}
\alias{predict.mcsimex}
\alias{refit.mcsimex}
\title{Misclassification in models using MCSIMEX}
\usage{
mcsimex(model, SIMEXvariable, mc.matrix, lambda = c(0.5, 1, 1.5, 2),
  B = 100, fitting.method = "quadratic",
  jackknife.estimation = "quadratic", asymptotic = TRUE)

\method{plot}{mcsimex}(x, xlab = expression((1 + lambda)),
  ylab = colnames(b)[-1], ask = FALSE, show = rep(TRUE, NCOL(b) - 1),
  ...)

\method{predict}{mcsimex}(object, newdata, ...)

\method{print}{mcsimex}(x, digits = max(3, getOption("digits") - 3), ...)

\method{print}{summary.mcsimex}(x, digits = max(3, getOption("digits") -
  3), ...)

\method{summary}{mcsimex}(object, ...)

\method{refit}{mcsimex}(object, fitting.method = "quadratic",
  jackknife.estimation = "quadratic", asymptotic = TRUE, ...)
}
\arguments{
\item{model}{the naive model, the misclassified variable must be a factor}

\item{SIMEXvariable}{vector of names of the variables for which the MCSIMEX-method should be applied}

\item{mc.matrix}{if one variable is misclassified it can be a matrix. If more
than one variable is misclassified it must be a list of the misclassification
matrices, names must match with the SIMEXvariable names, column- and row-names
must match with the factor levels. If a special misclassification is desired,
the name of a function can be specified (see details)}

\item{lambda}{vector of exponents for the misclassification matrix (without 0)}

\item{B}{number of iterations for each lambda}

\item{fitting.method}{\code{linear}, \code{quadratic} and \code{loglinear}
are implemented (first 4 letters are enough)}

\item{jackknife.estimation}{specifying the extrapolation method for jackknife
variance estimation. Can be set to \code{FALSE} if it should not be performed}

\item{asymptotic}{logical, indicating if asymptotic variance estimation should
be done, the option \code{x = TRUE} must be enabled in the naive model}

\item{x}{object of class 'mcsimex'}

\item{xlab}{optional name for the X-Axis}

\item{ylab}{vector containing the names for the Y-Axis}

\item{ask}{ogical. If \code{TRUE}, the user is asked for input, before a new figure is drawn}

\item{show}{vector of logicals indicating for which variables a plot should be produced}

\item{\dots}{arguments passed to other functions}

\item{object}{object of class 'mcsimex'}

\item{newdata}{optionally, a data frame in which to look for variables with
which to predict. If omitted, the fitted linear predictors are used}

\item{digits}{number of digits to be printed}
}
\value{
An object of class 'mcsimex' which contains:

\item{coefficients}{corrected coefficients of the MCSIMEX model,}
\item{SIMEX.estimates}{the MCSIMEX-estimates of the coefficients for each lambda,}
\item{lambda}{the values of lambda,}
\item{model}{the naive model,}
\item{mc.matrix}{the misclassification matrix,}
\item{B}{the number of iterations,}
\item{extrapolation}{the model object of the extrapolation step,}
\item{fitting.method}{the fitting method used in the extrapolation step,}
\item{SIMEXvariable}{name of the SIMEXvariables,}
\item{call}{the function call,}
\item{variance.jackknife}{the jackknife variance estimates,}
\item{extrapolation.variance}{the model object of the variance extrapolation,}
\item{variance.jackknife.lambda}{the data set for the extrapolation,}
\item{variance.asymptotic}{the asymptotic variance estimates,}
\item{theta}{all estimated coefficients for each lambda and B,}
...
}
\description{
Implementation of the misclassification MCSIMEX algorithm as described by Küchenhoff, Mwalili and Lesaffre.
}
\details{
If \code{mc.matrix} is a function the first argument of that function must
be the whole dataset used in the naive model, the second argument must be
the exponent (lambda) for the misclassification. The function must return
a \code{data.frame} containing the misclassified \code{SIMEXvariable}. An
example can be found below.

Asymptotic variance estimation is only implemented for \code{lm} and \code{glm}

The loglinear fit has the form \code{g(lambda, GAMMA) = exp(gamma0 + gamma1 * lambda)}.
It is realized via the \code{log()} function. To avoid negative values the
minimum +1 of the dataset is added and after the prediction later substracted
\code{exp(predict(...)) - min(data) - 1}.

The 'log2' fit is fitted via the \code{nls()} function for direct fitting of
the model \code{y ~ exp(gamma.0 + gamma.1 * lambda)}. As starting values the
results of a LS-fit to a linear model with a log transformed response are used.
If \code{nls} does not converge, the model with the starting values is returned.

\code{refit()} refits the object with a different extrapolation function.
}
\section{Methods (by generic)}{
\itemize{
\item \code{plot}: Plots of the simulation and extrapolation

\item \code{predict}: Predict with mcsimex correction

\item \code{print}: Nice printing

\item \code{print}: Print summary nicely

\item \code{summary}: Summary for mcsimex

\item \code{refit}: Refits the model with a different extrapolation function
}}

\examples{
x <- rnorm(200, 0, 1.142)
z <- rnorm(200, 0, 2)
y <- factor(rbinom(200, 1, (1 / (1 + exp(-1 * (-2 + 1.5 * x -0.5 * z))))))
Pi <- matrix(data = c(0.9, 0.1, 0.3, 0.7), nrow = 2, byrow = FALSE)
dimnames(Pi) <- list(levels(y), levels(y))
ystar <- misclass(data.frame(y), list(y = Pi), k = 1)[, 1]
naive.model <- glm(ystar ~ x + z, family = binomial, x = TRUE, y = TRUE)
true.model  <- glm(y ~ x + z, family = binomial)
simex.model <- mcsimex(naive.model, mc.matrix = Pi, SIMEXvariable = "ystar")

op <- par(mfrow = c(2, 3))
invisible(lapply(simex.model$theta, boxplot, notch = TRUE, outline = FALSE,
                 names = c(0.5, 1, 1.5, 2)))
                 plot(simex.model)
simex.model2 <- refit(simex.model, "line")
plot(simex.model2)
par(op)

# example using polr from the MASS package
\dontrun{
if(require(MASS)) {
  yord <- cut((1 / (1 + exp(-1 * (-2 + 1.5 * x -0.5 * z)))), 3, ordered=TRUE)
  Pi3 <- matrix(data = c(0.8, 0.1, 0.1, 0.2, 0.7, 0.1, 0.1, 0.2, 0.7), nrow = 3, byrow = FALSE)
  dimnames(Pi3) <- list(levels(yord), levels(yord))
  ystarord <- misclass(data.frame(yord), list(yord = Pi3), k = 1)[, 1]
  naive.ord.model <- polr(ystarord ~ x + z, Hess = TRUE)
  simex.ord.model <- mcsimex(naive.ord.model, mc.matrix = Pi3,
      SIMEXvariable = "ystarord", asymptotic=FALSE)
}
}

# example for a function which can be supplied to the function mcsimex()
# "ystar" is the variable which is to be misclassified
# using the example above
\dontrun{
my.misclass <- function (datas, k) {
    ystar <- datas$"ystar"
    p1 <- matrix(data = c(0.75, 0.25, 0.25, 0.75), nrow = 2, byrow = FALSE)
    colnames(p1) <- levels(ystar)
    rownames(p1) <- levels(ystar)
    p0 <- matrix(data = c(0.8, 0.2, 0.2, 0.8), nrow = 2, byrow = FALSE)

    colnames(p0) <- levels(ystar)
    rownames(p0) <- levels(ystar)
    ystar[datas$x < 0] <-
    misclass(data.frame(ystar = ystar[datas$x < 0]), list(ystar = p1), k = k)[, 1]
    ystar[datas$x > 0] <-
    misclass(data.frame(ystar = ystar[datas$x > 0]), list(ystar = p0), k = k)[, 1]
    ystar <- factor(ystar)
    return(data.frame(ystar))}

simex.model.differential <- mcsimex(naive.model, mc.matrix = "my.misclass", SIMEXvariable = "ystar")
}

}
\references{
Küchenhoff, H., Mwalili, S. M.  and Lesaffre, E. (2006) A general method for
dealing with misclassification in regression: The Misclassification SIMEX.
\emph{Biometrics}, \bold{62}, 85 -- 96

Küchenhoff, H., Lederer, W. and E. Lesaffre. (2006) Asymptotic Variance Estimation
for the Misclassification SIMEX.
\emph{Computational Statistics and Data Analysis}, \bold{51}, 6197 -- 6211

Lederer, W. and Küchenhoff, H. (2006) A short introduction to the SIMEX and MCSIMEX. \emph{R News}, \bold{6(4)}, 26--31
}
\seealso{
\code{\link[simex]{misclass}}, \code{\link[simex]{simex}}
}
\author{
Wolfgang Lederer, \email{wolfgang.lederer@gmail.com}
}
\keyword{models}
