/* PSPP - computes sample statistics.
   Copyright (C) 1997-9, 2000 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
   02110-1301, USA. */

/* This #if encloses the rest of the file. */
#if !NO_HTML

#include <config.h>
#include "htmlP.h"
#include "error.h"
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "alloc.h"
#include "error.h"
#include "filename.h"
#include "getl.h"
#include "getline.h"
#include "getlogin_r.h"
#include "output.h"
#include "som.h"
#include "tab.h"
#include "version.h"
#include "mkfile.h"

#include "gettext.h"
#define _(msgid) gettext (msgid)

/* Prototypes. */
static int postopen (struct file_ext *);
static int preclose (struct file_ext *);

static int
html_open_global (struct outp_class *this UNUSED)
{
  return 1;
}

static int
html_close_global (struct outp_class *this UNUSED)
{
  return 1;
}

static int
html_preopen_driver (struct outp_driver *this)
{
  struct html_driver_ext *x;

  assert (this->driver_open == 0);
  msg (VM (1), _("HTML driver initializing as `%s'..."), this->name);

  this->ext = x = xmalloc (sizeof *x);
  this->res = 0;
  this->horiz = this->vert = 0;
  this->width = this->length = 0;

  this->cp_x = this->cp_y = 0;

  x->prologue_fn = NULL;

  x->file.filename = NULL;
  x->file.mode = "w";
  x->file.file = NULL;
  x->file.sequence_no = &x->sequence_no;
  x->file.param = this;
  x->file.postopen = postopen;
  x->file.preclose = preclose;

  x->sequence_no = 0;

  return 1;
}

static int
html_postopen_driver (struct outp_driver *this)
{
  struct html_driver_ext *x = this->ext;

  assert (this->driver_open == 0);
  if (NULL == x->file.filename)
    x->file.filename = xstrdup ("pspp.html");
	
  if (x->prologue_fn == NULL)
    x->prologue_fn = xstrdup ("html-prologue");

  msg (VM (2), _("%s: Initialization complete."), this->name);
  this->driver_open = 1;

  return 1;
}

static int
html_close_driver (struct outp_driver *this)
{
  struct html_driver_ext *x = this->ext;

  assert (this->driver_open);
  msg (VM (2), _("%s: Beginning closing..."), this->name);
  fn_close_ext (&x->file);
  free (x->prologue_fn);
  free (x->file.filename);
  free (x);
  msg (VM (3), _("%s: Finished closing."), this->name);
  this->driver_open = 0;
  
  return 1;
}


/* Link the image contained in FILENAME to the 
   HTML stream in file F. */
static int
link_image (struct file_ext *f, char *filename)
{
  fprintf (f->file,
	   "<IMG SRC=\"%s\"/>", filename);

  if (ferror (f->file))
    return 0;

  return 1;
}


/* Generic option types. */
enum
{
  boolean_arg = -10,
  string_arg,
  nonneg_int_arg
};

/* All the options that the HTML driver supports. */
static struct outp_option option_tab[] =
{
  /* *INDENT-OFF* */
  {"output-file",		1,		0},
  {"prologue-file",		string_arg,	0},
  {"", 0, 0},
  /* *INDENT-ON* */
};
static struct outp_option_info option_info;

static void
html_option (struct outp_driver *this, const char *key, const struct string *val)
{
  struct html_driver_ext *x = this->ext;
  int cat, subcat;

  cat = outp_match_keyword (key, option_tab, &option_info, &subcat);
  switch (cat)
    {
    case 0:
      msg (SE, _("Unknown configuration parameter `%s' for HTML device "
	   "driver."), key);
      break;
    case 1:
      free (x->file.filename);
      x->file.filename = xstrdup (ds_c_str (val));
      break;
    case string_arg:
      {
	char **dest;
	switch (subcat)
	  {
	  case 0:
	    dest = &x->prologue_fn;
	    break;
	  default:
	    assert (0);
            abort ();
	  }
	if (*dest)
	  free (*dest);
	*dest = xstrdup (ds_c_str (val));
      }
      break;
    default:
      assert (0);
    }
}

/* Variables for the prologue. */
struct html_variable
  {
    const char *key;
    const char *value;
  };
  
static struct html_variable *html_var_tab;

/* Searches html_var_tab for a html_variable with key KEY, and returns
   the associated value. */
static const char *
html_get_var (const char *key)
{
  struct html_variable *v;

  for (v = html_var_tab; v->key; v++)
    if (!strcmp (key, v->key))
      return v->value;
  return NULL;
}

/* Writes the HTML prologue to file F. */
static int
postopen (struct file_ext *f)
{
  static struct html_variable dict[] =
    {
      {"generator", 0},
      {"date", 0},
      {"user", 0},
      {"host", 0},
      {"title", 0},
      {"subtitle", 0},
      {"source-file", 0},
      {0, 0},
    };
  char login[128], host[128];
  time_t curtime;
  struct tm *loctime;

  struct outp_driver *this = f->param;
  struct html_driver_ext *x = this->ext;

  char *prologue_fn = fn_search_path (x->prologue_fn, config_path, NULL);
  FILE *prologue_file;

  char *buf = NULL;
  int buf_size = 0;

  if (prologue_fn == NULL)
    {
      msg (IE, _("Cannot find HTML prologue.  The use of `-vv' "
		 "on the command line is suggested as a debugging aid."));
      return 0;
    }

  msg (VM (1), _("%s: %s: Opening HTML prologue..."), this->name, prologue_fn);
  prologue_file = fopen (prologue_fn, "rb");
  if (prologue_file == NULL)
    {
      fclose (prologue_file);
      free (prologue_fn);
      msg (IE, "%s: %s", prologue_fn, strerror (errno));
      goto error;
    }

  dict[0].value = version;

  curtime = time (NULL);
  loctime = localtime (&curtime);
  dict[1].value = asctime (loctime);
  {
    char *cp = strchr (dict[1].value, '\n');
    if (cp)
      *cp = 0;
  }

  if (getenv ("LOGNAME") != NULL)
    str_copy_rpad (login, sizeof login, getenv ("LOGNAME"));
  else if (getlogin_r (login, sizeof login))
    strcpy (login, _("nobody"));
  dict[2].value = login;

#ifdef HAVE_UNISTD_H
  if (gethostname (host, 128) == -1)
    {
      if (errno == ENAMETOOLONG)
	host[127] = 0;
      else
	strcpy (host, _("nowhere"));
    }
#else
  strcpy (host, _("nowhere"));
#endif
  dict[3].value = host;

  dict[4].value = outp_title ? outp_title : "";
  dict[5].value = outp_subtitle ? outp_subtitle : "";

  getl_location (&dict[6].value, NULL);
  if (dict[6].value == NULL)
    dict[6].value = "<stdin>";

  html_var_tab = dict;
  while (-1 != getline (&buf, &buf_size, prologue_file))
    {
      char *buf2;
      int len;

      if (strstr (buf, "!!!"))
	continue;
      
      {
	char *cp = strstr (buf, "!title");
	if (cp)
	  {
	    if (outp_title == NULL)
	      continue;
	    else
	      *cp = '\0';
	  }
      }
      
      {
	char *cp = strstr (buf, "!subtitle");
	if (cp)
	  {
	    if (outp_subtitle == NULL)
	      continue;
	    else
	      *cp = '\0';
	  }
      }
      
      /* PORTME: Line terminator. */
      buf2 = fn_interp_vars (buf, html_get_var);
      len = strlen (buf2);
      fwrite (buf2, len, 1, f->file);
      if (buf2[len - 1] != '\n')
	putc ('\n', f->file);
      free (buf2);
    }
  if (ferror (f->file))
    msg (IE, _("Reading `%s': %s."), prologue_fn, strerror (errno));
  fclose (prologue_file);

  free (prologue_fn);
  free (buf);

  if (ferror (f->file))
    goto error;

  msg (VM (2), _("%s: HTML prologue read successfully."), this->name);
  return 1;

error:
  msg (VM (1), _("%s: Error reading HTML prologue."), this->name);
  return 0;
}

/* Writes the HTML epilogue to file F. */
static int
preclose (struct file_ext *f)
{
  fprintf (f->file,
	   "</BODY>\n"
	   "</HTML>\n"
	   "<!-- end of file -->\n");

  if (ferror (f->file))
    return 0;
  return 1;
}

static int
html_open_page (struct outp_driver *this)
{
  struct html_driver_ext *x = this->ext;

  assert (this->driver_open && this->page_open == 0);
  x->sequence_no++;
  if (!fn_open_ext (&x->file))
    {
      if (errno)
	msg (ME, _("HTML output driver: %s: %s"), x->file.filename,
	     strerror (errno));
      return 0;
    }

  if (!ferror (x->file.file))
    this->page_open = 1;
  return !ferror (x->file.file);
}

static int
html_close_page (struct outp_driver *this)
{
  struct html_driver_ext *x = this->ext;

  assert (this->driver_open && this->page_open);
  this->page_open = 0;
  return !ferror (x->file.file);
}

static void output_tab_table (struct outp_driver *, struct tab_table *);

static void
html_submit (struct outp_driver *this, struct som_entity *s)
{
  extern struct som_table_class tab_table_class;
  struct html_driver_ext *x = this->ext;
  
  assert (this->driver_open && this->page_open);
  if (x->sequence_no == 0 && !html_open_page (this))
    {
      msg (ME, _("Cannot open first page on HTML device %s."), this->name);
      return;
    }

  assert ( s->class == &tab_table_class ) ;

  switch (s->type) 
    {
    case SOM_TABLE:
      output_tab_table ( this, (struct tab_table *) s->ext);
      break;
    case SOM_CHART:
      link_image( &x->file, ((struct chart *)s->ext)->filename);
      break;
    default:
      assert(0);
      break;
    }

}

/* Write string S of length LEN to file F, escaping characters as
   necessary for HTML. */
static void
escape_string (FILE *f, char *s, int len)
{
  char *ep = &s[len];
  char *bp, *cp;

  for (bp = cp = s; bp < ep; bp = cp)
    {
      while (cp < ep && *cp != '&' && *cp != '<' && *cp != '>' && *cp)
	cp++;
      if (cp > bp)
	fwrite (bp, 1, cp - bp, f);
      if (cp < ep)
	switch (*cp++)
	  {
	  case '&':
	    fputs ("&amp;", f);
	    break;
	  case '<':
	    fputs ("&lt;", f);
	    break;
	  case '>':
	    fputs ("&gt;", f);
	    break;
	  case 0:
	    break;
	  default:
	    assert (0);
	  }
    }
}
  
/* Write table T to THIS output driver. */
static void
output_tab_table (struct outp_driver *this, struct tab_table *t)
{
  struct html_driver_ext *x = this->ext;
  
  if (t->nr == 1 && t->nc == 1)
    {
      fputs ("<P>", x->file.file);
      if (!ls_empty_p (t->cc))
	escape_string (x->file.file, ls_c_str (t->cc), ls_length (t->cc));
      fputs ("</P>\n", x->file.file);
      
      return;
    }

  fputs ("<TABLE BORDER=1>\n", x->file.file);
  
  if (!ls_empty_p (&t->title))
    {
      fprintf (x->file.file, "  <TR>\n    <TH COLSPAN=%d>", t->nc);
      escape_string (x->file.file, ls_c_str (&t->title),
		     ls_length (&t->title));
      fputs ("</TH>\n  </TR>\n", x->file.file);
    }
  
  {
    int r;
    unsigned char *ct = t->ct;

    for (r = 0; r < t->nr; r++)
      {
	int c;
	
	fputs ("  <TR>\n", x->file.file);
	for (c = 0; c < t->nc; c++, ct++)
	  {
            struct fixed_string *cc;
	    int tag;
	    char header[128];
	    char *cp;
            struct tab_joined_cell *j = NULL;

            cc = t->cc + c + r * t->nc;
	    if (*ct & TAB_JOIN) 
              {
                j = (struct tab_joined_cell *) ls_c_str (cc);
                cc = &j->contents;
                if (j->x1 != c || j->y1 != r)
                  continue; 
              }

	    if (r < t->t || r >= t->nr - t->b
		|| c < t->l || c >= t->nc - t->r)
	      tag = 'H';
	    else
	      tag = 'D';
	    cp = stpcpy (header, "    <T");
	    *cp++ = tag;
	    
	    switch (*ct & TAB_ALIGN_MASK)
	      {
	      case TAB_RIGHT:
		cp = stpcpy (cp, " ALIGN=RIGHT");
		break;
	      case TAB_LEFT:
		break;
	      case TAB_CENTER:
		cp = stpcpy (cp, " ALIGN=CENTER");
		break;
	      default:
		assert (0);
	      }

	    if (*ct & TAB_JOIN)
	      {
		if (j->x2 - j->x1 > 1)
		  cp = spprintf (cp, " COLSPAN=%d", j->x2 - j->x1);
		if (j->y2 - j->y1 > 1)
		  cp = spprintf (cp, " ROWSPAN=%d", j->y2 - j->y1);

                cc = &j->contents;
	      }
	    
	    strcpy (cp, ">");
	    fputs (header, x->file.file);
	    
	    if ( ! (*ct & TAB_EMPTY)  ) 
	      {
		char *s = ls_c_str (cc);
		size_t l = ls_length (cc);

		while (l && isspace ((unsigned char) *s))
		  {
		    l--;
		    s++;
		  }
	      
		escape_string (x->file.file, s, l);
	      }

	    fprintf (x->file.file, "</T%c>\n", tag);
	  }
	fputs ("  </TR>\n", x->file.file);
      }
  }
	      
  fputs ("</TABLE>\n\n", x->file.file);
}

static void
html_initialise_chart(struct outp_driver *d UNUSED, struct chart *ch)
{

  FILE  *fp;

  make_unique_file_stream(&fp, &ch->filename);

#ifdef NO_CHARTS
  ch->lp = 0;
#else
  ch->pl_params = pl_newplparams();
  ch->lp = pl_newpl_r ("png", 0, fp, stderr, ch->pl_params);
#endif

}

static void 
html_finalise_chart(struct outp_driver *d UNUSED, struct chart *ch)
{
  free(ch->filename);
}



/* HTML driver class. */
struct outp_class html_class =
{
  "html",
  0xfaeb,
  1,

  html_open_global,
  html_close_global,
  NULL,

  html_preopen_driver,
  html_option,
  html_postopen_driver,
  html_close_driver,

  html_open_page,
  html_close_page,

  html_submit,

  NULL,
  NULL,
  NULL,

  NULL,
  NULL,
  NULL,
  NULL,

  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,

  html_initialise_chart,
  html_finalise_chart

};

#endif /* !NO_HTML */

