########################################################################
##
## Copyright (C) 2003-2023 The Octave Project Developers
##
## See the file COPYRIGHT.md in the top-level directory of this
## distribution or <https://octave.org/copyright/>.
##
## This file is part of Octave.
##
## Octave is free software: you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## Octave is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with Octave; see the file COPYING.  If not, see
## <https://www.gnu.org/licenses/>.
##
########################################################################

BEGIN {
  declare_types = 0;
  generate_ops = 0;
  op_class = "";
  sparse = 0;
  ntypes = 0;
  make_inclusive_header = 0;
  make_header = 0;
  make_source = 0;

  if (build_file)
    {
      if (build_file == "mx-ops.h" || build_file == "smx-ops.h" || build_file == "vx-ops.h")
        {
          make_inclusive_header = 1;

          print "// DO NOT EDIT -- generated by mk-ops.awk";
          tmp = build_file;
          prefix = substr (tmp, 1, index (tmp, "-")-1);
          gsub (/[\.-]/, "_", tmp);
          printf ("#if ! defined (octave_%s)\n", tmp);
          printf ("#define octave_%s 1\n", tmp);
          print "#include \"octave-config.h\"";
        }
      else
        {
          if (match (build_file, /\.h$/))
            {
              make_header = 1;

              tmp = substr (build_file, 1, length (build_file)-2);
            }
          else if (match (build_file, /\.cc$/))
            {
              make_source = 1;

              tmp = substr (build_file, 1, length (build_file)-3);
            }

          prefix = substr (tmp, 1, index (tmp, "-")-1);
          tmp = substr (tmp, index (tmp, "-")+1);
          lhs_arg = substr (tmp, 1, index (tmp, "-")-1);
          rhs_arg = substr (tmp, index (tmp, "-")+1);
        }
    }
}
/^#/ {
  if ($2 == "types")
    declare_types = 1;
  else if ($3 == "operators")
    {
      op_class = $2;
      generate_ops = 1;
      declare_types = 0;

      if (op_class == "vector")
        sparse = 0;
      else if (op_class == "full-matrix")
        sparse = 0;
      else if (op_class == "sparse-matrix")
        sparse = 1;
      else
        {
          printf ("unrecognized object type: %s", op_class);
          exit (1);
        }
    }
  next;
} {
  if (declare_types)
    {
      ntypes++;

      if (NF == 5 || NF == 6)
        {
          if (NF == 6)
            core_type[ntypes] = $6;

          fwd_decl_ok[ntypes] = $5 == "YES";
          header[ntypes] = $4 == "NONE" ? "" : $4;
          class[ntypes] = $3;
          type[ntypes] = $2;
          tag[ntypes] = $1;
          rev_tag[$1] = ntypes;
        }
      else
        printf ("skipping line %d: %s\n", NR, $0);
    }
  else if (generate_ops)
    {
      if (NF >= 4 || (sparse && NF >= 5))
        {
          n = 0;

          result_tag_1 = $(++n);
          if (sparse)
            result_tag_2 = $(++n);
          lhs_tag = $(++n);
          rhs_tag = $(++n);
          op_type = $(++n);

          bin_ops = index (op_type, "B") != 0;
          cmp_ops = index (op_type, "C") != 0;
          eqne_ops = index (op_type, "E") != 0;
          bool_ops = index (op_type, "L") != 0;

          k = 0
          while (NF > n)
            bool_headers[k++] = $(++n);

          cc_file = sprintf ("%s-%s-%s.cc", prefix, lhs_tag, rhs_tag);
          h_file = sprintf ("%s-%s-%s.h", prefix, lhs_tag, rhs_tag);

          if (list_cc_files)
            {
              printf (" liboctave/operators/%s", cc_file);
              next;
            }

          if (list_h_files)
            {
              printf (" liboctave/operators/%s", h_file);
              next;
            }

          if (make_inclusive_header)
            {
              printf ("#include \"%s\"\n", h_file);
              next;
            }

          if (lhs_arg != lhs_tag || rhs_arg != rhs_tag)
            next;

          h_guard = sprintf ("octave_%s_%s_%s_h", prefix, lhs_tag, rhs_tag);

          result_num_1 = rev_tag[result_tag_1];
          if (sparse)
            result_num_2 = rev_tag[result_tag_2];
          lhs_num = rev_tag[lhs_tag];
          rhs_num = rev_tag[rhs_tag];

          result_type_1 = type[result_num_1];
          if (sparse)
            result_type_2 = type[result_num_2];
          lhs_type = type[lhs_num];
          rhs_type = type[rhs_num];

          lhs_core_type = core_type[lhs_num];
          rhs_core_type = core_type[rhs_num];

          result_header_1 = header[result_num_1];
          if (sparse)
            result_header_2 = header[result_num_2];
          lhs_header = header[lhs_num];
          rhs_header = header[rhs_num];

          lhs_class = class[lhs_num];
          rhs_class = class[rhs_num];

          if (make_header)
            {
              print "// DO NOT EDIT -- generated by mk-ops.awk";

              printf ("#if ! defined (%s)\n", h_guard);
              printf ("#define %s 1\n", h_guard);
              print "#include \"octave-config.h\"";

              if (result_header_1)
                {
                  if (result_fwd_decl_ok)
                    printf ("class %s\n", result_type_1);
                  else
                    printf ("#include \"%s\"\n", result_header_1);
                }

              if (result_header_2 && ! (result_header_2 == result_header_1))
                {
                  if (result_fwd_decl_ok)
                    printf ("class %s\n", result_type_2);
                  else
                    printf ("#include \"%s\"\n", result_header_2);
                }

              if (lhs_header && ! (lhs_header == result_header_1 || lhs_header == result_header_2))
                {
                  if (result_fwd_decl_ok)
                    printf ("class %s\n", lhs_type);
                  else
                    printf ("#include \"%s\"\n", lhs_header);
                }

              if (rhs_header && ! (rhs_header == lhs_header || rhs_header == result_header_1 || rhs_header == result_header_2))
                {
                  if (result_fwd_decl_ok)
                    printf ("class %s\n", rhs_type);
                  else
                    printf ("#include \"%s\"\n", rhs_header);
                }

              if (sparse)
                {
                  lhs_is_sparse = 0;
                  rhs_is_sparse = 0;

                  xlhs_tag = "NONE";
                  xrhs_tag = "NONE";

                  if (lhs_tag == "sbm")
                    {
                      lhs_is_sparse = 1;
                      xlhs_tag = "b";
                    }
                  else if (lhs_tag == "scm")
                    {
                      lhs_is_sparse = 1;
                      xlhs_tag = "cs";
                    }
                  else if (lhs_tag == "sm")
                    {
                      lhs_is_sparse = 1;
                      xlhs_tag = "s";
                    }
                  else if (lhs_tag == "bm" || lhs_tag == "cm" || lhs_tag == "m")
                    xlhs_tag = lhs_tag;

                  if (rhs_tag == "sbm")
                    {
                      rhs_is_sparse = 1;
                      xrhs_tag = "b";
                    }
                  else if (rhs_tag == "scm")
                    {
                      rhs_is_sparse = 1;
                      xrhs_tag = "cs";
                    }
                  else if (rhs_tag == "sm")
                    {
                      rhs_is_sparse = 1;
                      xrhs_tag = "s";
                    }
                  else if (rhs_tag == "bm" || rhs_tag == "cm" || rhs_tag == "m")
                    xrhs_tag = rhs_tag;

                  same_types = ((xlhs_tag == "b" && xrhs_tag == "bm") || (xlhs_tag == "cs" && xrhs_tag == "cm") || (xlhs_tag == "s" && xrhs_tag == "m") || (xlhs_tag == "bm" && xrhs_tag == "b") || (xlhs_tag == "cm" && xrhs_tag == "cs") || (xlhs_tag == "m" && xrhs_tag == "s"));

                  if (! (same_types || (rhs_is_sparse && lhs_is_sparse) || xlhs_tag == "NONE" || xrhs_tag == "NONE"))
                    {
                      printf ("#include \"mx-%s-%s.h\"\n", xlhs_tag, xrhs_tag);
                      printf ("#include \"mx-%s-%s.h\"\n", xrhs_tag, xlhs_tag);

                      xxlhs_tag = xlhs_tag;
                      xxrhs_tag = xrhs_tag;

                      if (xxlhs_tag == "cs")
                        xxlhs_tag = "cm";
                      else if (xxlhs_tag == "s")
                        xxlhs_tag = "m";
                      else if (xxlhs_tag == "b")
                        xxlhs_tag = "NONE";

                      if (xxrhs_tag == "cs")
                        xxrhs_tag = "cm";
                      else if (xxrhs_tag == "s")
                        xxrhs_tag = "m";
                      else if (xxrhs_tag == "b")
                        xxrhs_tag = "NONE";

                      if (! (xxlhs_tag == "NONE" || xxrhs_tag == "NONE" || xxlhs_tag == xxrhs_tag))
                        {
                          printf ("#include \"mx-%s-%s.h\"\n", xxlhs_tag, xxrhs_tag);
                          printf ("#include \"mx-%s-%s.h\"\n", xxrhs_tag, xxlhs_tag);
                        }

                      xxlhs_tag = xlhs_tag;
                      xxrhs_tag = xrhs_tag;

                      if (xxlhs_tag == "m")
                        xxlhs_tag = "nda";
                      else if (xxlhs_tag == "cm")
                        xxlhs_tag = "cnda";
                      else if (xxlhs_tag == "bm")
                        xxlhs_tag = "NONE";

                      if (xxrhs_tag == "m")
                        xxrhs_tag = "nda";
                      else if (xxrhs_tag == "cm")
                        xxrhs_tag = "cnda";
                      else if (xxrhs_tag == "bm")
                        xxrhs_tag = "NONE";

                      if (! (xxlhs_tag == "NONE" || xxrhs_tag == "NONE"))
                        {
                          printf ("#include \"mx-%s-%s.h\"\n", xxlhs_tag, xxrhs_tag);
                          printf ("#include \"mx-%s-%s.h\"\n", xxrhs_tag, xxlhs_tag);
                        }
                    }

                  printf ("#include \"Sparse-op-defs.h\"\n");
                }

              if (bin_ops)
                emit_bin_op_decls(sparse, lhs_class, rhs_class,
                                  result_type_1, result_type_2,
                                  lhs_type, rhs_type);

              if (cmp_ops)
                emit_cmp_op_decls(sparse, lhs_class, rhs_class,
                                  lhs_type, rhs_type);

              if (eqne_ops)
                emit_eqne_op_decls(sparse, lhs_class, rhs_class,
                                   lhs_type, rhs_type);

              if (bool_ops)
                emit_bool_op_decls(sparse, lhs_class, rhs_class,
                                   lhs_type, rhs_type);

              print "#endif";

              exit (0);
            }
          else if (make_source)
            {
              print "// DO NOT EDIT -- generated by mk-ops.awk";

              print "#if defined (HAVE_CONFIG_H)";
              print "#  include \"config.h\"";
              print "#endif";

              print "#include \"Array-util.h\"";

              printf ("#include \"%s\"\n", h_file);

              if (! sparse)
                printf ("#include \"mx-op-defs.h\"\n");

              for (i in bool_headers)
                {
                  printf ("#include \"%s\"\n", bool_headers[i]);
                  delete bool_headers[i];
                }

              if (result_header_1)
                printf ("#include \"%s\"\n", result_header_1);

              if (result_header_2 && ! (result_header_2 == result_header_1))
                printf ("#include \"%s\"\n", result_header_2);

              if (lhs_header && ! (lhs_header == result_header_1 || lhs_header == result_header_2))
                printf ("#include \"%s\"\n", lhs_header);

              if (rhs_header && ! (rhs_header == lhs_header || rhs_header == result_header_1 || rhs_header == result_header_2))
                printf ("#include \"%s\"\n", rhs_header);

              if (bin_ops)
                emit_bin_ops(sparse, lhs_class, rhs_class,
                             result_type_1, result_type_2, lhs_type, rhs_type);

              if (cmp_ops)
                emit_cmp_ops(sparse, lhs_class, rhs_class, lhs_type, rhs_type);

              if (eqne_ops)
                emit_eqne_ops(sparse, lhs_class, rhs_class,
                              lhs_type, rhs_type);

              if (bool_ops)
                emit_bool_ops(sparse, lhs_class, rhs_class,
                              lhs_type, rhs_type);

              exit (0);
            }
        }
      else
        printf ("skipping line %d: %s\n", NR, $0);
    }
}
END {
  if (make_inclusive_header)
    print "#endif";
}

function emit_bin_op_decl (result_type, operator, lhs_type, rhs_type)
{
  if (operator != "")
    printf ("  extern OCTAVE_API %s %s (const %s&, const %s&);\n",
            result_type, operator, lhs_type, rhs_type);
}

function emit_bin_op_decls_1 (result_type, lhs_type, rhs_type,
                              add_op, sub_op, mul_op, div_op)
{
  emit_bin_op_decl(result_type, add_op, lhs_type, rhs_type);
  emit_bin_op_decl(result_type, sub_op, lhs_type, rhs_type);
  emit_bin_op_decl(result_type, mul_op, lhs_type, rhs_type);
  emit_bin_op_decl(result_type, div_op, lhs_type, rhs_type);
}

function emit_sparse_bin_op_decls (result_type_1, result_type_2,
                                   lhs_type, rhs_type,
                                   add_op, sub_op, mul_op, div_op)
{
  emit_bin_op_decl(result_type_1, add_op, lhs_type, rhs_type);
  emit_bin_op_decl(result_type_1, sub_op, lhs_type, rhs_type);
  emit_bin_op_decl(result_type_2, mul_op, lhs_type, rhs_type);
  emit_bin_op_decl(result_type_2, div_op, lhs_type, rhs_type);
}

function emit_bin_op_decls (sparse, lhs_class, rhs_class,
                            result_type_1, result_type_2,
                            lhs_type, rhs_type)
{
  if (sparse)
    {
      if ((lhs_class == "SM" && rhs_class == "S") \
          || (lhs_class == "S" && rhs_class == "SM"))
        emit_sparse_bin_op_decls(result_type_1, result_type_2,
                                 lhs_type, rhs_type,
                                 "operator +", "operator -",
                                 "operator *", "operator /");

      else if ((lhs_class == "M" && rhs_class == "SM")    \
               || (lhs_class == "SM" && rhs_class == "M") \
               || (lhs_class == "SM" && rhs_class == "SM"))
        emit_sparse_bin_op_decls(result_type_1, result_type_2,
                                 lhs_type, rhs_type,
                                 "operator +", "operator -",
                                 "product", "quotient");
    }
  else
    {
      if ((lhs_class == "M" && rhs_class == "S") \
          || (lhs_class == "ND" && rhs_class == "S") \
          || (lhs_class == "S" \
              && (rhs_class == "M" || rhs_class == "ND" || rhs_class == "V")) \
          || (lhs_class == "V" && rhs_class == "S"))
        emit_bin_op_decls_1(result_type_1, lhs_type, rhs_type,
                            "operator +", "operator -",
                            "operator *", "operator /");

      else if ((lhs_class == "M" && rhs_class == "M")      \
               || (lhs_class == "ND" && rhs_class == "ND") \
               || (lhs_class == "V" && rhs_class == "V"))
        emit_bin_op_decls_1(result_type_1, lhs_type, rhs_type,
                            "operator +", "operator -",
                            "product", "quotient");

      else if ((lhs_class == "DM" && rhs_class == "M")  \
          || (lhs_class == "M" && rhs_class == "DM"))
        emit_bin_op_decls_1(result_type_1, lhs_type, rhs_type,
                            "operator +", "operator -", "operator *", "");

      else if (lhs_class == "DM" && rhs_class == "DM")
        emit_bin_op_decls_1(result_type_1, lhs_type, rhs_type,
                            "operator +", "operator -", "product", "");

      else if (lhs_class == "DM" && rhs_class == "S")
        emit_bin_op_decls_1(result_type_1, lhs_type, rhs_type,
                            "", "", "operator *", "operator /");

      else if ((lhs_class == "M" && rhs_class == "PM") \
               || (lhs_class == "PM" && rhs_class == "M") \
               || (lhs_class == "S" && rhs_class == "DM"))
        emit_bin_op_decls_1(result_type_1, lhs_type, rhs_type,
                            "", "", "operator *", "");
    }
}

function emit_cmp_op_decls_1(result_type, lhs_type, rhs_type)
{
  emit_bin_op_decl(result_type, "mx_el_lt", lhs_type, rhs_type);
  emit_bin_op_decl(result_type, "mx_el_le", lhs_type, rhs_type);
  emit_bin_op_decl(result_type, "mx_el_ge", lhs_type, rhs_type);
  emit_bin_op_decl(result_type, "mx_el_gt", lhs_type, rhs_type);
  emit_bin_op_decl(result_type, "mx_el_eq", lhs_type, rhs_type);
  emit_bin_op_decl(result_type, "mx_el_ne", lhs_type, rhs_type);
}

function emit_cmp_op_decls (sparse, lhs_class, rhs_class, lhs_type, rhs_type)
{
  if (sparse)
    {
      if ((lhs_class == "M" && rhs_class == "SM") \
          || (lhs_class == "SM" && (rhs_class == "M" || rhs_class == "S" || rhs_class == "SM")) \
          || (lhs_class == "S" && rhs_class == "SM"))
        emit_cmp_op_decls_1("SparseBoolMatrix", lhs_type, rhs_type);
    }
  else
    {
      if ((lhs_class == "M" && rhs_class == "M") \
          || (lhs_class == "M" && rhs_class == "S") \
          || (lhs_class == "S" && rhs_class == "M"))
        emit_cmp_op_decls_1("boolMatrix", lhs_type, rhs_type);
      else if ((lhs_class == "ND" && rhs_class == "ND") \
               || (lhs_class == "ND" && rhs_class == "S") \
               || (lhs_class == "S" && rhs_class == "ND"))
        emit_cmp_op_decls_1("boolNDArray", lhs_type, rhs_type);
    }
}

function emit_eqne_op_decls_1(result_type, lhs_type, rhs_type)
{
  emit_bin_op_decl(result_type, "mx_el_eq", lhs_type, rhs_type);
  emit_bin_op_decl(result_type, "mx_el_ne", lhs_type, rhs_type);
}

function emit_eqne_op_decls (sparse, lhs_class, rhs_class, lhs_type, rhs_type)
{
  if (sparse)
    {
      if ((lhs_class == "M" && rhs_class == "SM") \
          || (lhs_class == "SM" && rhs_class == "M"))
        emit_eqne_op_decls_1("SparseBoolMatrix", lhs_type, rhs_type);
    }

  ## No separate eqne ops for full-matrix or vector.
}

function emit_bool_op_decls_1 (result_type, lhs_type, rhs_type)
{
  emit_bin_op_decl(result_type, "mx_el_and", lhs_type, rhs_type);
  emit_bin_op_decl(result_type, "mx_el_or", lhs_type, rhs_type);
}

function emit_bool_op_decls_2 (result_type, lhs_type, rhs_type)
{
  emit_bin_op_decl(result_type, "mx_el_not_and", lhs_type, rhs_type);
  emit_bin_op_decl(result_type, "mx_el_not_or", lhs_type, rhs_type);
}

function emit_bool_op_decls_3 (result_type, lhs_type, rhs_type)
{
  emit_bin_op_decl(result_type, "mx_el_and_not", lhs_type, rhs_type);
  emit_bin_op_decl(result_type, "mx_el_or_not", lhs_type, rhs_type);
}

function emit_bool_op_decls (sparse, lhs_class, rhs_class, lhs_type, rhs_type)
{
  if (sparse)
    {
      if ((lhs_class == "M" && rhs_class == "SM") \
          || (lhs_class == "SM" && (rhs_class == "M" || rhs_class == "S" || rhs_class == "SM")) \
          || (lhs_class == "S" && rhs_class == "SM"))
        emit_bool_op_decls_1("SparseBoolMatrix", lhs_type, rhs_type);
    }
  else
    {
      if ((lhs_class == "M" && rhs_class == "M") \
          || (lhs_class == "M" && rhs_class == "S") \
          || (lhs_class == "S" && rhs_class == "M"))
        emit_bool_op_decls_1("boolMatrix", lhs_type, rhs_type);
      else if (lhs_class == "ND" && rhs_class == "S")
        {
          emit_bool_op_decls_1("boolNDArray", lhs_type, rhs_type);
          emit_bool_op_decls_2("boolNDArray", lhs_type, rhs_type);
        }
      else if (lhs_class == "S" && rhs_class == "ND")
        {
          emit_bool_op_decls_1("boolNDArray", lhs_type, rhs_type);
          emit_bool_op_decls_3("boolNDArray", lhs_type, rhs_type);
        }
      else if (lhs_class == "ND" && rhs_class == "ND")
        {
          emit_bool_op_decls_1("boolNDArray", lhs_type, rhs_type);
          emit_bool_op_decls_2("boolNDArray", lhs_type, rhs_type);
          emit_bool_op_decls_3("boolNDArray", lhs_type, rhs_type);
        }
    }
}

function emit_sparse_bin_ops (lhs_class, rhs_class, result_type_1,
                              result_type_2, lhs_type, rhs_type)
{
  printf ("SPARSE_%s%s_BIN_OPS (%s, %s, %s, %s)\n",
          lhs_class, rhs_class, result_type_1,
          result_type_2, lhs_type, rhs_type);
}

function emit_dm_bin_ops (lhs_class, rhs_class, result_type_1,
                          lhs_type, rhs_type)
{
  printf ("%s%s_BIN_OPS (%s, %s, %s)\n",
          lhs_class, rhs_class, result_type_1, lhs_type, rhs_type);
}

function emit_mm_bin_op (result_t, op, lhs_t, rhs_t, fcn)
{
  printf ("\n" \
          "%s\n" \
          "%s (const %s& m1, const %s& m2)\n" \
          "{\n" \
          "  return do_mm_binary_op<%s::element_type, %s::element_type, %s::element_type> (m1, m2, %s, %s, %s, \"%s\");\n" \
          "}\n",
          result_t, op, lhs_t, rhs_t, result_t, lhs_t, rhs_t,
          fcn, fcn, fcn, op);
}

function emit_mm_bin_ops (result_t, lhs_t, rhs_t)
{
  emit_mm_bin_op(result_t, "operator +", lhs_t, rhs_t, "mx_inline_add");
  emit_mm_bin_op(result_t, "operator -", lhs_t, rhs_t, "mx_inline_sub");
  emit_mm_bin_op(result_t, "product", lhs_t, rhs_t, "mx_inline_mul");
  emit_mm_bin_op(result_t, "quotient", lhs_t, rhs_t, "mx_inline_div");
}

function emit_bin_ops (sparse, lhs_class, rhs_class,
                       result_type_1, result_type_2, lhs_type, rhs_type)
{
  if (sparse)
    emit_sparse_bin_ops(lhs_class, rhs_class, result_type_1,
                        result_type_2, lhs_type, rhs_type);
 else if ((lhs_class == "DM" && rhs_class == "M") \
          || (lhs_class == "M" && rhs_class == "DM"))
    emit_dm_bin_ops(lhs_class, rhs_class, result_type_1,
                    lhs_type, rhs_type);
  else if (lhs_class == "M" && rhs_class == "M")
    emit_mm_bin_ops(result_type_1, lhs_type, rhs_type);
  else
    printf ("%s%s_BIN_OPS (%s, %s, %s)\n",
            lhs_class, rhs_class, result_type_1, lhs_type, rhs_type);
}

function emit_cmp_ops (sparse, lhs_class, rhs_class, lhs_type, rhs_type)
{
  if (sparse)
    printf ("SPARSE_%s%s_CMP_OPS (%s, %s)\n",
            lhs_class, rhs_class, lhs_type, rhs_type);
  else
    printf ("%s%s_CMP_OPS (%s, %s)\n",
            lhs_class, rhs_class, lhs_type, rhs_type);
}

function emit_eqne_ops (sparse, lhs_class, rhs_class, lhs_type, rhs_type)
{
  if (sparse)
    printf ("SPARSE_%s%s_EQNE_OPS (%s, %s)\n",
            lhs_class, rhs_class, lhs_type, rhs_type);

  ## No separate eqne ops for full-matrix or vector.
}

function emit_bool_ops (sparse, lhs_class, rhs_class, lhs_type, rhs_type)
{
  if (sparse)
    printf ("SPARSE_%s%s_BOOL_OPS (%s, %s)\n",
            lhs_class, rhs_class, lhs_type, rhs_type);
  else
    printf ("%s%s_BOOL_OPS (%s, %s)\n",
            lhs_class, rhs_class, lhs_type, rhs_type);
}
