/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include        "bseiosave.h"
#include	"bsesong.h"
#include	"bsepattern.h"
#include	"bseinstrument.h"
#include	<stdio.h>
#include	<time.h>


/* --- defines --- */
/* conditionally write strings
 */
#define	CDUMP_STRING( arg_name, string )	G_STMT_START { \
  if (string && *string) \
  { \
    need_newline = FALSE; \
    fputc ('\n', f_out); \
    fputs (mkspace (indent), f_out); \
    fputc ('(', f_out); \
    fputs (#arg_name, f_out); \
    fputc (' ', f_out); \
    fputc ('"', f_out); \
    fputs ((string), f_out); \
    fputc ('"', f_out); \
    fputc (')', f_out); \
  } \
} G_STMT_END

/* write an integer
 */
#define	DUMP_INT( arg_name, num )		G_STMT_START { \
  need_newline = FALSE; \
  fprintf (f_out, "\n%s(%s %ld)", mkspace (indent), #arg_name, (glong) (num)); \
} G_STMT_END

/* write a flag
 */
#define	DUMP_BOOL( arg_true, arg_false, bool )	G_STMT_START { \
  need_newline = FALSE; \
  fputc ('\n', f_out); \
  fputs (mkspace (indent), f_out); \
  fputc ('(', f_out); \
  fputs ((bool) ? #arg_true : #arg_false, f_out); \
  fputc (')', f_out); \
} G_STMT_END

/* write a date
 */
#define	DUMP_DATE(arg_name, date)		G_STMT_START { \
  struct tm *bt; \
  time_t tt = (date); \
  bt = localtime (&tt); \
  need_newline = FALSE; \
  fprintf (f_out, "\n%s(%s \"%04d-%02d-%02d %02d:%02d:%02d\")", \
           mkspace (indent), #arg_name, \
	   bt->tm_year + 1900, bt->tm_mon + 1, bt->tm_mday, \
           bt->tm_hour, bt->tm_min, bt->tm_sec); \
} G_STMT_END


/* --- prototypes --- */
static inline const gchar*const	mkspace	(guint	n)	G_GNUC_CONST;


/* --- functions --- */
static inline const gchar*const
mkspace (guint n)
{
  register const guint l = 128;
  static gchar *z_buf = NULL;

  if (!z_buf)
    {
      z_buf = g_new (gchar, l + 1);
      memset (z_buf, ' ', l);
      z_buf[l] = 0;
    }

  n = MIN (l, n);
  return z_buf + l - n;
}

static inline BseNote*
get_note (BsePattern *pattern,
	  guint	      channel,
	  guint	      row)
{
  BseNote *note;
  
  while (row >= pattern->n_rows)
    {
      channel++;
      row -= pattern->n_rows;
    }
  if (channel < pattern->n_channels)
    {
      note = bse_pattern_get_note (pattern, channel, row);
      if (note->note == BSE_NOTE_VOID &&
	  !note->instrument &&
	  !note->next_effect)
	note = NULL;
    }
  else
    note = NULL;
  
  return note;
}

static gboolean
dump_note (FILE    *f_out,
	   guint    indent,
	   BseNote *note)
{
  fputc ('\n', f_out);
  fputs (mkspace (indent), f_out);
  fputc ('(', f_out);
  
  if (!note)
    {
      fputs ("skip)", f_out);
      return FALSE;
    }

  fputs ("note", f_out);
  fputc (' ', f_out);

  if (note->note == BSE_NOTE_VOID)
    fputs ("void", f_out);
  else
    {
      gchar *name;

      name = bse_note_2_string (note->note);
      fputs (name, f_out);
      g_free (name);
    }

  if (note->instrument)
    fprintf (f_out, " %02u",
	     bse_instrument_get_guid (note->instrument));

  fputc (')', f_out);

  return FALSE;
}

static gboolean
dump_pattern (FILE       *f_out,
	      guint	  indent,
	      BsePattern *pattern)
{
  gboolean need_newline = FALSE;
  guint r, c, n_r, n_c;
  gboolean force_next;
  
  fprintf (f_out, "\n%s(pattern\t;; Guid #%d",
	   mkspace (indent),
	   bse_pattern_get_guid (pattern));
  need_newline = TRUE;
  indent += 9;

  CDUMP_STRING (name, pattern->name);
  CDUMP_STRING (blurb, pattern->blurb);

  n_c = n_r = 0;
  force_next = get_note (pattern, 0, 0) || get_note (pattern, 0, 1);
  for (c = 0; c < pattern->n_channels; c++)
    {
      for (r = 0; r < pattern->n_rows; r++)
	{
	  BseNote *note, *note2, *note3;

	  note = get_note (pattern, c, r);
	  note2 = get_note (pattern, c, r + 1);
	  note3 = get_note (pattern, c, r + 2);

	  if (note || force_next)
	    {
	      if (n_c != c)
		DUMP_INT (set-channel, c + 1);
	      if (n_r != r)
		DUMP_INT (set-row, r + 1);

	      need_newline = dump_note (f_out, indent, note);
	      force_next = !note2 && note3;

	      n_c = c;
	      n_r = r + 1;
	      if (n_r >= pattern->n_rows)
		{
		  n_r = 0;
		  n_c++;
		}
	    }
	}
    }

  fputs (need_newline ? "\n)" : ")", f_out);

  return FALSE;
}

static gboolean
dump_instrument (FILE          *f_out,
		 guint	        indent,
		 BseInstrument *instrument)
{
  gboolean need_newline = FALSE;

  if (!instrument->type == BSE_INSTRUMENT_SAMPLE)
    return FALSE;
  
  fprintf (f_out, "\n%s(sample-instrument \"%s\"\t;; Guid #%d",
	   mkspace (indent),
	   instrument->sample->name,
	   bse_instrument_get_guid (instrument));
  need_newline = TRUE;
  indent += 19;

  CDUMP_STRING (sample-path, instrument->sample->file_name);
  CDUMP_STRING (name, instrument->name);
  CDUMP_STRING (blurb, instrument->blurb);
  DUMP_BOOL (with-interpolation, without-interpolation, instrument->interpolation);
  DUMP_BOOL (with-polyphony, without-polyphony, instrument->polyphony);
  DUMP_INT (volume, instrument->volume);
  DUMP_INT (balance, instrument->balance);
  DUMP_INT (transpose, instrument->transpose);
  DUMP_INT (fine-tune, instrument->fine_tune);
  DUMP_INT (delay-time, instrument->delay_time);
  DUMP_INT (attack-time, instrument->attack_time);
  DUMP_INT (attack-level, instrument->attack_level);
  DUMP_INT (decay-time, instrument->decay_time);
  DUMP_INT (sustain-time, instrument->sustain_time);
  DUMP_INT (sustain-level, instrument->sustain_level);
  DUMP_INT (release-time, instrument->release_time);
  DUMP_INT (release-level, instrument->release_level);

  fputs (need_newline ? "\n)" : ")", f_out);

  return FALSE;
}

void
bse_io_save_song_test (const gchar    *file_name,
		       BseSong        *song)
{
  BseIoDataFlags io_flags;
  GList *list;
  FILE *f_out;
  guint indent;
  gboolean need_newline = FALSE;

  g_return_if_fail (file_name != NULL);
  g_return_if_fail (song != NULL);

  f_out = fopen (file_name, "w");
  if (!f_out)
    return;
  
  /* identification
   */
  io_flags = BSE_IO_DATA_SONGS;
  fprintf (f_out, "(BSE-Data V1 %010o);\t\t-*- scheme -*-\n\n", io_flags);
  indent = 0;
  
  /* song header
   */
  fprintf (f_out, "%s(song \"%s\" %d %d\t\t;; name n-channels pattern-length",
	   mkspace (indent),
	   bse_song_get_name (song),
	   song->n_channels,
	   song->pattern_length);
  indent += 6;
  need_newline = TRUE;

  /* BseSong fields
   */
  CDUMP_STRING (author, song->author);
  DUMP_DATE (created, song->creation_time);
  CDUMP_STRING (copyright, song->copyright);
  CDUMP_STRING (blurb, song->blurb);
  CDUMP_STRING (bse-version, song->bse_version);
  DUMP_DATE (modified, song->last_modification_time);
  DUMP_INT (bpm, song->bpm);
  DUMP_INT (volume, song->master_volume);


  /* instruments
   */
  for (list = song->instruments; list; list = list->next)
    need_newline = dump_instrument (f_out, indent, list->data);

  /* patterns
   */
  for (list = song->patterns; list; list = list->next)
    need_newline = dump_pattern (f_out, indent, list->data);

  /* terminate song
   */
  fputs (need_newline ? "\n)\n" : ")\n", f_out);

  fclose (f_out);
}
