/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bseenums.h"


/* BseLoopType
 */
static const gchar *bse_loop_types[BSE_LOOP_LAST] = {
  "BSE_LOOP_NONE",
  "BSE_LOOP_PATTERN",
  "BSE_LOOP_PATTERN_ROWS",
  "BSE_LOOP_SONG",
};
const gchar*
bse_loop_type_get (BseLoopType type)
{
  g_return_val_if_fail (type >= 0, NULL);
  g_return_val_if_fail (type < BSE_LOOP_LAST, NULL);

  return bse_loop_types[type];
}

/* BseSampleType
 */
static const gchar *bse_sample_types[BSE_SAMPLE_LAST << 1] = {
  "BSE_SAMPLE_EFFECT_MUNKS",	"EffectMunks",
  "BSE_SAMPLE_NOTE_MUNKS",	"NoteMunks",
};
const gchar*
bse_sample_type_get (BseSampleType type)
{
  g_return_val_if_fail (type >= 0, NULL);
  g_return_val_if_fail (type < BSE_SAMPLE_LAST, NULL);

  return bse_sample_types[type << 1];
}
const gchar*
bse_sample_type_get_nick (BseSampleType type)
{
  g_return_val_if_fail (type >= 0, NULL);
  g_return_val_if_fail (type < BSE_SAMPLE_LAST, NULL);

  return bse_sample_types[(type << 1) | 1];
}

/* BseEffectType
 */
static const gchar *bse_effect_types[BSE_EFFECT_LAST << 1] = {
  "BSE_EFFECT_NONE",		"None",
  "BSE_EFFECT_VOLUME",		"Volume",
  "BSE_EFFECT_VIBRATO",		"Vibrato",
  "BSE_EFFECT_SLIDE",		"Slide",
  "BSE_EFFECT_PORTAMENTO",	"Portamento",
};
const gchar*
bse_effect_type_get (BseEffectType type)
{
  g_return_val_if_fail (type >= 0, NULL);
  g_return_val_if_fail (type < BSE_EFFECT_LAST, NULL);

  return bse_effect_types[type << 1];
}
const gchar*
bse_effect_type_get_nick (BseEffectType type)
{
  g_return_val_if_fail (type >= 0, NULL);
  g_return_val_if_fail (type < BSE_EFFECT_LAST, NULL);

  return bse_effect_types[(type << 1) | 1];
}

/* BseInstrumentType
 */
static const gchar *bse_instrument_types[BSE_INSTRUMENT_LAST << 1] = {
  "BSE_INSTRUMENT_NONE",	"None",
  "BSE_INSTRUMENT_SAMPLE",	"Sample",
  "BSE_INSTRUMENT_MIDI",	"Midi",
};
const gchar*
bse_instrument_type_get (BseInstrumentType type)
{
  g_return_val_if_fail (type >= 0, NULL);
  g_return_val_if_fail (type < BSE_INSTRUMENT_LAST, NULL);

  return bse_instrument_types[type << 1];
}
const gchar*
bse_instrument_type_get_nick (BseInstrumentType type)
{
  g_return_val_if_fail (type >= 0, NULL);
  g_return_val_if_fail (type < BSE_INSTRUMENT_LAST, NULL);

  return bse_instrument_types[(type << 1) | 1];
}

/* BseStreamAttribMask
 */
static const gchar* bse_stream_attrib_masks[] = {
  "BSE_SA_NONE",
  "BSE_SA_N_CHANNELS",
  "BSE_SA_PLAY_FREQUENCY",
  "BSE_SA_REC_FREQUENCY",
  "BSE_SA_FRAGMENT_SIZE",
};
const gchar*
bse_stream_attrib_mask_get (BseStreamAttribMask mask)
{
  guint i;

  for (i = 0; (1 << i) & BSE_SA_MASK; i++)
    {
      if (mask & (1 << i))
	break;
    }
  mask = (1 << i) & BSE_SA_MASK;

  return bse_stream_attrib_masks[mask];
}

/* BseIoDataFlags
 */
static const gchar* bse_iodata_flags[] = {
  "BSE_IO_DATA_NONE",
  "BSE_IO_DATA_BINARY_APPENDIX",
  "BSE_IO_DATA_SAMPLES",
  "BSE_IO_DATA_SONGS",
};
const gchar*
bse_iodata_flags_get (BseIoDataFlags flag)
{
  guint i;

  for (i = 0; (1 << i) & BSE_IO_DATA_MASK; i++)
    {
      if (flag & (1 << i))
	break;
    }
  flag = (1 << i) & BSE_IO_DATA_MASK;

  return bse_iodata_flags[flag];
}

/* BseErrorType
 */
static const gchar *bse_error_types[BSE_ERROR_LAST << 1] = {
  "BSE_ERROR_NONE",			"Everything went well",
  "BSE_ERROR_UNSPECIFIED",		"Unspecified error",
  "BSE_ERROR_INTERNAL",			"Internal error",
  "BSE_ERROR_FILE_IO",			"File I/O error",
  "BSE_ERROR_FILE_NOT_FOUND",		"File not found",
  "BSE_ERROR_FILE_TOO_SHORT",		"File too short",
  "BSE_ERROR_FILE_TOO_LONG",		"File too long",
  "BSE_ERROR_FORMAT_UNSUPPORTED",	"Format is not supported",
  "BSE_ERROR_FORMAT_TOO_NEW",		"Format is too new",
  "BSE_ERROR_FORMAT_TOO_OLD",		"Format is too old",
  "BSE_ERROR_HEADER_CORRUPT",		"Header is corrupt",
  "BSE_ERROR_SUB_HEADER_CORRUPT",	"Sub-header is corrupt",
  "BSE_ERROR_DATA_CORRUPT",		"Data is corrupt",
  "BSE_ERROR_BINARY_DATA_CORRUPT",	"Binary data is corrupt",
  "BSE_ERROR_STREAM_INTERNAL",		"Internal error in stream",
  "BSE_ERROR_STREAM_INVALID",		"Stream handle is invalid",
  "BSE_ERROR_STREAM_SUSPENDED",		"Stream is suspended",
  "BSE_ERROR_STREAM_DEVICE_BUSY",	"Stream device is busy",
  "BSE_ERROR_STREAM_READ_DENIED",	"Stream denies read access",
  "BSE_ERROR_STREAM_READ_FAILED",	"Reading from stream failed",
  "BSE_ERROR_STREAM_WRITE_DENIED",	"Stream denies write access",
  "BSE_ERROR_STREAM_WRITE_FAILED",	"Writing to stream failed",
  "BSE_ERROR_STREAM_IO",		"Stream I/O error",
  "BSE_ERROR_STREAM_SET_ATTRIB",	"Setting stream attribute failed",
};
const gchar*
bse_error_type_get (BseErrorType error)
{
  if (error < BSE_ERROR_NONE ||
      error >= BSE_ERROR_LAST)
    error = BSE_ERROR_UNSPECIFIED;

  return bse_error_types[error << 1];
}
const gchar*
bse_error_type_get_description (BseErrorType error)
{
  if (error < BSE_ERROR_NONE ||
      error >= BSE_ERROR_LAST)
    error = BSE_ERROR_UNSPECIFIED;
    
  return bse_error_types[(error << 1) | 1];
}
