package phrasestream

import (
	"bytes"
	"fmt"
	"strings"
	"testing"
	"unicode/utf8"
)

func TestMask(t *testing.T) {
	buf := new(bytes.Buffer)
	if err := Mask(buf, "foo", []byte("bar")); err != nil {
		t.Fatal(err)
	}

	if buf.String() != "[MASKED]" {
		t.Errorf("expected [MASKED], got %v", buf.String())
	}
}

func TestPassthrough(t *testing.T) {
	buf := new(bytes.Buffer)
	if err := Passthrough(buf, "foo", []byte("bar")); err != nil {
		t.Fatal(err)
	}

	if buf.String() != "foobar" {
		t.Errorf("expected foobar, got %v", buf.String())
	}
}

func TestStops(t *testing.T) {
	tests := []struct {
		input    string
		stop     func(byte) bool
		expected string
	}{
		{
			input:    "foobar\r",
			stop:     StopCRLF,
			expected: "foobar",
		},
		{
			input:    "foobar\n",
			stop:     StopCRLF,
			expected: "foobar",
		},
		{
			input:    "foobar\r\n",
			stop:     StopCRLF,
			expected: "foobar",
		},
		{
			input:    "foobar\r\n",
			stop:     StopNonAlphanumeric,
			expected: "foobar",
		},
		{
			input:    "hello_world",
			stop:     StopNonAlphanumeric,
			expected: "hello",
		},
		{
			input:    "abcXYZ0089 000",
			stop:     StopNonAlphanumeric,
			expected: "abcXYZ0089",
		},
		{
			input:    "hello\xffworld",
			stop:     StopNonASCII,
			expected: "hello",
		},
		{
			input:    "foo bar",
			stop:     StopNonASCIIPrint,
			expected: "foo",
		},
		{
			input:    "にほんご",
			stop:     StopAny("ん", false),
			expected: "にほ",
		},
	}

	for _, tc := range tests {
		t.Run(fmt.Sprintf("%v=%v", tc.input, tc.expected), func(t *testing.T) {
			var output []byte
			for _, b := range []byte(tc.input) {
				if tc.stop(b) {
					break
				}
				output = append(output, b)
			}

			result := strings.TrimRight(string(output), string(utf8.RuneError))
			if string(result) != tc.expected {
				t.Fatalf("expected %v, got %v", tc.expected, string(result))
			}
		})
	}
}
