package phrasestream

import (
	"io"
	"strings"
	"unicode/utf8"
)

// Mask writes [MASKED] to the provided writer.
func Mask(w io.Writer, _ string, buf []byte) error {
	_, err := w.Write([]byte("[MASKED]"))
	return err
}

// Passthrough writes phrase + buf to the provided writer.
func Passthrough(w io.Writer, phrase string, buf []byte) error {
	if _, err := w.Write([]byte(phrase)); err != nil {
		return err
	}
	if _, err := w.Write(buf); err != nil {
		return err
	}

	return nil
}

// StopCRLF returns false if b is a carriage return (CR) and line feed (LF).
func StopCRLF(b byte) bool {
	return b == '\n' || b == '\r'
}

// StopNonAlphanumeric returns false if b isn't alphanumeric (ASCII).
func StopNonAlphanumeric(b byte) bool {
	return !((b >= '0' && b <= '9') || (b >= 'A' && b <= 'Z') || (b >= 'a' && b <= 'z'))
}

// StopNonASCII returns false if b isn't an ASCII character.
func StopNonASCII(b byte) bool {
	return b > 0x7f
}

// StopNonASCIIPrint returns false if b isn't a printable ASCII character.
//
// The space character is considered non-printable for this function.
func StopNonASCIIPrint(b byte) bool {
	return b < '!' || b > '~'
}

// StopAny returns false if b is any character provided, unless inverse is set.
func StopAny(chars string, inverse bool) func(b byte) bool {
	fn := func(r rune) bool {
		/*var contains bool
		for _, rn := range chars {
			if rn == r {
				contains = true
				break
			}
		}*/

		contains := strings.ContainsAny(chars, string(r))
		if inverse {
			return !contains
		}
		return contains
	}

	return StopRune(fn)
}

// StopRune wraps a regular Stop function to accept runes instead, useful for
// matching utf8 characters.
//
// The stop function returned from this tracks state, so cannot be used with
// more than one phrasestream instance.
//
// The buffer returned to a callback will likely terminate in an invalid UTF-8
// character, this is because we cannot stop until we have a full rune, and by
// that point, some bytes of a rune might have been appended to the buffer.
//
// This can be easily resolved by converting the buffer to a string, and
// trimming utf8.RuneError.
func StopRune(fn func(r rune) bool) func(b byte) bool {
	var rb [4]byte
	var width int
	return func(b byte) bool {
		rb[width] = b
		width++
		if utf8.FullRune(rb[:width]) {
			r, _ := utf8.DecodeRune(rb[:width])
			width = 0
			return fn(r)
		}
		return false
	}
}
