package addmask

import (
	"slices"
	"testing"
)

func TestEncode(t *testing.T) {
	tests := []struct {
		input    string
		expected []string
	}{
		{
			input: "hello world",
			expected: []string{
				"ZWxsbyB3b3JsZA==",
				"aGVsbG8gd29ybGQ=",
				"bGxvIHdvcmxk",
				"hello world",
				"hello%20world",
			},
		},
		{
			input: "hello%0Aworld",
			expected: []string{
				"ZWxsbwp3b3JsZA==",
				"aGVsbG8Kd29ybGQ=",
				"bGxvCndvcmxk",
				"hello\nworld",
				"hello%0Aworld",
				"hello&#xA;world",
				`hello\nworld`,
			},
		},
		{
			input: "foobar\xffにほんご",
			expected: []string{
				"Zm9vYmFy/+OBq+OBu+OCk+OBlA==",
				"b29iYXL/44Gr44G744KT44GU",
				"b2Jhcv/jgavjgbvjgpPjgZQ=",
				"foobar%FF%E3%81%AB%E3%81%BB%E3%82%93%E3%81%94",
				"foobar\\ufffdにほんご",
				"foobar�にほんご",
				"foobar\xffにほんご",
			},
		},
		{
			input:    `"`,
			expected: []string{"\"", "%22", "&#34;", "Ig==", "\\\""},
		},
		{
			input:    `"a"`,
			expected: []string{"\"a\"", "%22a%22", "&#34;a&#34;", "Ig==", "ImEi", "YSI=", "\\\"a\\\""},
		},
		{
			input: `"12345678"`,
			expected: []string{
				"\"12345678\"",
				"%2212345678%22",
				"&#34;12345678&#34;",
				"12345678",
				"IjEyMzQ1Njc4Ig==",
				"MTIzNDU2Nzgi",
				"MjM0NTY3OCI=",
				"\\\"12345678\\\"",
			},
		},
	}

	for _, tc := range tests {
		t.Run(tc.input, func(t *testing.T) {
			result := encode(tc.input)

			if !slices.Equal(tc.expected, result) {
				t.Errorf("expected %q, got %q", tc.expected, result)
			}
		})
	}
}
