/*
** Copyright (C) 2004 Andrew R. Baker <andrewb@snort.org>
** 
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "barnyard.h"
#include "util.h"
#include "input-plugins/dp_alert.h"
#include "output-plugins/op_plugbase.h"
#include "classification.h"
#include "sid.h"
#include <netinet/in.h>

/* Output plug-in API functions */
static int OpAlertConsole_Setup(OutputPlugin *, char *args);
static int OpAlertConsole_Exit(OutputPlugin *);
static int OpAlertConsole_Start(OutputPlugin *, void *);
static int OpAlertConsole_Stop(OutputPlugin *);
static int OpAlertConsole_LogConfig(OutputPlugin *);
static int OpAlertConsole(void *, void *);

/* Initialize and register this output plug-in */
void OpAlertConsole_Init()
{
    OutputPlugin *outputPlugin;

    /* Register the output plugin */
    outputPlugin = RegisterOutputPlugin("alert_console", "alert");
    
    /* Set the functions */
    outputPlugin->setupFunc     = OpAlertConsole_Setup;
    outputPlugin->exitFunc      = OpAlertConsole_Exit;
    outputPlugin->startFunc     = OpAlertConsole_Start;
    outputPlugin->stopFunc      = OpAlertConsole_Stop;
    outputPlugin->logConfigFunc = OpAlertConsole_LogConfig;
    outputPlugin->outputFunc    = OpAlertConsole;
}

static int OpAlertConsole_Setup(OutputPlugin *outputPlugin, char *args)
{
    /* No instance specific data to setup */
    return 0;
}

static int OpAlertConsole_Exit(OutputPlugin *outputPlugin)
{
    /* No instance specific data to destroy */
    return 0;
}

static int OpAlertConsole_Start(OutputPlugin *outputPlugin, 
        void *spool_header)
{
    /* No instance specific handles to open */
    if(pv.verbose > 2)
        OpAlertConsole_LogConfig(outputPlugin);

    return 0;
}

static int OpAlertConsole_Stop(OutputPlugin *outputPlugin)
{
    /* No instance specific handles to close */
    return 0;
}

static int OpAlertConsole_LogConfig(OutputPlugin *outputPlugin)
{
    if(!outputPlugin)
        return -1;

    LogMessage("OpAlertConsole configured\n");
    /* No instance specific configuration to display */

    return 0;
}

/* output function */
static int OpAlertConsole(void *context, void *data)
{
    char timestamp[256];
    UnifiedAlertRecord *alert = (UnifiedAlertRecord *)data;
    ClassType *class;
    Sid *sid = NULL;
    char sip[16];
    char dip[16];

    if(!data)
        return -1;

    sid = GetSid(alert->event.sig_generator, alert->event.sig_id);
    class = GetClassType(alert->event.classification);

    if(RenderTimeval(&alert->ts, timestamp, 256) == -1)
    {
        /* could not render the timeval */
        LogMessage("ERROR: OpAlertFast failed to render timeval\n");
        return -1;  
    }

    snprintf(sip, 16, "%u.%u.%u.%u", 
            (alert->sip >> 24) & 0xff,
            (alert->sip >> 16) & 0xff,
            (alert->sip >> 8)  & 0xff, 
             alert->sip        & 0xff);
    snprintf(dip, 16, "%u.%u.%u.%u", 
            (alert->dip >> 24) & 0xff,
            (alert->dip >> 16) & 0xff,
            (alert->dip >> 8)  & 0xff, 
             alert->dip        & 0xff);

    if(alert->protocol == IPPROTO_TCP ||
            alert->protocol == IPPROTO_UDP)
    {
        fprintf(stdout, "%s {%s} %s:%d -> %s:%d\n"
                "[**] [%d:%d:%d] %s [**]\n"
                "[Classification: %s] [Priority: %d]\n", timestamp, 
                protocol_names[alert->protocol], sip, alert->sp, 
                dip, alert->dp, alert->event.sig_generator, 
                alert->event.sig_id, alert->event.sig_rev,
                sid ? sid->msg : "ALERT", 
                class ? class->name : "Unknown", 
                alert->event.priority);
    }
    else
    {
        fprintf(stdout, "%s {%s} %s -> %s\n"
                "[**] [%d:%d:%d] %s [**]\n"
                "[Classification: %s] [Priority: %d]\n", timestamp, 
                protocol_names[alert->protocol], sip, dip, 
                alert->event.sig_generator, alert->event.sig_id, 
                alert->event.sig_rev, sid ? sid->msg : "ALERT", 
                class ? class->name : "Unknown", 
                alert->event.priority);
    }

    PrintXref(alert->event.sig_generator, alert->event.sig_id, stdout);

    fprintf(stdout, "-------------------------------------------"
            "-----------------------------\n");

    fflush(stdout);
    return 0;
}
