// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package auth

import (
	"database/sql"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/request"
	"github.com/drone/drone/mock"
	"github.com/sirupsen/logrus"

	"github.com/golang/mock/gomock"
)

func init() {
	logrus.SetOutput(ioutil.Discard)
}

func TestAuth(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{
		ID:      1,
		Login:   "octocat",
		Admin:   true,
		Machine: true,
		Hash:    "$2a$04$rR2VvGjM9iqAAoyLSE4IrexAlxDbIS3M5YKtj9ANs7vraki0ybYJq 197XXbZablx0RPQ8",
	}

	session := mock.NewMockSession(controller)
	session.EXPECT().Get(gomock.Any()).Return(mockUser, nil)

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/?access_token=VA.197XXbZablx0RPQ8", nil)

	HandleAuthentication(session)(
		http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			// use dummy status code to signal the next handler in
			// the middleware chain was properly invoked.
			w.WriteHeader(http.StatusTeapot)

			// verify the user was added to the request context
			if user, _ := request.UserFrom(r.Context()); user != mockUser {
				t.Errorf("Expect user in context")
			}
		}),
	).ServeHTTP(w, r)

	if got, want := w.Code, http.StatusTeapot; got != want {
		t.Errorf("Want status code %d, got %d", want, got)
	}
}

func TestAuth_Guest(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/", nil)

	session := mock.NewMockSession(controller)
	session.EXPECT().Get(gomock.Any()).Return(nil, sql.ErrNoRows)

	HandleAuthentication(session)(
		http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			// use dummy status code to signal the next handler in
			// the middleware chain was properly invoked.
			w.WriteHeader(http.StatusTeapot)

			// verify the user was added to the request context
			if _, ok := request.UserFrom(r.Context()); ok {
				t.Errorf("Expect guest mode, no user in context")
			}
		}),
	).ServeHTTP(w, r)

	if got, want := w.Code, http.StatusTeapot; got != want {
		t.Errorf("Want status code %d, got %d", want, got)
	}
}
