
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk.h 2661 2007-08-22 15:08:37Z mschwerin $
 *
 */

#ifndef HAVE_ODK_H
#define HAVE_ODK_H

#include <xine.h>

#include "meta_info.h"
#include "types.h"

/**
 * @todo This is supposed to work with all xine-lib version. We have to find
 *       a way of finding out if a distribution has disabled this.
 */
#define HAVE_IMAGE_PLAYBACK

#if defined(HAVE_IMAGEMAGICK) || defined(HAVE_GDK_PIXBUF)

/**
 * This controls the availability of images painted on the xine-lib OSD.
 */
#define HAVE_OSD_IMAGE
#endif

#if defined(HAVE_IMAGEMAGICK) && defined(HAVE_IMAGE_PLAYBACK)

/**
 * This controls the availability of image rotation.
 */
#define HAVE_IMAGE_ROTATION
#endif

#define AUDIO_CHANNEL_OFF       (-2)
#define AUDIO_CHANNEL_AUTO      (-1)

#define SPU_CHANNEL_OFF         (-2)
#define SPU_CHANNEL_AUTO        (-1)


/// Operation modes.
typedef enum {
    /// No playback
    ODK_MODE_NULL,
    /// Normal playback
    ODK_MODE_NORMAL,
    /// Logo/ background playback
    ODK_MODE_LOGO
} odk_mode_t;


/// Stream parameters
typedef enum {
    /// The position in the stream in percent.
    ODK_PARAM_POSITION,
    /// The current speed. See <code>odk_speed_t</code>.
    ODK_PARAM_SPEED = XINE_PARAM_SPEED,
    /// The audio offset (1/9 ms).
    ODK_PARAM_AUDIO_OFFSET = XINE_PARAM_AV_OFFSET,
    /// The current audio channel (-1 is auto, -2 is off).
    ODK_PARAM_AUDIO_CHANNEL = XINE_PARAM_AUDIO_CHANNEL_LOGICAL,
    /// The current subtitle offset (1/9 ms).
    ODK_PARAM_SPU_OFFSET = XINE_PARAM_SPU_OFFSET,
    /// The current subtitle channel (-1 is auto, -2 is off).
    ODK_PARAM_SPU_CHANNEL = XINE_PARAM_SPU_CHANNEL,
    /// The current audio volume in percent.
    ODK_PARAM_AO_VOLUME = XINE_PARAM_AUDIO_VOLUME,
    /// If current mute status (1 is muted, 0 is not muted).
    ODK_PARAM_AO_VOLUME_MUTE = XINE_PARAM_AUDIO_MUTE,
    /// The current deinterlace status (boolean).
    ODK_PARAM_VO_DEINTERLACE = XINE_PARAM_VO_DEINTERLACE,
    /// The current aspect ratio. See <code>odk_aspect_ratio_t</code>.
    ODK_PARAM_VO_ASPECT_RATIO = XINE_PARAM_VO_ASPECT_RATIO,
    /// The current video hue (0-65535).
    ODK_PARAM_VO_HUE = XINE_PARAM_VO_HUE,
    /// The current video saturation (0-65535).
    ODK_PARAM_VO_SATURATION = XINE_PARAM_VO_SATURATION,
    /// The current video contrast (0-65535).
    ODK_PARAM_VO_CONTRAST = XINE_PARAM_VO_CONTRAST,
    /// The current video brightness (0-65535).
    ODK_PARAM_VO_BRIGHTNESS = XINE_PARAM_VO_BRIGHTNESS,
    /// The x-component of the current video zoom (-85 - 400).
    ODK_PARAM_VO_ZOOM_X = XINE_PARAM_VO_ZOOM_X,
    /// The y-component of the current video zoom (-85 - 400).
    ODK_PARAM_VO_ZOOM_Y = XINE_PARAM_VO_ZOOM_Y,
    /// Gapless playback.
    ODK_PARAM_GAPLESS = XINE_PARAM_GAPLESS_SWITCH,
} odk_stream_param_t;


/// Stream information
typedef enum {
    /// Does the current stream have video.
    ODK_STREAM_INFO_HAS_VIDEO = XINE_STREAM_INFO_HAS_VIDEO,
    /// Does the current stream have audio.
    ODK_STREAM_INFO_HAS_AUDIO = XINE_STREAM_INFO_HAS_AUDIO,
    /// The number of audio channels of the current stream.
    ODK_STREAM_INFO_MAX_AUDIO_CHANNEL = XINE_STREAM_INFO_MAX_AUDIO_CHANNEL,
    /// The number of subtitle channels of the current stream.
    ODK_STREAM_INFO_MAX_SPU_CHANNEL = XINE_STREAM_INFO_MAX_SPU_CHANNEL,
    /// The number of the current DVD title.
    ODK_STREAM_INFO_DVD_TITLE_NUMBER = XINE_STREAM_INFO_DVD_TITLE_NUMBER,
    /// The number of titles of the current DVD.
    ODK_STREAM_INFO_DVD_TITLE_COUNT = XINE_STREAM_INFO_DVD_TITLE_COUNT,
    /// The number of the current DVD chapter.
    ODK_STREAM_INFO_DVD_CHAPTER_NUMBER = XINE_STREAM_INFO_DVD_CHAPTER_NUMBER,
    /// The number of chapters of the current DVD title.
    ODK_STREAM_INFO_DVD_CHAPTER_COUNT = XINE_STREAM_INFO_DVD_CHAPTER_COUNT
} odk_stream_info_t;


/// Available playback speeds.
typedef enum {
    /// Paused.
    ODK_SPEED_PAUSE = XINE_SPEED_PAUSE,
    /// Very slow.
    ODK_SPEED_SLOW_4 = XINE_SPEED_SLOW_4,
    /// Rather slow.
    ODK_SPEED_SLOW_2 = XINE_SPEED_SLOW_2,
    /// Normal speed.
    ODK_SPEED_NORMAL = XINE_SPEED_NORMAL,
    /// Rather fast.
    ODK_SPEED_FAST_2 = XINE_SPEED_FAST_2,
    /// Very fast.
    ODK_SPEED_FAST_4 = XINE_SPEED_FAST_4
} odk_speed_t;


/// Available video aspect ratios.
typedef enum {
    /// Automatic
    ODK_VO_ASPECT_RATIO_AUTO = XINE_VO_ASPECT_AUTO,
    /// Square
    ODK_VO_ASPECT_RATIO_SQUARE = XINE_VO_ASPECT_SQUARE,
    /// 4:3
    ODK_VO_ASPECT_RATIO_4_3 = XINE_VO_ASPECT_4_3,
    /// Anamorphic
    ODK_VO_ASPECT_RATIO_ANAMORPHIC = XINE_VO_ASPECT_ANAMORPHIC,
    /// DVB
    ODK_VO_ASPECT_RATIO_DVB = XINE_VO_ASPECT_DVB
} odk_aspect_ratio_t;


/// The available deinterlace settings.
typedef enum {
    /// Deinterlacing off.
    ODK_VO_DEINTERLACE_OFF = 0,
    /// Deinterlacing on.
    ODK_VO_DEINTERLACE_ON = 1,
} odk_deinterlace_t;


/// The available mute settings.
typedef enum {
    /// Mute off.
    ODK_AO_VOLUME_MUTE_OFF = 0,
    /// Mute on.
    ODK_AO_VOLUME_MUTE_ON = 1,
} odk_mute_t;


#define ODK_AUDIO_OFFSET_MIN    -90000
#define ODK_AUDIO_OFFSET_MAX    +90000
#define ODK_AUDIO_OFFSET_DEF    0
#define ODK_AUDIO_OFFSET_STEP   +9

#define ODK_SPU_OFFSET_MIN      -90000
#define ODK_SPU_OFFSET_MAX      +90000
#define ODK_SPU_OFFSET_DEF      0
#define ODK_SPU_OFFSET_STEP     +9

#define ODK_VO_ZOOM_MIN         0
#define ODK_VO_ZOOM_MAX         400
#define ODK_VO_ZOOM_DEF         100
#define ODK_VO_ZOOM_STEP        1

#define ODK_VO_HUE_MIN          0
#define ODK_VO_HUE_MAX          65535
#define ODK_VO_HUE_DEF          32896
#define ODK_VO_HUE_STEP         656

#define ODK_VO_SATURATION_MIN   0
#define ODK_VO_SATURATION_MAX   65535
#define ODK_VO_SATURATION_DEF   32896
#define ODK_VO_SATURATION_STEP  656

#define ODK_VO_CONTRAST_MIN     0
#define ODK_VO_CONTRAST_MAX     65535
#define ODK_VO_CONTRAST_DEF     32896
#define ODK_VO_CONTRAST_STEP    656

#define ODK_VO_BRIGHTNESS_MIN   0
#define ODK_VO_BRIGHTNESS_MAX   65535
#define ODK_VO_BRIGHTNESS_DEF   32896
#define ODK_VO_BRIGHTNESS_STEP  656


/// Public data struct for input devices.
typedef struct {
    pthread_t thread;
} odk_input_t;


/// Opaque odk data type
typedef struct odk_s odk_t;

/*
 * ***************************************************************************
 * Include OSD methods
 * ***************************************************************************
 */
#include "odk_osd.h"

/*
 * ***************************************************************************
 * Include event methods
 * ***************************************************************************
 */
#include "odk_event.h"

/**
 * Initialize the ODK structure. 
 *
 * @param xine                  A valid <code>xine_t</code> data structure.
 */
odk_t *odk_init (xine_t * xine);


/**
 * Frees the ODK data structure.
 *
 * @param odk                   The ODK data structure.
 */
void odk_free (odk_t * odk);


/**
 * Returns TRUE if the specified audio driver is available. The availability
 * of a driver depends on the version of xine-lib.
 *
 * @param audio_driver          The audio driver ID.
 */
bool odk_is_available_audio_driver (odk_t * odk, const char *audio_driver);


/**
 * Returns TRUE if the specified video driver is supported.
 *
 * @param video_driver          The video driver ID.
 */
bool odk_is_supported_video_driver (const char *video_driver);


/// Shows the output window.
void odk_show_window (odk_t * odk, bool fullscreen, bool show_border,
                      bool stay_on_top, int x, int y, int w, int h);


/// Hides the output window.
void odk_hide_window (odk_t * odk);


/// Sets the geometry of the output window.
void odk_window_set_geometry (odk_t * odk, int x, int y, int w, int h);


/// Sets the output window to fullscreen.
void odk_window_set_fullscreen (odk_t * odk, bool fullscreen);


/// Returns the current fullscreen status of the output window.
bool odk_window_is_fullscreen (odk_t * odk);


/**
 * Starts the playback of the specified MRL.
 *
 * @param odk                   The ODK data structure.
 * @param title                 The title of the new stream.
 * @param mrl                   The MRL of the new stream.
 * @param sub_mrl               The MRL of the new subtitle stream.
 * @param start_time            The time at which to start playback.
 * @param mode                  The mode in which to start playback.
 */
bool odk_play_stream (odk_t * odk, const char *title,
                      const char *mrl, const char *sub_mrl,
                      int start_time, odk_mode_t mode);


/**
 * Stops the playback of the current stream.
 *
 * @param odk                   The ODK data structure.
 */
bool odk_stop_stream (odk_t * odk);


/**
 * Starts to play a background stream.
 *
 * @param odk                   The ODK data structure.
 * @param mrl                   The MRL of the new stream.
 */
void odk_play_background_stream (odk_t * odk, const char *mrl);


/// Returns the specified stream parameter.
int odk_get_stream_param (odk_t * odk, odk_stream_param_t param);


/// Sets the specicied stream parameter.
int odk_set_stream_param (odk_t * odk, odk_stream_param_t param, int value);


/// Changes the specified stream parameter.
int odk_change_stream_param (odk_t * odk, odk_stream_param_t param,
                             int how, int min, int max);


/**
 * Returns a string representation of the specified stream parameter.
 *
 * @return                      A string representation of the specified
 *                              stream parameter. The returned string must be
 *                              freed after use.
 */
char *odk_get_stream_param_as_string (odk_t * odk, odk_stream_param_t param);


/**
 * Returns the name of the specified stream parameter.
 *
 * @return                      The name of the specified stream parameter.
 *                              The returned string must be freed after use.
 */
char *odk_get_stream_param_name (odk_t * odk, odk_stream_param_t param);


/**
 * Returns a string representation of the specified stream parameter.
 *
 * @return                      A string representation of the specified
 *                              stream parameter. The returned string must be
 *                              freed after use.
 */
char *odk_get_stream_param_value_as_string (odk_t * odk,
                                            odk_stream_param_t param,
                                            int param_value);


/// Returns the title of the current stream.
inline const char *odk_current_get_title (odk_t * odk);


/// Returns the MRL of the current stream.
inline const char *odk_current_get_mrl (odk_t * odk);


/// Returns TRUE if the current mode is LOGO mode.
inline bool odk_current_is_logo_mode (odk_t * odk);


/// Returns TRUE if the current mode is NORMAL mode.
inline bool odk_current_is_playback_mode (odk_t * odk);


/// Returns TRUE if the current stream is a DVD stream.
inline bool odk_current_is_dvd (odk_t * odk);


/// Returns TRUE if the current stream is a video CD stream.
inline bool odk_current_is_vcd (odk_t * odk);


/// Returns TRUE if the current stream is a V4L stream.
inline bool odk_current_is_v4l (odk_t * odk);


/// Returns TRUE if the current stream is a DVB stream.
inline bool odk_current_is_dvb (odk_t * odk);


/// Returns TRUE if the current stream is a VDR stream.
inline bool odk_current_is_vdr (odk_t * odk);


/// Returns TRUE if the current stream is a television stream.
inline bool odk_current_is_television (odk_t * odk);


/// Returns TRUE if the current stream is a audio CD stream.
inline bool odk_current_is_cdda (odk_t * odk);


/// Returns TRUE if the current stream is an image.
inline bool odk_current_is_image (odk_t * odk);


/// Returns TRUE if the current stream is audio only.
inline bool odk_current_is_audio (odk_t * odk);


/// Returns TRUE if the current stream has video.
inline bool odk_current_has_video (odk_t * odk);


/// Returns TRUE if the current stream has audio.
inline bool odk_current_has_audio (odk_t * odk);


/// Returns TRUE if the current stream has chapters.
inline bool odk_current_has_chapters (odk_t * odk);


/// Returns meta information of the specified type.
char *odk_get_meta_info (odk_t * odk, meta_info_t type);


/// Returns stream information of the specified type.
uint32_t odk_get_stream_info (odk_t * odk, odk_stream_info_t info);


/**
 * Returns the subtitle language of the current stream.
 *
 * @param odk                   The ODK data structure.
 * @param channel               The current subtitle channel.
 * @param lang                  A pointer to a character array that must be at
 *                              least XINE_LANG_MAX characters long.
 * @return                      On success TRUE is returned.
 */
inline bool odk_get_spu_lang (odk_t * odk, int channel, char *lang);


/**
 * Returns the audio language of the current stream.
 *
 * @param odk                   The ODK data structure.
 * @param channel               The current audio channel.
 * @param lang                  A pointer to a character array that must be at
 *                              least XINE_LANG_MAX characters long.
 * @return                      On success TRUE is returned.
 */
inline bool odk_get_audio_lang (odk_t * odk, int channel, char *lang);


/**
 * Get information on the current stream position and length. Depending of
 * the nature and system layer of the stream, some or all of this
 * information may be unavailable or incorrect (e.g. live network streams
 * may not have a valid length).
 *
 * @param odk                   The ODK data structure.
 * @param pos                   The current position in the stream (1-65535).
 * @param time                  The current position in the stream (ms).
 * @param length                The length of the stream (ms).
 * @return                      On success TRUE is returned.
 */
bool odk_get_pos_length (odk_t * odk, int *pos, int *time, int *length);


/**
 * Rotate the current image by the specified angle. If we're not showing an
 * image at the moment, the function does nothing.
 *
 * @param odk                   The ODK data structure.
 * @param angle                 The angle by which to rotate. Positive
 *                              angles rotate clockwise, negative angles
 *                              rotate counter-clockwise.
 */
#ifdef HAVE_IMAGE_ROTATION
void odk_rotate_current_image (odk_t * odk, int angle);
#endif


/**
 * Log the last error of a stream.
 *
 * @param file                  The file where the error occured.
 * @param line                  The line where the error occured.
 * @param stream                The stream that produced the error.
 * @param msg                   The error message.
 * @param mrl                   The MRL of the stream.
 */
void odk_log_stream_error (xine_stream_t * stream,
                           const char *file, int line,
                           const char *err_msg, const char *mrl);


#endif /* HAVE_ODK_H */
