
/*
 * Copyright (C) 2007 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: aex.c 2402 2007-07-02 15:22:05Z mschwerin $
 *
 */

#include "config.h"

#ifdef HAVE_AEX

#include <errno.h>
#include <string.h>

#include <avahi-client/client.h>
#include <avahi-client/lookup.h>

#include <avahi-common/simple-watch.h>
#include <avahi-common/malloc.h>
#include <avahi-common/error.h>

#include "aex.h"
#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "mutex.h"
#include "oxine.h"
#include "odk.h"
#include "utils.h"

#define SERVICE_TYPE            "_raop._tcp."

extern oxine_t *oxine;

static AvahiSimplePoll *simple_poll = NULL;

static l_list_t *aex_list = NULL;                               /* The linked list containing the entries. */
static pthread_t thread;
static pthread_mutex_t mutex;

static void aex_free (aex_t * aex);

static void
resolve_callback (AvahiServiceResolver * r,
                  AVAHI_GCC_UNUSED AvahiIfIndex interface,
                  AVAHI_GCC_UNUSED AvahiProtocol protocol,
                  AvahiResolverEvent event, const char *name,
                  const char *type, const char *domain, const char *host_name,
                  const AvahiAddress * address, uint16_t port,
                  AvahiStringList * txt, AvahiLookupResultFlags flags,
                  AVAHI_GCC_UNUSED void *userdata)
{
    bool repaint = false;
    assert (r);

    mutex_lock (&mutex);

    /* Called whenever a service has been resolved successfully or the request
     * has timed out. */
    switch (event) {
    case AVAHI_RESOLVER_FAILURE:
        {
            AvahiClient *c = avahi_service_resolver_get_client (r);
            int e = avahi_client_errno (c);
            error (_("Could not resolve service '%s' "
                     "of type '%s' in domain '%s': %s"),
                   name, type, domain, avahi_strerror (e));
        }
        break;
    case AVAHI_RESOLVER_FOUND:
        {
            char a[AVAHI_ADDRESS_STR_MAX];
            char *title = index (name, '@');
            if (title) {
                title += 1;
            }
            avahi_address_snprint (a, sizeof (a), address);

            aex_t *aex = ho_new (aex_t);
            aex->name = ho_strdup (name);
            aex->title = title ? ho_strdup (title) : ho_strdup (name);
            aex->address = ho_strdup (a);
            aex->port = port;
            l_list_append (aex_list, aex);

            debug ("Adding airport express '%s' (%s:%d).",
                   aex->title, aex->address, aex->port);

            repaint = true;
        }
    }

    avahi_service_resolver_free (r);

    mutex_unlock (&mutex);

    if (repaint) {
        oxine_event_t ev;
        ev.type = OXINE_EVENT_GUI_REPAINT;
        odk_oxine_event_send (oxine->odk, &ev);
    }
}


static void
browse_callback (AvahiServiceBrowser * b, AvahiIfIndex interface,
                 AvahiProtocol protocol, AvahiBrowserEvent event,
                 const char *name, const char *type, const char *domain,
                 AVAHI_GCC_UNUSED AvahiLookupResultFlags flags,
                 void *userdata)
{
    AvahiClient *c = userdata;
    assert (b);
    bool repaint = false;

    mutex_lock (&mutex);

    /* Called whenever a new services becomes available on the LAN or is
     * removed from the LAN. */
    switch (event) {
    case AVAHI_BROWSER_FAILURE:
        {
            AvahiClient *c = avahi_service_browser_get_client (b);
            int e = avahi_client_errno (c);
            error (_("Browser failure: %s!"), avahi_strerror (e));
            avahi_simple_poll_quit (simple_poll);
        }
        break;
    case AVAHI_BROWSER_NEW:
        debug ("Adding service '%s' of type '%s' in domain '%s'.",
               name, type, domain);

        /* We ignore the returned resolver object. In the callback function we
         * free it. If the server is terminated before the callback function
         * is called the server will free the resolver for us. */
        if (!(avahi_service_resolver_new (c, interface, protocol, name, type,
                                          domain, AVAHI_PROTO_UNSPEC, 0,
                                          resolve_callback, c))) {
            error (_("Could not resolve '%s': %s!"), name,
                   avahi_strerror (avahi_client_errno (c)));
        }
        break;
    case AVAHI_BROWSER_REMOVE:
        {
            debug ("Removing service '%s' of type '%s' in domain '%s'.",
                   name, type, domain);

            aex_t *aex = aex_first ();
            while (aex) {
                if (strcmp (aex->name, name) == 0) {
                    debug ("Removing airport express '%s' (%s).",
                           aex->title, aex->address);

                    l_list_remove (aex_list, aex);
                    aex_free (aex);

                    repaint = true;

                    break;
                }
                aex = aex_next (aex);
            }
        }
        break;
    default:
        break;
    }

    mutex_unlock (&mutex);

    if (repaint) {
        oxine_event_t ev;
        ev.type = OXINE_EVENT_GUI_REPAINT;
        odk_oxine_event_send (oxine->odk, &ev);
    }
}


static void
client_callback (AvahiClient * c, AvahiClientState state,
                 AVAHI_GCC_UNUSED void *userdata)
{
    assert (c);

    /* Called whenever the client or server state changes */
    if (state == AVAHI_CLIENT_FAILURE) {
        error (_("Server connection failure: %s"),
               avahi_strerror (avahi_client_errno (c)));
        avahi_simple_poll_quit (simple_poll);
    }
}


static void *
aex_monitor_thread (void *data)
{
    AvahiClient *client = NULL;
    AvahiServiceBrowser *sb = NULL;
    int error;

    info (_("Successfully started AEX monitor thread."));
    debug ("AEX monitor thread: 0x%X", (int) pthread_self ());

    /* Allocate main loop object */
    if (!(simple_poll = avahi_simple_poll_new ())) {
        error (_("Failed to create simple poll object!"));
        goto fail;
    }

    /* Allocate a new client */
    if (!(client = avahi_client_new (avahi_simple_poll_get (simple_poll),
                                     0, client_callback, NULL, &error))) {
        error (_("Failed to create client: %s!"), avahi_strerror (error));
        goto fail;
    }

    /* Create the service browser */
    if (!(sb = avahi_service_browser_new (client,
                                          AVAHI_IF_UNSPEC,
                                          AVAHI_PROTO_UNSPEC,
                                          SERVICE_TYPE,
                                          NULL, 0, browse_callback,
                                          client))) {
        error (("Failed to create service browser: %s!"),
               avahi_strerror (avahi_client_errno (client)));
        goto fail;
    }

    /* Run the main loop */
    avahi_simple_poll_loop (simple_poll);

  fail:
    /* Cleanup things */
    if (sb) {
        avahi_service_browser_free (sb);
    }
    if (client) {
        avahi_client_free (client);
    }
    if (simple_poll) {
        avahi_simple_poll_free (simple_poll);
    }

    pthread_exit (NULL);
    return NULL;
}


bool
aex_monitor_start (void)
{
    aex_list = l_list_new ();

    pthread_mutex_init (&mutex, NULL);

    if (pthread_create (&thread, NULL, aex_monitor_thread, NULL) != 0) {
        error (_("Could not create AEX monitor thread: %s!"),
               strerror (errno));
        return false;
    }

    return true;
}


static void
aex_free (aex_t * aex)
{
    ho_free (aex->name);
    ho_free (aex->title);
    ho_free (aex->address);
    ho_free (aex);
}


static void
aex_free_cb (void *p)
{
    aex_free ((aex_t *) p);
}


bool
aex_monitor_free (void)
{
    void *ret = NULL;

    if (simple_poll) {
        avahi_simple_poll_quit (simple_poll);
    }
    pthread_join (thread, &ret);

    mutex_lock (&mutex);
    l_list_free (aex_list, aex_free_cb);
    aex_list = NULL;
    mutex_unlock (&mutex);

    pthread_mutex_destroy (&mutex);

    info (_("Successfully stopped AEX monitor thread."));

    return true;
}


aex_t *
aex_first (void)
{
    return (aex_t *) l_list_first (aex_list);
}


aex_t *
aex_next (aex_t * aex)
{
    return (aex_t *) l_list_next (aex_list, aex);
}


void
aex_monitor_lock (void)
{
    mutex_lock (&mutex);
}


void
aex_monitor_unlock (void)
{
    mutex_unlock (&mutex);
}


#endif /* HAVE_AEX */
