//////////////////////////////////////////////////////////////////////////////
//
// 		  Copyright (C) 1996,1997  Matthew Doar  doar@pobox.com
// 
// Permission to use, copy, and distribute this software and its documentation 
// for any purpose with or without fee is hereby granted, provided that the 
// above copyright notice appears with all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// 
// Permission to modify the software is granted, but not the right to 
// distribute the modified code. Modifications are to be distributed as 
// patches to the released version. 
// 
// This software is provided "as is" without express or implied warranty. 
//
//////////////////////////////////////////////////////////////////////////////

// tiers_red_intra.cc

#include <iostream.h>
#include <limits.h>	// for ULONG_MAX

#ifndef _TIERS_HH
#include "tiers.hh"
#endif

#define CASE_WAN 0
#define CASE_MAN 1
#define CASE_LAN 2

extern "C" 
{
  long random(void);
}


////////////////////////////////////////////////////////////////////////
// Model::CreateWanRedundancy
//
// Simplest approach is to add extra edges from each node to its closest 
// neighbours
////////////////////////////////////////////////////////////////////////
bool Model::CreateWanRedundancy()
{
  return(CreateRedundancy(WanList, CASE_WAN));
}



////////////////////////////////////////////////////////////////////////
// Model::CreateManRedundancy
//
// Simplest approach is to add extra edges from each node to its closest 
// neighbours
// Currently identical to CreateWanRedundancy. 
////////////////////////////////////////////////////////////////////////
bool Model::CreateManRedundancy()
{
  return(CreateRedundancy(ManList, CASE_MAN));
}



////////////////////////////////////////////////////////////////////////
// Model::CreateRedundancy
//
// Create redundancy in a WAN or MAN
////////////////////////////////////////////////////////////////////////
bool Model::CreateRedundancy(NodesAndEdges *NetworkList, int network_type)
{
  bool ret = true;
  unsigned long int specific;
  
  Node *nodes = 0;
  Edge *edges = 0;

  for (unsigned long int network = 0; 
	   network < 1 /* NW should equal 1 */; 
	   network++)
	{
	  const unsigned long int NumNodes = NetworkList[network].NumNodes();	

	  // This test added for bug noted in CHANGES 1/2/497
	  if (NumNodes == 0)
		{
		  continue;
		}

	  const unsigned long int MaxNumEdges = NetworkList[network].MaxNumEdges();	

	  nodes = NetworkList[network].nodes; // an alias
	  edges = NetworkList[network].newedges; // an alias

	  // Examine node j. If it hasn't enough edges connected to it, add some
	  // Randomise the starting point chosen for j - should be totally random
	  unsigned long int j = random() % NumNodes;
	  for (unsigned long int plain_index = 0;
		   plain_index < NumNodes;
		   plain_index++)		    
		{
		  j = (j+plain_index) % NumNodes;

		  // NumEdges count includes both ACTIVE and REDUNDANT edges
		  const unsigned long int NumEdges = NetworkList[network].NumEdges(j);	
		  switch (network_type) {
		  case CASE_WAN:
		    specific = RWdist(RW);
		    break;
		  case CASE_MAN:
		    specific = RMdist(RM);
		    break;
		  default:
		    specific = RWdist(RW);
		  }
		  
		  unsigned long int NumActiveEdges = 0;
		  // Calculate the number of ACTIVE edges
		  // REDUNDANT edges are edges which have been already added as active
		  for (unsigned long int k = 0; k < NumEdges; k++)
			{
			  if ((edges[j*MaxNumEdges+k].state == Model::ACTIVE) ||
				  (edges[j*MaxNumEdges+k].state == Model::REDUNDANT))
				{
				  NumActiveEdges++;
				}
			}

		  // If there are too few active edges, find some and add them
		  while (NumActiveEdges < specific)
			{
			  unsigned long int EdgeClosestTo = NumEdges;
			  unsigned long int MinimumCost = ULONG_MAX;

			  // find the least costly edge not already in use
			  for (unsigned long int k = 0; k < NumEdges; k++)
				{
				  // This explicitly uses delay for the metric
				  unsigned long int index = j*MaxNumEdges+k;
				  unsigned long int cost = edges[index].delay;
				  if ((cost < MinimumCost) && 
					  (edges[index].state == INACTIVE))
					{
					  MinimumCost = cost;
					  EdgeClosestTo = k;
					}
				} // k

			  // Mark the edge number EdgeClosestTo active but redundant
			  if (EdgeClosestTo != NumEdges)
				{
				  const unsigned long int index = j*MaxNumEdges+EdgeClosestTo;
				  edges[index].state = REDUNDANT;

				  // Activate the edge from the 'to' node also
				  // We know the start and end node numbers only
				  // so we have to do a linear search here, O(2*NumNodes)
				  unsigned long int row = 0;
				  while ((edges[row*MaxNumEdges+0].start != edges[index].end) &&
						 (row != NumNodes))
					{
					  row++;
					}

				  unsigned long int col = 0;
				  if (row != NumNodes)
					{
					  // Found the row, now find the index of the edge (column)
					  while ((edges[row*MaxNumEdges+col].end != edges[index].start) 
							 && (col != MaxNumEdges))
						{
						  col++;
						}
					}

				  if ((row != NumNodes) && (col != MaxNumEdges))
					{
					  // The correct edge was found so mark it active but 
					  // redundant
					  edges[row*MaxNumEdges+col].state = REDUNDANT;
					}
				  else
					{
					  cerr << "CreateRedundancy:: symmetrical edge not found" << endl;
					  return false;
					}

				  // And increase the number of active edges
				  NumActiveEdges++;
				}
			  else
				{
				  // All edges are already  in use
				  // To break the while loop
				  NumActiveEdges = specific;
				}

			} // while

		} // plain_index/j
	} // network

  return ret;
}




////////////////////////////////////////////////////////////////////////
// Model::CreateLanRedundancy
//
// LAN redundancy is not supported in this model for simplicity
////////////////////////////////////////////////////////////////////////
bool Model::CreateLanRedundancy()
{
  bool ret = true;

  if (RL != 1)
	{
	  cerr << "CreateLanRedundancy:: LAN redundancy not supported" << endl;
	  return false;
	}

  return ret;
}

// end of file

