// Tested with JeeLink v3 (2012-02-11)
// Supported devices: see FHEM wiki
// info    : http://forum.jeelabs.net/node/110
//           http://fredboboss.free.fr/tx29/tx29_sw.php
//           http://www.f6fbb.org/domo/sensors/
//           http://www.mikrocontroller.net/topic/67273 
//           benedikt.k org rinie,marf,joop 1 nov 2011, slightly modified by Rufik (r.markiewicz@gmail.com)
// Changelog: 2015-11-13: initial release 0.1

#define RESYNC_THRESHOLD 50       // max. number of lost packets from a station before a full rediscovery
#define LATE_PACKET_THRESH 5000   // packet is considered missing after this many micros

#define PROGNAME        "DAVIS"
#define PROGVERS        "0.2" 

#include "RFMxx.h"
#include "JeeLink.h"
#include "Help.h"
#include "TimerOne.h"
#include "PacketFifo.h"
#include "InternalSensors.h"

// KeyValueProtocol with full format keys
//#define KVP_LONG_KEY_FORMAT 1
#include "KVPSensors.h"

// defaults
#define SERIAL_BAUD   57600

#define RECEIVER_ENABLED       0                     // Set to 0 if you don't want to receive 

// The following settings can also be set from FHEM
#define DEBUG_DISABLED        0x0
#define DEBUG_MODE            0x1
#define DEBUG_PACKET_RAW      0x2
#define DEBUG_UNDECODED_RAW   0x4
byte DEBUG                   = DEBUG_DISABLED;        // bit mask 1=debug, 2=packet raw data, 3=undecoded packet raw data

byte DETECTION_MODE          = 0x0;
PacketFifo fifo;
volatile uint32_t packets, lostPackets, numResyncs;
volatile byte stationsFound = 0;
volatile byte curStation = 0;

// define devices
#define NUM_STATIONS_MAX 8
byte NUM_STATIONS = 0;

// id, type, active
Station stations[NUM_STATIONS_MAX] = {
    { 0, STYPE_OFF,         false },
    { 0, STYPE_OFF,         false },
    { 0, STYPE_OFF,         false },
    { 0, STYPE_OFF,         false },
    { 0, STYPE_OFF,         false },
    { 0, STYPE_OFF,         false },
    { 0, STYPE_OFF,         false },
    { 0, STYPE_OFF,         false },
};

// init
RFMxx rfm1;

JeeLink jeeLink;
InternalSensors internalSensors;

void setup() {
  Serial.begin(SERIAL_BAUD);
#ifdef _DEBUG
  DEBUG = DEBUG & DEBUG_MODE;
#endif

  rfm1.InitialzeVantage(FREQ_BAND_EU);

  initStations();

  if (RECEIVER_ENABLED)
    rfm1.EnableReceiver(true);
  
  // periodical interrupts every ms for missed packet detection
  Timer1.initialize(1000);
  Timer1.attachInterrupt(handleTimerInt, 0);
  rfm1.setUserInterrupt(handleRadioInt);

  internalSensors.TryInitializeBMP180();

  // version info
  handleCommandV();
  Serial.println();

  sendDictionary();
}

void loop() {
  // process input
  if (Serial.available()) {
    handleSerialPort(Serial.read());
  }

  // Periodically send own sensor data
  internalSensors.TryHandleData();

  if (fifo.hasElements()) {
    decode_packet(fifo.dequeue());
  }
}

void handleCommandV() {
  Serial.print("[");
  Serial.print(PROGNAME);
  Serial.print('.');
  Serial.print(PROGVERS);

  Serial.print(" (");
  Serial.print(rfm1.GetRadioName());
  Serial.print(" b:");
  Serial.print(rfm1.GetBand());
  Serial.print(")");

  if (internalSensors.HasBMP180())
    Serial.print(" + BMP180");

  Serial.print("] ");
}

void handleSerialPort(char c) {
  static unsigned long value, value2;
  bool hasValue, hasValue2;
  char r = c;

  // reset variables
  value = 0; hasValue = false;
  value2 = 0; hasValue2 = false;
  
  // char is a number
  if (r >= '0' && r <= '9') {
    byte delays = 2;
    while ((r >= '0' && r <= '9') || r == ',') {
      // check value separator
      if (r == ',') {
        if (!hasValue || hasValue2) {
          print_warning(2, "format");
          return;
        }
        
        hasValue2 = true;
      } else {
        if (!hasValue || !hasValue2) {
          value = value * 10 + (r - '0');
          hasValue = true;
        } else {
          value2 = value2 * 10 + (r - '0');
          hasValue2 = true;
        }
      }
      
      // wait a little bit for more input
      while (Serial.available() == 0 && delays > 0) {
        delay(20);
        delays--;
      }

      // more input available
      if (delays == 0 && Serial.available() == 0) {
        return;
      }

      r = Serial.read();
    }
  }
  
  switch (r) {
    case 'b':
      // band
      if (!hasValue)
        print_warning(1, "b");
      else
        if (rfm1.GetBand() != value) {
          rfm1.EnableReceiver(false);
          initStations();
          rfm1.SetBand(value);
        }
      break;
    case 'd':
      // debug
      setDebugMode(hasValue && (value == 1));
      break;
    case 'D':
      if (hasValue && value != 0 && DETECTION_MODE == 0) {
        if (rfm1._mode != RF69_MODE_RX) {
          rfm1.SetChannel(0);
          rfm1.EnableReceiver(true);
          DETECTION_MODE = 1;
        }
      } else if ((!hasValue || value == 0) && DETECTION_MODE == 1) {
          rfm1.EnableReceiver(false);
          DETECTION_MODE = 0;
      }
      break;
    case 'h':
      // height
      if (!hasValue)
        print_warning(1, "h");
      else
        internalSensors.SetAltitudeAboveSeaLevel(value);
      break;
    case 'l':
      jeeLink.EnableLED(hasValue && (value == 1));
      break;
    case 'p':
      // use bit 2 and 3 for raw packet output
      DEBUG = (DEBUG & 0xF9) + (value > 3 ? 0 : (value < 1 ? 0 : value << 1));
      break;
    // station add
    case 's':
      if (!hasValue)
        listStations();
      else if (!hasValue2)
        print_warning(1, "s");
      else
        addStation(value, value2);
      break;
    case 'S':
      internalSensors.SetInterval((hasValue ? value : 60));
      break;
    case 'r':
      rfm1.EnableReceiver(!(hasValue && (value == 0)));
      if (hasValue && (value == 0))
        resetStations();
      break;
    case 'v':
      // Version info
      handleCommandV();
      print_config();
      break;
    case ' ':
    case '\n':
    case '\r':
      break;
    default:
      handleCommandV();
#ifndef NOHELP
      Help::Show();
#endif
      break;
    }
}

void handleReceivedData(RFMxx *rfm) {
  rfm->EnableReceiver(false);

  unsigned long lastRx = micros();

  byte payload[PAYLOADSIZE];
  rfm->GetPayload(payload);

  unsigned int rxCrc = word(payload[6], payload[7]);          // received CRC
  unsigned int calcCrc = rfm->GetCRC16CCITT(&payload[0], 6);  // calculated CRC

  // ignore packet
  if (!(rxCrc == calcCrc && rxCrc != 0)) {
    rfm->EnableReceiver(true);

    // output raw payload to serial
    if (DEBUG & DEBUG_UNDECODED_RAW) {
      Serial.print(DiagnosticHeader(PROGNAME));
      packetToHex(payload, 10);
      Serial.println();
    }
      
    return;
  }

  byte id = payload[0] & 7;
  int stIx = findStation(id);

  if (DETECTION_MODE == 1) {
    Serial.print(DiscoveryHeader(PROGNAME, id));
    Serial.println(SensorDataValue(RSSI, -rfm->RSSI));
  }
  
  // if we have no station cofigured for this id (at all; can still be be !active), ignore the packet
  if (stIx < 0) {
    rfm->EnableReceiver(true);
    return;
  }
   
  // Phase 1: station discovery
  // stay on a fixed channel and store last rx timestamp of discovered stations in station array
  // interval is used as 'station seen' flag
  if (stationsFound < NUM_STATIONS && stations[stIx].interval == 0) {

    stations[stIx].channel = nextChannel(rfm->GetChannel());
    stations[stIx].lastRx = lastRx;
    stations[stIx].lostPackets = 0;
    stations[stIx].interval = (41 + id) * 1000000 / 16; // Davis' official tx interval in us
    stationsFound++;
    nextStation(); // skip to next station expected to tx

    // reset current radio channel in Phase 1 or start Phase 2
    if (stationsFound < NUM_STATIONS) {
      rfm->SetChannel(rfm->GetChannel());
    } else {
      rfm->SetChannel(stations[curStation].channel);
    }
    
    rfm->EnableReceiver(true);
    return;

  } else {
    // Phase 2: normal reception
    // 8 received packet bytes including received CRC, the channel and RSSI go to the fifo
    if (stationsFound == NUM_STATIONS && stations[curStation].active) {
      packets++;
      fifo.queue((byte*)payload, rfm->GetChannel(), -rfm->RSSI, rfm->FEI, lastRx - stations[curStation].lastRx);
    }

    // successful reception - skip to next/earliest expected station
    stations[curStation].lostPackets = 0;
    stations[curStation].lastRx = lastRx;
    stations[curStation].channel = nextChannel(rfm->GetChannel());
    nextStation();
    if (stationsFound < NUM_STATIONS) { // Phase 1/2 check as usual
      rfm->SetChannel(rfm->GetChannel());
    } else {
      rfm->SetChannel(stations[curStation].channel);
    }
  }
    
  rfm->EnableReceiver(true);
}

// Calculate the next hop of the specified channel
byte nextChannel(byte channel) {
  return ++channel % rfm1.GetBandFrequencies();
}

void addStation(int id, int stype) {
  int stIdx = findStation(id);
  
  // check station id
  if (stIdx >= 0) {
    // only disable station
    if (stype == STYPE_OFF)
      removeStation(stIdx);
    else
      print_warning(3, "id used");

    return;
  }

  // add station
  if (NUM_STATIONS < NUM_STATIONS_MAX) {
    setStation(NUM_STATIONS, stype, id, true);
    NUM_STATIONS += 1;
  } else {
    print_warning(3, "max. exceeded");
  }
}

void removeStation(byte idx) {
  bool isModeRx = (rfm1._mode == RF69_MODE_RX);

  // stop receiver
  if (isModeRx)
    rfm1.EnableReceiver(false);

  // modify global station counter
  NUM_STATIONS--;
  if (stations[idx].interval > 0)
    stationsFound--;

  // move additional stations
  for (int i=idx; i < NUM_STATIONS; i++)
    stations[i] = stations[i + 1];

  // reset last station
  setStation(NUM_STATIONS, STYPE_OFF, 0, false);
  initStation(NUM_STATIONS);

  // restart receiver
  if (isModeRx)
    rfm1.EnableReceiver(true);
}

void setStation(byte idx, byte type, byte id, bool active) {
  stations[idx].type = type;
  stations[idx].id = id;
  stations[idx].active = active;
}

// Find the station index in stations[] for station expected to tx the earliest and update curStation
void nextStation() {
  unsigned long earliest = 0xffffffff;
  unsigned long now = micros();
  for (int i = 0; i < NUM_STATIONS; i++) {
    unsigned long current = stations[i].lastRx + stations[i].interval - now;
    if (stations[i].active && stations[i].interval > 0 && current < earliest) {
      earliest = current;
      curStation = i;
    }
  }
}

// Find station index in stations[] for a station ID (-1 if doesn't exist)
int findStation(byte id) {
  for (byte i = 0; i < NUM_STATIONS; i++) {
    if (stations[i].id == id)
      return i;
  }
  return -1;
}

// Reset station array to safe defaults
void initStations() {
  for (byte i = 0; i < NUM_STATIONS_MAX; i++)
    initStation(i);
}

void initStation(byte idx) {
    stations[idx].channel = 0;
    stations[idx].lastRx = 0;
    stations[idx].interval = 0;
    stations[idx].lostPackets = 0;
}

void resetStations() {
  stationsFound = 0;
  initStations();
  packets = 0;
  lostPackets = 0;
}

void listStations() {
  handleCommandV();

  Serial.print("stations: ");
  Serial.println(NUM_STATIONS);
  for (int i=0; i < NUM_STATIONS; i++) {
    Serial.print("#");
    Serial.print(i);
    Serial.print(": ");
    Serial.print(stations[i].id);
    Serial.print(",");
    Serial.print(stations[i].type);
    Serial.println((stations[i].active ? " a" : " i"));
  }
}

// Handle missed packets. Called from Timer1 ISR every ms
void handleTimerInt() {

  // don't handle stations via timer event
  if (DETECTION_MODE)
    return;

  unsigned long lastRx = micros();
  bool readjust = false;

  // find and adjust 'last seen' rx time on all stations with older reception timestamp than their period + threshold
  // that is, find missed packets
  for (byte i = 0; i < NUM_STATIONS; i++) {
    if (stations[i].interval > 0 && (lastRx - stations[i].lastRx) > stations[i].interval + LATE_PACKET_THRESH) {
      if (stationsFound == NUM_STATIONS && stations[curStation].active)
        lostPackets++;
      stations[i].lostPackets++;

      if (stations[i].lostPackets > RESYNC_THRESHOLD) {
        numResyncs++;
        resetStations();
        return;
      } else {
        stations[i].lastRx += stations[i].interval; // when packet should have been received
        stations[i].channel = nextChannel(stations[i].channel); // skip station's next channel in hop seq
        readjust = true;
      }
    }
  }

  // find/set earliest expected rx channel or in Phase 1, reset radio on current channel
  if (readjust) {
    nextStation();
    if (stationsFound < NUM_STATIONS) {
      rfm1.SetChannel(rfm1.GetChannel());
    } else {
      rfm1.SetChannel(stations[curStation].channel);
    }
  }
}

void handleRadioInt() {
  if (rfm1.PayloadIsReady())
    handleReceivedData(&rfm1);

  jeeLink.Blink(1);
}

void decode_packet(RadioData* rd) {
  int val;
  byte* packet = rd->packet;

  byte id = packet[0] & 7;
  int stIx = findStation(id);

  Serial.print(SensorDataHeader(PROGNAME, id));

  Serial.print(SensorDataValue(Channel, rd->channel));
  Serial.print(SensorDataValue(RSSI, -rd->rssi));
  Serial.print(SensorDataValue(Battery, (char*)(packet[0] & 0x8 ? "err" : "ok")));

  // All packet payload values are printed unconditionally, either properly
  // calculated or flagged with a special "missing sensor" value, mostly -1.
  // It's the CPE's responsibility to interpret our output accordingly.

  // wind data is present in every packet, windd == 0 (packet[2] == 0) means there's no anemometer
  if (packet[2] != 0) {
    if (stations[stIx].type == STYPE_VUE) {
      val = (packet[2] << 1) | (packet[4] & 2) >> 1;
      val = round(val * 360 / 512);
    } else {
      val = 9 + 342 * packet[2] / 255;
    }
  } else {
    val = 0;
  }

  // 8 bit + Umrechnung kmh * 1.609
  Serial.print(SensorDataValue(WindSpeed, (float)packet[1] * 1.609));
  Serial.print(SensorDataValue(WindDirection, val));

  switch (packet[0] >> 4) {

    case VP2P_UV:
      if (packet[3] != 0xff) {
        val = ((packet[3] << 8 | packet[4]) >> 6) - 1;
        Serial.print(SensorDataValue(UV, val));
      }
      break;

    case VP2P_SOLAR:
      if (packet[3] != 0xff) {
        val = ((packet[3] << 8 | packet[4]) >> 6) - 1;
        Serial.print(SensorDataValue(Solar, (float)(val / 0.5675)));
      }
      break;

    // 7 bit rain tip count
    case VP2P_RAIN:
      if (packet[3] != 0x80)
        Serial.print(SensorDataValue(RainTipCount, packet[3]));
      break;

    case VP2P_RAINSECS:
      if (packet[3] != 0xff) {
        if (packet[4] & 0x40) { // strong rain flag
          // strong rain: byte4[5:4] as value[5:4]
          // strong rain: byte3[3:0) as value[3:0]
          // 6 bits total
          val = ((packet[4] & 0x30) | (packet[3] >> 4));
        } else { // light rain
          // light rain: byte4[5:4] as value[9:8] 
          // light rain: byte3[7:0) as value[7:0]
          // 10 bits total
          val = (packet[4] & 0x30) << 4 | packet[3];
        }
      } else
        val = -1;
      Serial.print(SensorDataValue(RainSecs, val));
      break;

    case VP2P_TEMP:
      if (packet[3] != 0xff) {
        val = (int)packet[3] << 4 | packet[4] >> 4;
        // °F (val / 10.0) + Umrechnung in °C
        Serial.print(SensorDataValue(Temperature, (float)(val / 10.0 - 32.0) * 5.0 / 9.0));
      }
      break;

    case VP2P_HUMIDITY:
      val = ((packet[4] >> 4) << 8 | packet[3]) / 10; // 0 -> no sensor
      if (val > 0)
        Serial.print(SensorDataValue(Humidity, (float)val));
      break;

    case VP2P_WINDGUST:
      // 8 bit + Umrechnung kmh * 1.609
      Serial.print(SensorDataValue(WindGust, (float)packet[3] * 1.609));
      Serial.print(SensorDataValue(WindGustRef, (packet[3] != 0 ? (packet[5] & 0xf0 >> 4) : -1)));
      break;

    case VP2P_SOIL_LEAF:
      // no public documentation of the packet type yet
      Serial.print(SensorDataValue(SoilLeaf, -1));
      break;

    case VUEP_VCAP:
      val = (packet[3] << 2) | (packet[4] & 0xc0) >> 6;
      Serial.print(SensorDataValue(VoltageCapacity, (float)(val / 100.0)));
      break;

    case VUEP_VSOLAR:
      val = (packet[3] << 2) | (packet[4] & 0xc0) >> 6;
      Serial.print(SensorDataValue(VoltageSolar, val));
  }

//  print_value("fei", round(rd->fei * RF69_FSTEP / 1000));
//  print_value("delta", rd->delta);

  if (DEBUG & DEBUG_PACKET_RAW)
    packetToHex(packet, 10);

  Serial.println();
}

void packetToHex(volatile byte* packet, byte len) {
  char hs[40];
  char* hex = "0123456789abcdef";

  hs[len * 3 - 1] = 0;
  int x = 0;
  for (byte i = 0; i < len; i++) {
    hs[x++] = hex[packet[i] >> 4];
    hs[x++] = hex[packet[i] & 0x0f];
    if (i < len - 1) hs[x++] = '-';
  }

  Serial.print(SensorDataValue(PacketDump, hs));
}

void setDebugMode(bool enable) {
  DEBUG = (DEBUG & 0xFE) + (enable ? DEBUG_MODE : DEBUG_DISABLED);
  rfm1.SetDebugMode(enable);
}

// helper
void print_config() {
  String blank = F(" ");
  
  Serial.print(F("config:"));
  if (DEBUG & DEBUG_MODE)  
    Serial.print(" 1d");
  if (DEBUG & (DEBUG_PACKET_RAW | DEBUG_UNDECODED_RAW)) {
    Serial.print(blank);
    Serial.print((DEBUG & (DEBUG_PACKET_RAW | DEBUG_UNDECODED_RAW)) >> 1);
    Serial.print("p");
  }
  Serial.print(blank);
  Serial.print(rfm1.GetBand());
  Serial.print("b");
  if (internalSensors.GetAltitudeAboveSeaLevel() != 0) {
    Serial.print(blank);
    Serial.print(internalSensors.GetAltitudeAboveSeaLevel());
    Serial.print("h");
  }
  if (internalSensors.GetInterval() != 60) {
    Serial.print(blank);
    Serial.print(internalSensors.GetInterval());
    Serial.print("S");
  }
  for (int i=0; i<NUM_STATIONS; i++) {
    Serial.print(blank);
    Serial.print(stations[i].id);
    Serial.print(",");
    Serial.print(stations[i].type);
    Serial.print("s");
  }
  if (jeeLink.isLEDEnabled())
    Serial.print(" 1l");
  if (rfm1._mode != RF69_MODE_RX)
    Serial.print(" 0r");
  if (DETECTION_MODE == 1)
    Serial.print(" D");

  Serial.println();
}

void print_warning(byte type, char *msg) {
  return;
  Serial.print(F("\nwarning: "));
  if (type == 1)
    Serial.print(F("skipped incomplete command "));
  if (type == 2)
    Serial.print(F("wrong parameter "));
  if (type == 3)
    Serial.print(F("failed: "));
  Serial.println(msg);
}

void sendDictionary() {
#ifndef KVP_LONG_KEY_FORMAT
  // wait for JeeLink
  delay(1000);

  Serial.print(DictionaryHeader);
  Serial.print(DictionaryValue(Temperature));
  Serial.print(DictionaryValue(Pressure));
  Serial.print(DictionaryValue(Humidity));
  Serial.print(DictionaryValue(WindSpeed));
  Serial.print(DictionaryValue(WindDirection));
  Serial.print(DictionaryValue(WindGust));
  Serial.print(DictionaryValue(WindGustRef));
  Serial.print(DictionaryValue(RainTipCount));
  Serial.print(DictionaryValue(RainSecs));
  Serial.print(DictionaryValue(Solar));
  Serial.print(DictionaryValue(VoltageSolar));
  Serial.print(DictionaryValue(VoltageCapacity));
  Serial.print(DictionaryValue(SoilLeaf));
  Serial.print(DictionaryValue(Channel));
  Serial.print(DictionaryValue(Battery));
  Serial.print(DictionaryValue(RSSI));
  Serial.print(DictionaryValue(PacketDump));
  Serial.println();
#endif
}

