"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.create = void 0;
const language_service_1 = require("@volar/language-service");
const volar_service_html_1 = require("volar-service-html");
const pug = require("./lib/languageService");
function create({ documentSelector = ['jade'], getCustomData, onDidChangeCustomData, } = {}) {
    const _htmlService = (0, volar_service_html_1.create)({
        getCustomData,
        onDidChangeCustomData,
    });
    return {
        ..._htmlService,
        name: 'pug',
        create(context) {
            const htmlService = _htmlService.create(context);
            const pugDocuments = new WeakMap();
            const pugLs = pug.getLanguageService(htmlService.provide['html/languageService']());
            return {
                ...htmlService,
                provide: {
                    'pug/pugDocument': getPugDocument,
                    'pug/languageService': () => pugLs,
                },
                provideCompletionItems(document, position, _) {
                    return worker(document, pugDocument => {
                        return pugLs.doComplete(pugDocument, position, context, htmlService.provide['html/documentContext']());
                    });
                },
                provideDiagnostics(document) {
                    return worker(document, (pugDocument) => {
                        if (pugDocument.error) {
                            return [{
                                    source: 'pug',
                                    severity: 1,
                                    code: pugDocument.error.code,
                                    message: pugDocument.error.msg,
                                    range: {
                                        start: { line: pugDocument.error.line, character: pugDocument.error.column },
                                        end: { line: pugDocument.error.line, character: pugDocument.error.column },
                                    },
                                }];
                        }
                        return [];
                    });
                },
                provideHover(document, position) {
                    return worker(document, async (pugDocument) => {
                        const hoverSettings = await context.env.getConfiguration?.('html.hover');
                        return pugLs.doHover(pugDocument, position, hoverSettings);
                    });
                },
                provideDocumentHighlights(document, position) {
                    return worker(document, pugDocument => {
                        return pugLs.findDocumentHighlights(pugDocument, position);
                    });
                },
                provideDocumentLinks(document) {
                    return worker(document, pugDocument => {
                        return pugLs.findDocumentLinks(pugDocument, htmlService.provide['html/documentContext']());
                    });
                },
                provideDocumentSymbols(document, token) {
                    return worker(document, async (pugDoc) => {
                        const htmlResult = await htmlService.provideDocumentSymbols?.(pugDoc.map.embeddedDocument, token) ?? [];
                        const pugResult = htmlResult.map(htmlSymbol => (0, language_service_1.transformDocumentSymbol)(htmlSymbol, range => pugDoc.map.getSourceRange(range))).filter((symbol) => symbol !== undefined);
                        return pugResult;
                    });
                },
                provideFoldingRanges(document) {
                    return worker(document, pugDocument => {
                        return pugLs.getFoldingRanges(pugDocument);
                    });
                },
                provideSelectionRanges(document, positions) {
                    return worker(document, pugDocument => {
                        return pugLs.getSelectionRanges(pugDocument, positions);
                    });
                },
                async provideAutoInsertionEdit(document, selection, change) {
                    // selection must at end of change
                    if (document.offsetAt(selection) !== change.rangeOffset + change.text.length) {
                        return;
                    }
                    return worker(document, async (pugDocument) => {
                        if (change.rangeLength === 0 && change.text.endsWith('=')) {
                            const enabled = (await context.env.getConfiguration?.('html.autoCreateQuotes')) ?? true;
                            if (enabled) {
                                const text = pugLs.doQuoteComplete(pugDocument, selection, await context.env.getConfiguration?.('html.completion'));
                                if (text) {
                                    return text;
                                }
                            }
                        }
                    });
                },
            };
            function worker(document, callback) {
                const pugDocument = getPugDocument(document);
                if (!pugDocument) {
                    return;
                }
                return callback(pugDocument);
            }
            function getPugDocument(document) {
                if (!matchDocument(documentSelector, document)) {
                    return;
                }
                const cache = pugDocuments.get(document);
                if (cache) {
                    const [cacheVersion, cacheDoc] = cache;
                    if (cacheVersion === document.version) {
                        return cacheDoc;
                    }
                }
                const doc = pugLs.parsePugDocument(document.getText());
                pugDocuments.set(document, [document.version, doc]);
                return doc;
            }
        },
    };
}
exports.create = create;
function matchDocument(selector, document) {
    for (const sel of selector) {
        if (sel === document.languageId || (typeof sel === 'object' && sel.language === document.languageId)) {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=index.js.map