/**
 * Copyright (c) 2010-2014, Zoltan Ujhelyi, IncQuery Labs Ltd.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.patternlanguage.emf.types;

import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.logging.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.types.BottomTypeKey;
import org.eclipse.viatra.query.patternlanguage.emf.types.ITypeSystem;
import org.eclipse.viatra.query.patternlanguage.emf.types.TypeInformation;
import org.eclipse.viatra.query.patternlanguage.emf.types.judgements.ConditionalJudgement;
import org.eclipse.viatra.query.patternlanguage.emf.types.judgements.ParameterTypeJudgement;
import org.eclipse.viatra.query.patternlanguage.emf.types.judgements.TypeConformJudgement;
import org.eclipse.viatra.query.patternlanguage.emf.types.judgements.TypeJudgement;
import org.eclipse.viatra.query.patternlanguage.emf.types.judgements.XbaseExpressionTypeJudgement;
import org.eclipse.viatra.query.patternlanguage.emf.util.AggregatorUtil;
import org.eclipse.viatra.query.patternlanguage.emf.vql.AggregatedValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.BoolValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.CallableRelation;
import org.eclipse.viatra.query.patternlanguage.emf.vql.CheckConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.CompareConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.CompareFeature;
import org.eclipse.viatra.query.patternlanguage.emf.vql.EntityType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Expression;
import org.eclipse.viatra.query.patternlanguage.emf.vql.FunctionEvaluationValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.JavaConstantValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.JavaType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ListValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.NumberValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PathExpressionConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternCall;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternCompositionConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ReferenceType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.StringValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.TypeCheckConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.UnaryTypeConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ValueReference;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Variable;
import org.eclipse.viatra.query.patternlanguage.emf.vql.VariableReference;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.context.common.JavaTransitiveInstancesKey;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.typesystem.IBatchTypeResolver;
import org.eclipse.xtext.xbase.typesystem.computation.NumberLiterals;

/**
 * @author Zoltan Ujhelyi
 * @since 2.0
 */
@SuppressWarnings("all")
public class PatternLanguageTypeRules {
  @Inject
  private ITypeSystem typeSystem;
  
  @Inject
  private IBatchTypeResolver typeResolver;
  
  @Inject
  private NumberLiterals literals;
  
  @Inject
  private Logger logger;
  
  /**
   * @since 1.7
   */
  public void loadParameterVariableTypes(final Pattern pattern, final TypeInformation information) {
    final Consumer<Variable> _function = (Variable parameter) -> {
      final IInputKey typeKey = this.typeSystem.extractTypeDescriptor(parameter.getType());
      information.declareType(parameter, typeKey);
    };
    pattern.getParameters().forEach(_function);
  }
  
  protected void _inferTypes(final Pattern pattern, final TypeInformation information) {
    final Consumer<Variable> _function = (Variable parameter) -> {
      boolean _isValidType = this.typeSystem.isValidType(parameter.getType());
      if (_isValidType) {
        final IInputKey typeKey = this.typeSystem.extractTypeDescriptor(parameter.getType());
        information.declareType(parameter, typeKey);
      }
      final Consumer<Variable> _function_1 = (Variable ref) -> {
        information.provideType(new TypeConformJudgement(ref, parameter) {
          @Override
          public Set<Expression> getDependingExpressions() {
            return Collections.<Expression>unmodifiableSet(CollectionLiterals.<Expression>newHashSet());
          }
        });
        TypeConformJudgement _typeConformJudgement = new TypeConformJudgement(parameter, ref);
        information.provideType(_typeConformJudgement);
      };
      PatternLanguageHelper.getLocalReferencesOfParameter(parameter).forEach(_function_1);
    };
    pattern.getParameters().forEach(_function);
  }
  
  protected void _inferTypes(final CheckConstraint constraint, final TypeInformation information) {
  }
  
  protected void _inferTypes(final CompareConstraint constraint, final TypeInformation information) {
    if ((((constraint.getFeature() == CompareFeature.EQUALITY) && (constraint.getLeftOperand() != null)) && (constraint.getRightOperand() != null))) {
      ValueReference _leftOperand = constraint.getLeftOperand();
      ValueReference _rightOperand = constraint.getRightOperand();
      TypeConformJudgement _typeConformJudgement = new TypeConformJudgement(_leftOperand, _rightOperand);
      information.provideType(_typeConformJudgement);
      ValueReference _rightOperand_1 = constraint.getRightOperand();
      ValueReference _leftOperand_1 = constraint.getLeftOperand();
      TypeConformJudgement _typeConformJudgement_1 = new TypeConformJudgement(_rightOperand_1, _leftOperand_1);
      information.provideType(_typeConformJudgement_1);
    }
  }
  
  protected void _inferTypes(final PatternCompositionConstraint constraint, final TypeInformation information) {
    boolean _isNegative = constraint.isNegative();
    boolean _not = (!_isNegative);
    if (_not) {
      final CallableRelation call = constraint.getCall();
      this.inferCallTypes(((PatternCall) call), information);
    }
  }
  
  private void inferCallTypes(final PatternCall call, final TypeInformation information) {
    final Pattern pattern = call.getPatternRef();
    for (int i = 0; (i < Math.min(call.getParameters().size(), pattern.getParameters().size())); i++) {
      ValueReference _get = call.getParameters().get(i);
      Variable _get_1 = pattern.getParameters().get(i);
      ParameterTypeJudgement _parameterTypeJudgement = new ParameterTypeJudgement(_get, _get_1);
      information.provideType(_parameterTypeJudgement);
    }
  }
  
  protected void _inferTypes(final TypeCheckConstraint constraint, final TypeInformation information) {
    boolean _isNonSimpleConstraint = PatternLanguageHelper.isNonSimpleConstraint(constraint);
    if (_isNonSimpleConstraint) {
      return;
    }
    final EntityType constraintType = constraint.getType();
    if (((constraintType instanceof JavaType) && this.typeSystem.isValidType(constraintType))) {
      final IInputKey sourceType = this.typeSystem.extractTypeDescriptor(constraintType);
      if ((sourceType != null)) {
        VariableReference _var = constraint.getVar();
        TypeJudgement _typeJudgement = new TypeJudgement(_var, sourceType);
        information.provideType(_typeJudgement);
      }
    }
  }
  
  protected void _inferTypes(final PathExpressionConstraint constraint, final TypeInformation information) {
    boolean _isNonSimpleConstraint = PatternLanguageHelper.isNonSimpleConstraint(constraint);
    if (_isNonSimpleConstraint) {
      return;
    }
    IInputKey _xifexpression = null;
    boolean _isValidType = this.typeSystem.isValidType(constraint.getSourceType());
    boolean _not = (!_isValidType);
    if (_not) {
      _xifexpression = BottomTypeKey.INSTANCE;
    } else {
      _xifexpression = this.typeSystem.extractTypeDescriptor(constraint.getSourceType());
    }
    final IInputKey sourceType = _xifexpression;
    ReferenceType tailType = IterableExtensions.<ReferenceType>last(constraint.getEdgeTypes());
    IInputKey _xifexpression_1 = null;
    boolean _isValidType_1 = this.typeSystem.isValidType(tailType);
    boolean _not_1 = (!_isValidType_1);
    if (_not_1) {
      _xifexpression_1 = BottomTypeKey.INSTANCE;
    } else {
      _xifexpression_1 = this.typeSystem.extractTypeDescriptor(tailType);
    }
    final IInputKey targetType = _xifexpression_1;
    if (((sourceType != null) && (targetType != null))) {
      VariableReference _src = constraint.getSrc();
      TypeJudgement _typeJudgement = new TypeJudgement(_src, sourceType);
      information.provideType(_typeJudgement);
      ValueReference _dst = constraint.getDst();
      TypeJudgement _typeJudgement_1 = new TypeJudgement(_dst, targetType);
      information.provideType(_typeJudgement_1);
    }
  }
  
  protected void _inferTypes(final AggregatedValue reference, final TypeInformation information) {
    JvmDeclaredType _aggregator = null;
    if (reference!=null) {
      _aggregator=reference.getAggregator();
    }
    boolean _tripleEquals = (_aggregator == null);
    if (_tripleEquals) {
      return;
    }
    boolean _and = false;
    CallableRelation _call = reference.getCall();
    if (!(_call instanceof PatternCall)) {
      _and = false;
    } else {
      CallableRelation _call_1 = reference.getCall();
      Pattern _patternRef = null;
      if (((PatternCall) _call_1)!=null) {
        _patternRef=((PatternCall) _call_1).getPatternRef();
      }
      boolean _tripleEquals_1 = (_patternRef == null);
      _and = _tripleEquals_1;
    }
    if (_and) {
      return;
    }
    final List<VariableReference> values = AggregatorUtil.getAllAggregatorVariables(reference);
    int _size = values.size();
    boolean _tripleEquals_2 = (_size == 0);
    if (_tripleEquals_2) {
      boolean _mustHaveAggregatorVariables = AggregatorUtil.mustHaveAggregatorVariables(reference);
      if (_mustHaveAggregatorVariables) {
        return;
      }
      final List<JvmType> returnTypes = AggregatorUtil.getReturnTypes(reference.getAggregator());
      if (((returnTypes == null) || (returnTypes.size() != 1))) {
        this.logger.warning(
          String.format("Return type for aggregator %s is non uniquely specified.", 
            reference.getAggregator().getSimpleName()));
        return;
      }
      final JvmType returnType = returnTypes.get(0);
      String _identifier = returnType.getIdentifier();
      JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(_identifier);
      TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
      information.provideType(_typeJudgement);
    } else {
      if (((values.size() != 1) || (!AggregatorUtil.mustHaveAggregatorVariables(reference)))) {
        return;
      }
      final List<JvmType> parameterTypes = AggregatorUtil.getParameterTypes(reference.getAggregator());
      final List<JvmType> returnTypes_1 = AggregatorUtil.getReturnTypes(reference.getAggregator());
      if (((returnTypes_1 == null) || (returnTypes_1.size() != parameterTypes.size()))) {
        this.logger.warning(
          String.format(
            "Incorrect aggregator type annotation for aggregator %s: Different number of parameters and return types", 
            reference.getAggregator().getIdentifier()));
        return;
      }
      final HashSet<JvmType> returnTypeSet = new HashSet<JvmType>(returnTypes_1);
      int _size_1 = returnTypeSet.size();
      int _size_2 = returnTypes_1.size();
      final boolean returnTypeUnique = (_size_1 == _size_2);
      final int index = AggregatorUtil.getAggregateVariableIndex(reference);
      final List<ValueReference> callParameters = PatternLanguageHelper.getCallParameters(reference.getCall());
      for (int i = 0; (i < returnTypes_1.size()); i++) {
        {
          String _identifier_1 = returnTypes_1.get(i).getIdentifier();
          JavaTransitiveInstancesKey _javaTransitiveInstancesKey_1 = new JavaTransitiveInstancesKey(_identifier_1);
          ValueReference _get = callParameters.get(index);
          String _identifier_2 = parameterTypes.get(i).getIdentifier();
          JavaTransitiveInstancesKey _javaTransitiveInstancesKey_2 = new JavaTransitiveInstancesKey(_identifier_2);
          ConditionalJudgement _conditionalJudgement = new ConditionalJudgement(reference, _javaTransitiveInstancesKey_1, _get, _javaTransitiveInstancesKey_2);
          information.provideType(_conditionalJudgement);
          if (returnTypeUnique) {
            ValueReference _get_1 = callParameters.get(index);
            String _identifier_3 = parameterTypes.get(i).getIdentifier();
            JavaTransitiveInstancesKey _javaTransitiveInstancesKey_3 = new JavaTransitiveInstancesKey(_identifier_3);
            String _identifier_4 = returnTypes_1.get(i).getIdentifier();
            JavaTransitiveInstancesKey _javaTransitiveInstancesKey_4 = new JavaTransitiveInstancesKey(_identifier_4);
            ConditionalJudgement _conditionalJudgement_1 = new ConditionalJudgement(_get_1, _javaTransitiveInstancesKey_3, reference, _javaTransitiveInstancesKey_4);
            information.provideType(_conditionalJudgement_1);
          }
        }
      }
      this.inferParameterType(reference.getCall(), information, callParameters.get(index), index);
    }
  }
  
  private void _inferParameterType(final PatternCall call, final TypeInformation information, final ValueReference value, final int parameterIndex) {
    if (((parameterIndex < call.getParameters().size()) && (parameterIndex < call.getPatternRef().getParameters().size()))) {
      Variable _get = call.getPatternRef().getParameters().get(parameterIndex);
      ParameterTypeJudgement _parameterTypeJudgement = new ParameterTypeJudgement(value, _get);
      information.provideType(_parameterTypeJudgement);
    }
  }
  
  private void _inferParameterType(final UnaryTypeConstraint constraint, final TypeInformation information, final ValueReference value, final int parameterIndex) {
    IInputKey _extractTypeDescriptor = this.typeSystem.extractTypeDescriptor(constraint.getType());
    TypeJudgement _typeJudgement = new TypeJudgement(value, _extractTypeDescriptor);
    information.provideType(_typeJudgement);
  }
  
  private void _inferParameterType(final PathExpressionConstraint constraint, final TypeInformation information, final ValueReference value, final int parameterIndex) {
    if ((parameterIndex == 0)) {
      IInputKey _extractTypeDescriptor = this.typeSystem.extractTypeDescriptor(constraint.getSourceType());
      TypeJudgement _typeJudgement = new TypeJudgement(value, _extractTypeDescriptor);
      information.provideType(_typeJudgement);
    } else {
      if ((parameterIndex == 1)) {
        EList<ReferenceType> _edgeTypes = constraint.getEdgeTypes();
        int _size = constraint.getEdgeTypes().size();
        int _minus = (_size - 1);
        IInputKey _extractColumnDescriptor = this.typeSystem.extractColumnDescriptor(_edgeTypes.get(_minus), 1);
        TypeJudgement _typeJudgement_1 = new TypeJudgement(value, _extractColumnDescriptor);
        information.provideType(_typeJudgement_1);
      } else {
        throw new IllegalArgumentException("Invalid parameter index");
      }
    }
  }
  
  protected void _inferTypes(final Expression reference, final TypeInformation information) {
  }
  
  /**
   * @since 2.7
   */
  protected void _inferTypes(final JavaConstantValue reference, final TypeInformation information) {
    String _identifier = reference.getFieldRef().getType().getType().getIdentifier();
    final JavaTransitiveInstancesKey type = new JavaTransitiveInstancesKey(_identifier);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, type);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final FunctionEvaluationValue reference, final TypeInformation information) {
    XExpression _expression = reference.getExpression();
    boolean _isUnwind = reference.isUnwind();
    XbaseExpressionTypeJudgement _xbaseExpressionTypeJudgement = new XbaseExpressionTypeJudgement(reference, _expression, this.typeResolver, _isUnwind);
    information.provideType(_xbaseExpressionTypeJudgement);
  }
  
  protected void _inferTypes(final BoolValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(Boolean.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  /**
   * @since 1.5
   */
  protected void _inferTypes(final NumberValue reference, final TypeInformation information) {
    if (((reference.getValue() != null) && (!reference.getValue().eIsProxy()))) {
      final Class<? extends Number> type = this.literals.getJavaType(reference.getValue());
      JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(type);
      TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
      information.provideType(_typeJudgement);
    }
  }
  
  protected void _inferTypes(final ListValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(List.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final StringValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(String.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final VariableReference reference, final TypeInformation information) {
  }
  
  public void inferTypes(final EObject reference, final TypeInformation information) {
    if (reference instanceof AggregatedValue) {
      _inferTypes((AggregatedValue)reference, information);
      return;
    } else if (reference instanceof BoolValue) {
      _inferTypes((BoolValue)reference, information);
      return;
    } else if (reference instanceof FunctionEvaluationValue) {
      _inferTypes((FunctionEvaluationValue)reference, information);
      return;
    } else if (reference instanceof ListValue) {
      _inferTypes((ListValue)reference, information);
      return;
    } else if (reference instanceof NumberValue) {
      _inferTypes((NumberValue)reference, information);
      return;
    } else if (reference instanceof StringValue) {
      _inferTypes((StringValue)reference, information);
      return;
    } else if (reference instanceof JavaConstantValue) {
      _inferTypes((JavaConstantValue)reference, information);
      return;
    } else if (reference instanceof TypeCheckConstraint) {
      _inferTypes((TypeCheckConstraint)reference, information);
      return;
    } else if (reference instanceof VariableReference) {
      _inferTypes((VariableReference)reference, information);
      return;
    } else if (reference instanceof CheckConstraint) {
      _inferTypes((CheckConstraint)reference, information);
      return;
    } else if (reference instanceof CompareConstraint) {
      _inferTypes((CompareConstraint)reference, information);
      return;
    } else if (reference instanceof PathExpressionConstraint) {
      _inferTypes((PathExpressionConstraint)reference, information);
      return;
    } else if (reference instanceof PatternCompositionConstraint) {
      _inferTypes((PatternCompositionConstraint)reference, information);
      return;
    } else if (reference instanceof Expression) {
      _inferTypes((Expression)reference, information);
      return;
    } else if (reference instanceof Pattern) {
      _inferTypes((Pattern)reference, information);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(reference, information).toString());
    }
  }
  
  private void inferParameterType(final CallableRelation constraint, final TypeInformation information, final ValueReference value, final int parameterIndex) {
    if (constraint instanceof PathExpressionConstraint) {
      _inferParameterType((PathExpressionConstraint)constraint, information, value, parameterIndex);
      return;
    } else if (constraint instanceof PatternCall) {
      _inferParameterType((PatternCall)constraint, information, value, parameterIndex);
      return;
    } else if (constraint instanceof UnaryTypeConstraint) {
      _inferParameterType((UnaryTypeConstraint)constraint, information, value, parameterIndex);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(constraint, information, value, parameterIndex).toString());
    }
  }
}
