/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.lmdb;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import org.eclipse.rdf4j.common.io.IOUtil;
import org.eclipse.rdf4j.common.iteration.CloseableIteration;
import org.eclipse.rdf4j.model.IRI;
import org.eclipse.rdf4j.model.Resource;
import org.eclipse.rdf4j.model.ValueFactory;
import org.eclipse.rdf4j.sail.SailException;
import org.eclipse.rdf4j.sail.lmdb.LmdbSailStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ContextStore
implements Iterable<Resource> {
    static final Logger logger = LoggerFactory.getLogger(ContextStore.class);
    private static final String FILE_NAME = "contexts.dat";
    private static final byte[] MAGIC_NUMBER = new byte[]{110, 99, 102};
    private static final byte FILE_FORMAT_VERSION = 1;
    private final File file;
    private final Map<Resource, Long> contextInfoMap;
    private volatile boolean contentsChanged;
    private final ValueFactory valueFactory;
    private final LmdbSailStore store;

    ContextStore(LmdbSailStore store, File dataDir) throws IOException {
        Objects.requireNonNull(store);
        Objects.requireNonNull(dataDir);
        this.file = new File(dataDir, FILE_NAME);
        this.valueFactory = store.getValueFactory();
        this.store = store;
        this.contextInfoMap = new HashMap<Resource, Long>(16);
        try {
            this.readContextsFromFile();
        }
        catch (FileNotFoundException fe) {
            logger.debug("context index has not been created yet: " + fe.getMessage());
            this.initializeContextCache();
            this.writeContextsToFile();
            logger.debug("context index construction complete");
        }
        catch (IOException ioe) {
            logger.info("could not read context index: " + ioe.getMessage(), (Throwable)ioe);
            logger.debug("attempting reconstruction from store (this may take a while)");
            this.initializeContextCache();
            this.writeContextsToFile();
            logger.info("context index reconstruction complete");
        }
    }

    void increment(Resource context) {
        this.contextInfoMap.merge(context, 1L, (size, one) -> size + one);
        this.contentsChanged = true;
    }

    void decrementBy(Resource context, long amount) {
        this.contextInfoMap.computeIfPresent(context, (c, size) -> size <= amount ? null : Long.valueOf(size - amount));
        this.contentsChanged = true;
    }

    @Override
    public Iterator<Resource> iterator() {
        return this.contextInfoMap.keySet().iterator();
    }

    void clear() {
        if (!this.contextInfoMap.isEmpty()) {
            this.contextInfoMap.clear();
            this.contentsChanged = true;
        }
    }

    void close() {
    }

    void sync() throws IOException {
        if (this.contentsChanged) {
            this.writeContextsToFile();
            this.contentsChanged = false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeContextsToFile() throws IOException {
        File file = this.file;
        synchronized (file) {
            try (DataOutputStream out = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(this.file)));){
                out.write(MAGIC_NUMBER);
                out.writeByte(1);
                out.writeInt(this.contextInfoMap.size());
                for (Map.Entry<Resource, Long> entry : this.contextInfoMap.entrySet()) {
                    out.writeLong(entry.getValue());
                    out.writeBoolean(entry.getKey() instanceof IRI);
                    out.writeUTF(entry.getKey().stringValue());
                }
            }
        }
    }

    private void initializeContextCache() throws IOException {
        logger.debug("initializing context cache");
        try (CloseableIteration<Resource, SailException> contextIter = this.store.getContexts();){
            while (contextIter.hasNext()) {
                this.increment((Resource)contextIter.next());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readContextsFromFile() throws IOException {
        File file = this.file;
        synchronized (file) {
            if (!this.file.exists()) {
                throw new FileNotFoundException("context index file " + this.file + " does not exist");
            }
            try (DataInputStream in = new DataInputStream(new BufferedInputStream(new FileInputStream(this.file)));){
                byte[] magicNumber = IOUtil.readBytes((InputStream)in, MAGIC_NUMBER.length);
                if (!Arrays.equals(magicNumber, MAGIC_NUMBER)) {
                    throw new IOException("File doesn't contain compatible context data");
                }
                byte version = in.readByte();
                if (version > 1) {
                    throw new IOException("Unable to read context file; it uses a newer file format");
                }
                if (version != 1) {
                    throw new IOException("Unable to read context file; invalid file format version: " + version);
                }
                int size = in.readInt();
                try {
                    while (true) {
                        long contextSize = in.readLong();
                        boolean isIRI = in.readBoolean();
                        String contextId = in.readUTF();
                        Resource context = isIRI ? this.valueFactory.createIRI(contextId) : this.valueFactory.createBNode(contextId);
                        this.contextInfoMap.put(context, contextSize);
                    }
                }
                catch (EOFException e) {
                }
                catch (IllegalArgumentException e) {
                    throw new IOException("unable to parse context identifier: ", e);
                }
                if (this.contextInfoMap.size() != size) {
                    throw new IOException("Unable to read context file; size checksum validation failed");
                }
            }
        }
    }
}

