/*****************************************************************************
 * Copyright (c) 2014, 2017, 2019, 2021, 2025 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Benoit Maggi (CEA LIST) benoit.maggi@cea.fr - Initial API and implementation
 *  Gabriel Pascual (ALL4TEC) gabriel.pascual@all4tec.net - bug 438511
 *  Thanh Liem PHAN (ALL4TEC) thanhliem.phan@all4tec.net - bug 511045
 *  Ansgar Radermacher (CEA LIST) ansgar.radermacher@cea.fr - bug 521279 (copy/paste between models), bug 573807
 *  Pauline DEVILLE (CEA LIST) pauline.deville@cea.fr - bug 552410
 *  Vincent LORENZO (CEA LIST) vincent.lorenzo@cea.fr - Issue 16
 *****************************************************************************/
package org.eclipse.papyrus.uml.tools.commands;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.papyrus.infra.emf.utils.EMFHelper;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Extension;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * A Command to apply a Stereotype and its data to an UML Element
 *
 * @author Benoit Maggi
 */
public class DuplicateStereotypeCommand extends RecordingCommand {

	protected Element element;

	protected EObject stereotypeApplication;

	protected Stereotype stereotypeInTargetContext;

	/**
	 * This map is used to share the stereotype applications with the command {@link UpdateStereotypeValueCommand}. @see issue GL-15
	 */
	private Map<EObject, EObject> sourceSteAppVSCopiedSteApp;

	/**
	 *
	 * Constructor.
	 *
	 * @param domain
	 *            The editing domain
	 * @param element
	 *            The UML Element on which the stereotype will be applied
	 * @param stereotypeApplication
	 *            The stereotype to apply
	 */
	public DuplicateStereotypeCommand(TransactionalEditingDomain domain, Element element, EObject stereotypeApplication) {
		this(domain, element, element, stereotypeApplication);
	}

	/**
	 *
	 * Constructor.
	 *
	 * @param domain
	 *            The editing domain
	 * @param element
	 *            The UML Element on which the stereotype will be applied
	 * @param targetContainer
	 *            target container for the element. This information is required to reload the stereotype to the target context.
	 *            It can not be deduced from the element, since the latter has not been added to the target container yet.
	 * @param stereotypeApplication
	 *            The stereotype to apply
	 */
	public DuplicateStereotypeCommand(TransactionalEditingDomain domain, Element element, Element targetContainer, EObject stereotypeApplication) {
		this(domain, element, targetContainer, stereotypeApplication, null);
	}

	/**
	 *
	 * Constructor.
	 *
	 * @param domain
	 *            The editing domain
	 * @param element
	 *            The UML Element on which the stereotype will be applied
	 * @param targetContainer
	 *            target container for the element. This information is required to reload the stereotype to the target context.
	 *            It can not be deduced from the element, since the latter has not been added to the target container yet.
	 * @param stereotypeApplication
	 *            The stereotype to apply
	 * @param papyrusClipboard
	 *            the PapyrusClipboard
	 */
	public DuplicateStereotypeCommand(TransactionalEditingDomain domain, Element element, Element targetContainer, EObject stereotypeApplication, Map<EObject, EObject> sharedStereotypeApplicationMap) {
		super(domain);
		this.element = element;
		this.stereotypeApplication = stereotypeApplication;
		// reload the stereotype in the new Context-ResourceSet (Required because in org.eclipse.uml2.uml.internal.operations.PackageOperations
		// L960 in getProfileApplication the test is using == instead of equals)
		Stereotype stereotype = UMLUtil.getStereotype(stereotypeApplication);
		// might be null, if copied model element does no longer exist (since editor is closed)
		if (stereotype != null) {
			Stereotype stereotypeInTargetContext = EMFHelper.reloadIntoContext(stereotype, targetContainer);
			this.stereotypeInTargetContext = stereotypeInTargetContext;
		}
		this.sourceSteAppVSCopiedSteApp = sharedStereotypeApplicationMap;
	}

	public Stereotype getStereotypeInTargetContext() {
		return stereotypeInTargetContext;
	}

	@Override
	protected void doExecute() {
		// Retrieve the stereotype application for the element
		EObject applyStereotype = element.getStereotypeApplication(stereotypeInTargetContext);
		// If the stereotype is not applied yet
		if (null == applyStereotype && null != element.eContainer()) {
			// Then apply it safely without triggering the exception when applying an already applied stereotype (bug 511045)
			applyStereotype = element.applyStereotype(stereotypeInTargetContext);
		}

		// fill the map with the initial stereotype application and the stereotype application applied on the pasted element
		if (sourceSteAppVSCopiedSteApp != null) {
			sourceSteAppVSCopiedSteApp.put(stereotypeApplication, applyStereotype);
		}
		EList<EStructuralFeature> eStructuralFeatures = applyStereotype.eClass().getEAllStructuralFeatures();
		for (EStructuralFeature eStructuralFeature : eStructuralFeatures) {
			String name = eStructuralFeature.getName();
			if (!name.startsWith(Extension.METACLASS_ROLE_PREFIX) && eStructuralFeature.isChangeable()) {
				if (eStructuralFeature instanceof EReference && ((EReference) eStructuralFeature).isContainment()) {
					// In case that the structural feature is containment then we should copy the value not only change the reference
					Object value = stereotypeApplication.eGet(eStructuralFeature);
					if (value instanceof EObject) {
						EObject valueCopy = EcoreUtil.copy((EObject) value);
						applyStereotype.eSet(eStructuralFeature, valueCopy);
					} else if (value instanceof Collection<?>) {
						Collection<?> listValue = (Collection<?>) value;
						if (listValue.stream().allMatch(v -> v instanceof EObject)) {
							Collection<?> valueCopy = EcoreUtil.copyAll(listValue);
							applyStereotype.eSet(eStructuralFeature, valueCopy);
						}
					}
				} else if (eStructuralFeature instanceof EReference eRef && !eRef.isContainment() && isTypedWithAStereotype(eRef)) {
					// do nothing.
					// it is managed by org.eclipse.papyrus.uml.tools.commands.UpdateStereotypeValueCommand
				} else {
					applyStereotype.eSet(eStructuralFeature, stereotypeApplication.eGet(eStructuralFeature));
				}
			}
		}
	}

	/**
	 *
	 * @param eobject
	 *            an eobject
	 * @return
	 *         <code>true</code> if the EReference is typed with a Stereotype
	 */
	private boolean isTypedWithAStereotype(final EReference eReference) {
		final EClassifier eType = eReference.getEType();
		boolean isStereotype = false;
		if (eType instanceof EClass eClass) {
			final Iterator<EReference> iter = eClass.getEAllReferences().iterator();
			while (iter.hasNext() && !isStereotype) {
				EReference eRef = iter.next();
				isStereotype = !eRef.getName().startsWith(Extension.METACLASS_ROLE_PREFIX)
						&& eRef.isChangeable()
						&& !eRef.isContainment();
			}
		}
		return isStereotype;
	}
}
