/* Any copyright is dedicated to the Public Domain.
 http://creativecommons.org/publicdomain/zero/1.0/ */

"use strict";

// Test that the dimensions of the preview tooltips are correctly updated to fit their
// content.

// Small 32x32 image.
const BASE_64_URL =
  "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr" +
  "0AAAAUElEQVRYR+3UsQkAQAhD0TjJ7T+Wk3gbxMIizbcVITwwJWlkZtptpXp+v94TAAEE4gLTvgfOf770RB" +
  "EAAQTiAvEiIgACCMQF4kVEAAQQSAt8xsyeAW6R8eIAAAAASUVORK5CYII=";

add_task(async function () {
  await pushPref("devtools.inspector.chrome.three-pane-enabled", false);
  await addTab(
    "data:text/html;charset=utf-8," +
      encodeURIComponent(`
      <style>
        html {
          /* Using a long value to ensure preview tooltip for variable will be */
          /* wider than the preview tooltip for the test 32x32 image. */
          --test-var-wider-than-image: color(from lightgoldenrodyellow xyz calc(x + 0.75) y calc(z - 0.35));
        }

        #target {
          color: var(--test-var-wider-than-image);
          background: url(${BASE_64_URL});
        }
      </style>
      <div id="target">inspect me</div>
    `)
  );
  const { inspector, view } = await openRuleView();
  await selectNode("#target", inspector);

  // Note: See intermittent Bug 1721743.
  // On linux webrender opt, the inspector might open the ruleview before it has
  // been populated with the rules for the div.
  info("Wait until the rule view property is rendered");
  const colorPropertyElement = await waitFor(() =>
    getRuleViewProperty(view, "#target", "color")
  );

  // Retrieve the element for `--test-var` on which the CSS variable tooltip will appear.
  const colorPropertySpan = colorPropertyElement.valueSpan;
  const colorVariableElement =
    colorPropertySpan.querySelector(".ruleview-variable");
  ok(!!colorVariableElement, "Found the variable element");

  // Retrieve the element for the background url on which the image preview will appear.
  const backgroundPropertySpan = getRuleViewProperty(
    view,
    "#target",
    "background"
  ).valueSpan;
  const backgroundUrlElement =
    backgroundPropertySpan.querySelector(".theme-link");

  info("Show preview tooltip for CSS variable");
  let previewTooltip = await assertShowPreviewTooltip(
    view,
    colorVariableElement
  );
  // Measure tooltip dimensions.
  let tooltipRect = previewTooltip.panel.getBoundingClientRect();
  const originalHeight = tooltipRect.height;
  const originalWidth = tooltipRect.width;
  info(`Original dimensions: ${originalWidth} x ${originalHeight}`);
  await assertTooltipHiddenOnMouseOut(previewTooltip, colorVariableElement);

  info("Show preview tooltip for background url");
  previewTooltip = await assertShowPreviewTooltip(view, backgroundUrlElement);
  // Compare new tooltip dimensions to previous measures.
  tooltipRect = previewTooltip.panel.getBoundingClientRect();
  info(
    `Image preview dimensions: ${tooltipRect.width} x ${tooltipRect.height}`
  );
  Assert.greater(
    tooltipRect.height,
    originalHeight,
    "Tooltip is taller for image preview"
  );
  Assert.less(
    tooltipRect.width,
    originalWidth,
    "Tooltip is narrower for image preview"
  );
  await assertTooltipHiddenOnMouseOut(previewTooltip, colorVariableElement);

  info("Show preview tooltip for CSS variable again");
  previewTooltip = await assertShowPreviewTooltip(view, colorVariableElement);
  // Check measures are identical to initial ones.
  tooltipRect = previewTooltip.panel.getBoundingClientRect();
  info(
    `CSS variable tooltip dimensions: ${tooltipRect.width} x ${tooltipRect.height}`
  );
  is(
    tooltipRect.height,
    originalHeight,
    "Tooltip has the same height as the original"
  );
  is(
    tooltipRect.width,
    originalWidth,
    "Tooltip has the same width as the original"
  );
  await assertTooltipHiddenOnMouseOut(previewTooltip, colorVariableElement);
});
