// Copyright 2011 The Closure Library Authors. All Rights Reserved.
// Use of this source code is governed by the Apache License, Version 2.0.

goog.provide('goog.vec.Vec3Test');
goog.setTestOnly('goog.vec.Vec3Test');

goog.require('goog.vec.Float32Array');
goog.require('goog.vec.Vec3');
goog.require('goog.testing.jsunit');

function testDeprecatedConstructor() {
  var v = goog.vec.Vec3.create();
  assertElementsEquals(0, v[0]);
  assertEquals(0, v[1]);
  assertEquals(0, v[2]);

  assertElementsEquals([0, 0, 0], goog.vec.Vec3.create());

  assertElementsEquals([1, 2, 3], goog.vec.Vec3.createFromValues(1, 2, 3));

  assertElementsEquals([1, 2, 3], goog.vec.Vec3.createFromArray([1, 2, 3]));

  v = goog.vec.Vec3.createFromValues(1, 2, 3);
  assertElementsEquals([1, 2, 3], goog.vec.Vec3.clone(v));
}

function testConstructor() {
  var v = goog.vec.Vec3.createFloat32();
  assertElementsEquals(0, v[0]);
  assertEquals(0, v[1]);
  assertEquals(0, v[2]);

  assertElementsEquals([0, 0, 0], goog.vec.Vec3.createFloat32());

  goog.vec.Vec3.setFromValues(v, 1, 2, 3);
  assertElementsEquals([1, 2, 3], v);

  var w = goog.vec.Vec3.createFloat64();
  assertElementsEquals(0, w[0]);
  assertEquals(0, w[1]);
  assertEquals(0, w[2]);

  assertElementsEquals([0, 0, 0], goog.vec.Vec3.createFloat64());

  goog.vec.Vec3.setFromValues(w, 1, 2, 3);
  assertElementsEquals([1, 2, 3], w);
}


function testSet() {
  var v = goog.vec.Vec3.createFloat32();
  goog.vec.Vec3.setFromValues(v, 1, 2, 3);
  assertElementsEquals([1, 2, 3], v);

  goog.vec.Vec3.setFromArray(v, [4, 5, 6]);
  assertElementsEquals([4, 5, 6], v);

  var w = goog.vec.Vec3.createFloat32();
  goog.vec.Vec3.setFromValues(w, 1, 2, 3);
  assertElementsEquals([1, 2, 3], w);

  goog.vec.Vec3.setFromArray(w, [4, 5, 6]);
  assertElementsEquals([4, 5, 6], w);
}

function testAdd() {
  var v0 = goog.vec.Vec3.createFloat32FromArray([1, 2, 3]);
  var v1 = goog.vec.Vec3.createFloat32FromArray([4, 5, 6]);
  var v2 = goog.vec.Vec3.cloneFloat32(v0);

  goog.vec.Vec3.add(v2, v1, v2);
  assertElementsEquals([1, 2, 3], v0);
  assertElementsEquals([4, 5, 6], v1);
  assertElementsEquals([5, 7, 9], v2);

  goog.vec.Vec3.add(goog.vec.Vec3.add(v0, v1, v2), v0, v2);
  assertElementsEquals([6, 9, 12], v2);
}

function testSubtract() {
  var v0 = goog.vec.Vec3.createFloat32FromArray([1, 2, 3]);
  var v1 = goog.vec.Vec3.createFloat32FromArray([4, 5, 6]);
  var v2 = goog.vec.Vec3.cloneFloat32(v0);

  goog.vec.Vec3.subtract(v2, v1, v2);
  assertElementsEquals([1, 2, 3], v0);
  assertElementsEquals([4, 5, 6], v1);
  assertElementsEquals([-3, -3, -3], v2);

  goog.vec.Vec3.setFromValues(v2, 0, 0, 0);
  goog.vec.Vec3.subtract(v1, v0, v2);
  assertElementsEquals([3, 3, 3], v2);

  v2 = goog.vec.Vec3.cloneFloat32(v0);
  goog.vec.Vec3.subtract(v2, v1, v2);
  assertElementsEquals([-3, -3, -3], v2);

  goog.vec.Vec3.subtract(goog.vec.Vec3.subtract(v1, v0, v2), v0, v2);
  assertElementsEquals([2, 1, 0], v2);
}

function testNegate() {
  var v0 = goog.vec.Vec3.createFloat32FromArray([1, 2, 3]);
  var v1 = goog.vec.Vec3.createFloat32();

  goog.vec.Vec3.negate(v0, v1);
  assertElementsEquals([-1, -2, -3], v1);
  assertElementsEquals([1, 2, 3], v0);

  goog.vec.Vec3.negate(v0, v0);
  assertElementsEquals([-1, -2, -3], v0);
}

function testAbs() {
  var v0 = goog.vec.Vec3.createFloat32FromValues(-1, -2, -3);
  var v1 = goog.vec.Vec3.createFloat32();

  goog.vec.Vec3.abs(v0, v1);
  assertElementsEquals([1, 2, 3], v1);
  assertElementsEquals([-1, -2, -3], v0);

  goog.vec.Vec3.abs(v0, v0);
  assertElementsEquals([1, 2, 3], v0);
}

function testScale() {
  var v0 = goog.vec.Vec3.createFloat32FromArray([1, 2, 3]);
  var v1 = goog.vec.Vec3.createFloat32();

  goog.vec.Vec3.scale(v0, 4, v1);
  assertElementsEquals([4, 8, 12], v1);
  assertElementsEquals([1, 2, 3], v0);

  goog.vec.Vec3.setFromArray(v1, v0);
  goog.vec.Vec3.scale(v1, 5, v1);
  assertElementsEquals([5, 10, 15], v1);
}

function testMagnitudeSquared() {
  var v0 = goog.vec.Vec3.createFloat32FromArray([1, 2, 3]);
  assertEquals(14, goog.vec.Vec3.magnitudeSquared(v0));
}

function testMagnitude() {
  var v0 = goog.vec.Vec3.createFloat32FromArray([1, 2, 3]);
  assertEquals(Math.sqrt(14), goog.vec.Vec3.magnitude(v0));
}

function testNormalize() {
  var v0 = goog.vec.Vec3.createFloat32FromArray([2, 3, 4]);
  var v1 = goog.vec.Vec3.create();
  var v2 = goog.vec.Vec3.create();
  goog.vec.Vec3.scale(
      v0, 1 / goog.vec.Vec3.magnitude(v0), v2);

  goog.vec.Vec3.normalize(v0, v1);
  assertElementsEquals(v2, v1);
  assertElementsEquals([2, 3, 4], v0);

  goog.vec.Vec3.setFromArray(v1, v0);
  goog.vec.Vec3.normalize(v1, v1);
  assertElementsEquals(v2, v1);
}

function testDot() {
  var v0 = goog.vec.Vec3.createFloat32FromArray([1, 2, 3]);
  var v1 = goog.vec.Vec3.createFloat32FromArray([4, 5, 6]);
  assertEquals(32, goog.vec.Vec3.dot(v0, v1));
  assertEquals(32, goog.vec.Vec3.dot(v1, v0));
}

function testCross() {
  var v0 = goog.vec.Vec3.createFloat32FromArray([1, 2, 3]);
  var v1 = goog.vec.Vec3.createFloat32FromArray([4, 5, 6]);
  var crossVec = goog.vec.Vec3.create();

  goog.vec.Vec3.cross(v0, v1, crossVec);
  assertElementsEquals([1, 2, 3], v0);
  assertElementsEquals([4, 5, 6], v1);
  assertElementsEquals([-3, 6, -3], crossVec);

  goog.vec.Vec3.setFromArray(crossVec, v1);
  goog.vec.Vec3.cross(crossVec, v0, crossVec);
  assertElementsEquals([1, 2, 3], v0);
  assertElementsEquals([4, 5, 6], v1);
  assertElementsEquals([3, -6, 3], crossVec);

  goog.vec.Vec3.cross(v0, v0, v0);
  assertElementsEquals([0, 0, 0], v0);
}

function testDistanceSquared() {
  var v0 = goog.vec.Vec3.createFloat32FromValues(1, 2, 3);
  var v1 = goog.vec.Vec3.createFloat32FromValues(1, 2, 3);
  assertEquals(0, goog.vec.Vec3.distanceSquared(v0, v1));
  goog.vec.Vec3.setFromValues(v0, 1, 2, 3);
  goog.vec.Vec3.setFromValues(v1, -1, -2, -1);
  assertEquals(36, goog.vec.Vec3.distanceSquared(v0, v1));
}

function testDistance() {
  var v0 = goog.vec.Vec3.createFloat32FromValues(1, 2, 3);
  var v1 = goog.vec.Vec3.createFloat32FromValues(1, 2, 3);
  assertEquals(0, goog.vec.Vec3.distance(v0, v1));
  goog.vec.Vec3.setFromValues(v0, 1, 2, 3);
  goog.vec.Vec3.setFromValues(v1, -1, -2, -1);
  assertEquals(6, goog.vec.Vec3.distance(v0, v1));
}

function testDirection() {
  var v0 = goog.vec.Vec3.createFloat32FromValues(1, 2, 3);
  var v1 = goog.vec.Vec3.createFloat32FromValues(1, 2, 3);
  var dirVec = goog.vec.Vec3.createFloat32FromValues(4, 5, 6);
  goog.vec.Vec3.direction(v0, v1, dirVec);
  assertElementsEquals([0, 0, 0], dirVec);
  goog.vec.Vec3.setFromValues(v0, 0, 0, 0);
  goog.vec.Vec3.setFromValues(v1, 1, 0, 0);
  goog.vec.Vec3.direction(v0, v1, dirVec);
  assertElementsEquals([1, 0, 0], dirVec);
  goog.vec.Vec3.setFromValues(v0, 1, 1, 1);
  goog.vec.Vec3.setFromValues(v1, 0, 0, 0);
  goog.vec.Vec3.direction(v0, v1, dirVec);
  assertElementsRoughlyEqual(
      [-0.5773502588272095, -0.5773502588272095, -0.5773502588272095],
      dirVec, goog.vec.EPSILON);
}

function testLerp() {
  var v0 = goog.vec.Vec3.createFloat32FromValues(1, 2, 3);
  var v1 = goog.vec.Vec3.createFloat32FromValues(10, 20, 30);
  var v2 = goog.vec.Vec3.cloneFloat32(v0);

  goog.vec.Vec3.lerp(v2, v1, 0, v2);
  assertElementsEquals([1, 2, 3], v2);
  goog.vec.Vec3.lerp(v2, v1, 1, v2);
  assertElementsEquals([10, 20, 30], v2);
  goog.vec.Vec3.lerp(v0, v1, .5, v2);
  assertElementsEquals([5.5, 11, 16.5], v2);
}

function testMax() {
  var v0 = goog.vec.Vec3.createFloat32FromValues(10, 20, 30);
  var v1 = goog.vec.Vec3.createFloat32FromValues(5, 25, 35);
  var v2 = goog.vec.Vec3.createFloat32();

  goog.vec.Vec3.max(v0, v1, v2);
  assertElementsEquals([10, 25, 35], v2);
  goog.vec.Vec3.max(v1, v0, v1);
  assertElementsEquals([10, 25, 35], v1);
  goog.vec.Vec3.max(v2, 20, v2);
  assertElementsEquals([20, 25, 35], v2);
}

function testMin() {
  var v0 = goog.vec.Vec3.createFloat32FromValues(10, 20, 30);
  var v1 = goog.vec.Vec3.createFloat32FromValues(5, 25, 35);
  var v2 = goog.vec.Vec3.createFloat32();

  goog.vec.Vec3.min(v0, v1, v2);
  assertElementsEquals([5, 20, 30], v2);
  goog.vec.Vec3.min(v1, v0, v1);
  assertElementsEquals([5, 20, 30], v1);
  goog.vec.Vec3.min(v2, 20, v2);
  assertElementsEquals([5, 20, 20], v2);
}

function testEquals() {
  var v0 = goog.vec.Vec3.createFloat32FromValues(1, 2, 3);
  var v1 = goog.vec.Vec3.cloneFloat32(v0);
  assertElementsEquals(v0, v1);

  v1[0] = 4;
  assertFalse(goog.vec.Vec3.equals(v0, v1));

  v1 = goog.vec.Vec3.cloneFloat32(v0);
  v1[1] = 4;
  assertFalse(goog.vec.Vec3.equals(v0, v1));

  v1 = goog.vec.Vec3.cloneFloat32(v0);
  v1[2] = 4;
  assertFalse(goog.vec.Vec3.equals(v0, v1));
}
