## -*-Tcl-*-
 # ###################################################################
 #	Vince's	Additions -	an extension package for Alpha
 # 
 #	FILE: "TeXCompletions.tcl"
 #					  created: 26/2/96 {2:27:17 pm} 
 #					  last update: 28/2/1999 {2:42:04 am} 
 #	Author:	Vince Darley
 #	E-mail:	<darley@fas.harvard.edu>
 #	  mail:	Division of	Applied	Sciences, Harvard University
 #			Oxford Street, Cambridge MA	02138, USA
 #	   www:	<http://www.fas.harvard.edu/~darley/>
 #	
 #  Adds completion routines for TeX mode.  This includes reference
 #  (\label) completion, citation completion, environment completion,
 #  environment item insertion.
 #	
 #  Cool new feature: the '{' key is bound to electric completion.  This
 #  means you can just type as normal in most circumstances, and when you
 #  hit '{', if the previous text is capable of being extended as a command
 #  (e.g. \begin, \frac, ...), then it is!
 # 
 #  modified by  rev reason
 #  -------- --- --- -----------
 #  18/12/97 VMD 1.1 added TeX::IncludeFile completions, better handling of '*'
 #  14/1/98  VMD 1.2 Env completion rewritten, some in core latex mode
 #  28/1/98  VMD 1.3 huge thanks to Pierre Basso for improvements.
 # ###################################################################
 ##

#  completions & expanders  #
ensureset completions(TeX) {beginContraction Env Cmd Electric Reference Cite Word}
ensureset expanders(TeX) {ExCmd}

#  Preferences  #

# Add an item inside any environment 
# bound to shift-opt-i
newPref binding TeXAddItem "/I<U<I" TeX "" 1

newPref f showTitlesWithTeXCiteCompletion 1 TeX
# complete an environment even if we don't recognise it!
newPref f acceptAnyTeXEnvironment 1 TeX
# if we don't recognise the environment, create a body on the fly
newPref f promptToCreateTeXEnvironment 1 TeX
newPref f electricContractions 1 TeX

#  extra invoker key  #

proc TeX::electricLeft {} {
    global TeXmodeVars
    set p [getPos]
    if {$TeXmodeVars(electricContractions)} {
	catch {TeX::Completion::beginContraction}
    }
    if {[pos::compare [getPos] == $p]} {
	catch {TeX::Completion::Electric}
    }
    if {[pos::compare [getPos] == $p]} {
	insertText "\{"
    }
}

proc TeX::electricRight {} {
    insertText "\}"
    catch {blink [matchIt "\}" [pos::math [getPos] - 2]]}
}

#  Completions  #

## 
 # -------------------------------------------------------------------------
 # 
 # "TeX::Completion::beginContraction" --
 # 
 #  The idea here is to see if you see a hint in the form of 
 #  "b'<some-word>", if so replace the "b'" with "\begin ".
 # -------------------------------------------------------------------------
 ##
proc TeX::Completion::beginContraction {{dummy ""}} {
    set lastword [completion::lastTwoWords leadingHint]
    if {$leadingHint != "b'"} {return 0} 
    set curPos [getPos]
    backwardWord
    set startEnvPos [getPos]
    set evironmentHint [getText $startEnvPos $curPos]
    backwardWord
    deleteText [getPos] $curPos
    insertText "\\begin"
    TeX::Completion::Electric "begin"
    typeText $lastword
    return 0
}

## 
 # -------------------------------------------------------------------------
 #	 
 # "TeX::Completion::Env" --
 #	
 #  Complete the contents of a \begin...\end pair as appropriate.  Uses the
 #  TeXbodies array.  You can either type '\begin<cmd-Tab>figure<cmd-Tab>'
 #  (for example) or just '\begin{figure}<cmd-Tab>'. 
 # -------------------------------------------------------------------------
 ##
proc TeX::Completion::Env {dummy} {
    set cmd [completion::lastTwoWords begin]
    if { $begin != "\\begin\{" } { return 0 }
    if [regexp {^(.*)\}$} $cmd "" cmd] {
	# hmm
    }
    set matches [completion::fromList $cmd TeXenvironments]
    if { $matches == "" } {
	global TeXmodeVars TeXbodies
	if [info exists TeXbodies($cmd)] {
	    set match $cmd
	} else {			
	    if {$TeXmodeVars(acceptAnyTeXEnvironment)} {
		if $TeXmodeVars(promptToCreateTeXEnvironment) {
		    set y 40
		    set yb 200
		    set res [eval dialog -w 400 -h 340 \
		      [dialog::title "New TeX environment" 400] \
		      [dialog::button "OK" 310 yb] \
		      [dialog::button "Cancel" 310 yb] \
		      [dialog::text "Enter the template for the body of the environment" 10 y] \
		      [dialog::text "Write 'prompt message' for each template stop," 10 y] \
		      [dialog::text "'\\r','\\\{','\\t',... for return, brace, tab etc." 10 y] \
		      [dialog::edit "body" 10 y 35 6] \
		      ]
		    set match $cmd
		    if {![lindex $res 1]} {
			eval set "TeXbodies($match)" [lindex $res 2]
			addUserLine "set TeXbodies($match) \"[quoteExpr3 $TeXbodies($match)]\""
		    } else {
			# we cancelled, so move on
			completion::already error
			return 1
		    }
		} else {
		    message "Warning: I don't recognise that environment"
		    set match $cmd
		}
	    } else {
		return 0
	    }
	}
    } else {
	set match [completion::Find $cmd $matches]
    }	
    if [string length $match] {
	# we completed or cancelled, so move on
	completion::already error
	if { $match == 1 } {
	    return 1
	} else {			
	    if {![ring::type]} {
		endOfLine
		set p [getPos]
		ring::+
		insertText $match
		goto $p
		return [elec::findCmd $match TeXbodies ""]
	    }
	    if {![ring::TMarkAt "environment name" [pos::math [getPos] - [string length $match]]]} {
		# we probably typed '\begin{name}' all in one go
		set i ""
		if {[lookAt [pos::math [getPos] - 1]] != "\}"} {
		    append i "\}"
		}
		append i "\r\\end\{${match}\}\r"
		elec::Insertion $i
		endOfLine
	    } else {
		endOfLine
		set p [getPos]
		# delete the stop of the \end{}
		ring::+
		ring::deleteStop
		# we need to fill in the '\end{}'
		insertText $match
		goto $p
	    }
	    set ret [elec::findCmd $match TeXbodies ""]
	    # delete the stop of \begin{}
	    # we do this afterwards, otherwise we lose the nesting of
	    # templates, which is bad.
	    ring::-
	    ring::deleteStopAndMove
	    return $ret
	}
    } else {
	completion::already TeX::Completion::Env
	return 1
    }	
}

## 
 # -------------------------------------------------------------------------
 #	 
 # "TeX::Completion::Cmd" --
 #	
 #  Takes account of the backslash which commands in TeX use
 # -------------------------------------------------------------------------
 ##
proc TeX::Completion::Cmd {dummy} {
    set cmd [completion::lastWord pos]
    if {[regexp {^\\([^\*]*)\*?$} $cmd "" cmd]} {
	return [completion::cmd $cmd]
    } else {
	return 0
    }
}

proc TeX::Completion::Insert {what} {
    insertText "\\${what}"
    bind::Completion
}

## 
 # -------------------------------------------------------------------------
 #	 
 #	"TeX::Completion::Electric"	--
 #	
 #  An example of calling the completion::electric procedure.  In TeX mode,
 #  '{}' is a good default. 
 # -------------------------------------------------------------------------
 ##
proc TeX::Completion::Electric { {cmd ""} } {
    if {![string length $cmd]} { 
	if {[containsSpace $cmd]} { return 0 }
	set cmd [completion::lastWord]
    }
    if {[regexp {^\\([^\*]*)\*?$} $cmd "" cmd]} {
	# nothing
    } elseif {[regexp {\]\{?$} $cmd got]} {
	# we might have an optional [...] after the command we really want.
	# This should work but doesn't (Alpha bug)!
	#{matchIt "]" [pos::math [getPos] - [expr 1 + [string length $got]]]}
	if {![catch {search -s -f 0 -r 0 -m 0 "\[" [getPos]} where]} {
	    set p [getPos]
	    goto [lindex $where 0]
	    set cmd [completion::lastWord]
	    goto $p
	    regexp {^\\([^\*]*)\*?$} $cmd "" cmd
	}
    }
    return [completion::electric $cmd "\{\}"]
}

	
## 
 # -------------------------------------------------------------------------
 #	 
 # "TeX::Completion::Reference"	--
 #	
 #  If we're in any kind of reference, search for appropriate labels to get
 #  the information from and fill them in.  'TeXRefCompletion' in
 #  "latexEngine.tcl" is obsolete. 
 # -------------------------------------------------------------------------
 ##
proc TeX::Completion::Reference { {dummy ""} } {
    global completion::in_progress_pos completion_got completion_looking
    global _texrefprefixes
    # cursor changed place?
    set pos [getPos]
    if {[pos::compare $pos == ${completion::in_progress_pos}]} {
	completion::update Reference $completion_got $completion_looking
	message "press <Cmd Tab> for another label"
	return 1
    }
    
    global TeXmodeVars
    
    set lastword [completion::lastTwoWords prevword]
    set gotprefix ""
    set prevword [string trim [string range $prevword 1 end] "\{"]
    if {[set ref [lsearch -exact $TeXmodeVars(refCommands) $prevword]] != -1} {
	set gotprefix $lastword
	set lastword $prevword
    } else {
	# trim the backslash and opening brace:
	set lastword [string trim [string range $lastword 1 end] "\{"]
	# check if it's a \ref-like command:
	set ref [lsearch -exact $TeXmodeVars(refCommands) $lastword]
    }	
    
    if { $ref != -1 } {
	# got a \ref-like command:
	set completion_got "\\[lindex $TeXmodeVars(refCommands) $ref]\{${gotprefix}"
	# make sure we have the brace:
	if { $gotprefix == "" && [lookAt [pos::math $pos - 1]] != "\{" } {
	    insertText "\{"
	}
	set completion_looking "label\{${gotprefix}"
	TeX::updateCompletion Reference $completion_got $completion_looking
	completion::already Reference
	return 1
    } else {
	return 0 
    }
}

## 
 # -------------------------------------------------------------------------
 #	 
 # "TeX::Completion::Cite" --
 #	
 #  Checks for any \cite like command, and looks up the partial argument in
 #  the known bibliographies, completing the entry as appropriate. 
 # -------------------------------------------------------------------------
 ##
proc TeX::Completion::Cite { {dummy ""} } {
    global completion::in_progress_pos completion_got completion_looking TeXmodeVars
    # cursor changed place?
    if {[pos::compare [getPos] == ${completion::in_progress_pos}]} {
	set lastword [completion::lastWord]
	set completion_got [completion::fromList $lastword completion_got]
    } else {
	global TeXmodeVars
	set a [getText [lineStart [getPos]] [getPos]]
	# got a \cite-like command:
	if ![regexp "\\\\([join [string trim $TeXmodeVars(citeCommands)] |])\\\{(\[a-zA-Z0-9\]+,)*(\[a-zA-Z0-9\]+)$" $a d d d lastword] {
	    return 0
	}
	set completion_got [Bib::_FindAllEntries $lastword \
	  $TeXmodeVars(showTitlesWithTeXCiteCompletion)]
	if {$completion_got == ""} {
	    if [catch {dialog::optionMenu \
	      "No matching citations found.  Perhaps you should\
	      rebuild your bib data-base, or create a new entry." \
	      [list "Rebuild database" "New entry" "New entry in new file"]} res] {
		# user cancelled
		return 0
	    }
	    switch $res {
		"Rebuild database" {
		    if $TeXmodeVars(showTitlesWithTeXCiteCompletion) {
			Bib::rebuildDatabase
		    } else {
			Bib::rebuildIndex
		    }
		    # try again
		    return [TeX::Completion::Cite $dummy]
		}
		"New entry" {
		    Bib::_newEntry $lastword
		}
		"New entry in new file" {
		    Bib::_newEntry $lastword 1
		}
	    }
	    return 0
	}
    }
    if $TeXmodeVars(showTitlesWithTeXCiteCompletion) {
	set query "Rebuild Bibliography Database"
	set rebuild Bib::rebuildDatabase
    } else {
	set query "Rebuild Bibliography Index"
	set rebuild Bib::rebuildIndex
    }
    set match [completion::Find $lastword $completion_got \
      $TeXmodeVars(showTitlesWithTeXCiteCompletion) 1 $query $rebuild]
    if {$match != ""} {
	if {[lookAt [getPos]] != "\}"} { insertText "\}" }
    }
    # we never bother with calling ourselves again, since we forced the above
    # 'completion::Find' call to complete.
    completion::already error
    return 1
}

proc TeX::Completion::Word {dummy} {
    # we only complete the word if it doesn't end in some command
    if {[lookAt [pos::math [getPos] - 1]] != "\{" } {
	return [completion::word $dummy]
    }
}

#  helpers  #

proc TeX::updateCompletion { proc {got ""} {looking ""} } {
    if [completion::general $got $looking] {
	if {[lookAt [getPos]] != "\}"} {
	    insertText "\}"
	} 
	completion::already $proc
	message "press <Cmd Tab> for another label"
	return 1
    } else {
	completion::already error
	# if {[lookAt [expr [getPos] - 1]] != "\}"} {
	#     elec::Insertion "\}"		
	# } 
	error ""
	return 0
    }
}	

#  setup various arrays for electrics  #
uplevel \#0 [list source [file join ${HOME} Tcl Completions TeXcmds.tcl]]

hook::register TeX::labelDelimChanged TeX::adjustElectricLabels
proc TeX::adjustElectricLabels {args} {
uplevel \#0 {
set _x $TeXmodeVars(standardTeXLabelDelimiter)
set TeXelectrics(*section) "\{section name\}\n"
set TeXelectrics(frac) "\{numerator\}\{denominator\}"
set TeXelectrics(sum) "_\{from\}^\{to\}"
set TeXelectrics(emph) "1"
set TeXelectrics(includegraphics) "\[TeX::IncludeFile\]"
set TeXelectrics(begin) "\{environment name\}\n\\end\{\}\n"
set TeXelectrics(Sec*) "kill0Section~\\ref\{sec${_x}label\}"
set TeXelectrics(Table) "~\\ref\{tab${_x}label\}"
set TeXelectrics(App*) "kill0Appendix~\\ref\{sec${_x}label\}"
set TeXelectrics(Eq.) "~\\eqref\{eq${_x}label\}"
set TeXelectrics(Fig*) "kill0Figure~\\ref\{fig${_x}label\}"
set TeXelectrics(Cha*) "kill0Chapter~\\ref\{chap${_x}label\}"
set TeXelectrics(mbox) "\{\}"
set TeXelectrics(fbox) "\{\}"
set TeXelectrics(mbox) "\{\}"
set TeXelectrics(parbox) "\[TeX::parbox\]"
set TeXelectrics(makebox) "\[TeX::boxes\]"
set TeXelectrics(framebox) "\[TeX::boxes\]"
set TeXelectrics(raisebox) "\[TeX::raisebox\]"
set TeXelectrics(newsavebox) "\{\}"
set TeXelectrics(usebox) "\{\}"
set TeXelectrics(sbox) "\[TeX::sbox\]"
set TeXelectrics(savebox) "\[TeX::savebox\]"
set TeXelectrics(rule) "\[TeX::rule\]"
#set TeXelectrics(subfigure) "\[caption\]\{\\label\{fig${_x}\}\}\%\r\\includegraphics\[width=,height=\]\{eps file\}\}"
set TeXbodies(array) "\[TeX::BuildTabular array\]"
set TeXbodies(equation) "\n\tequation body\n\t\\label\{eq${_x}label\}"
set TeXbodies(description) "\[TeX::BuildList description\]"
set TeXbodies(enumerate) "\[TeX::BuildList enumerate\]"
set TeXbodies(itemize) "\[TeX::BuildList itemize\]"
set TeXbodies(list) "\[TeX::BuildList list\]"
set TeXbodies(trivlist) "\[TeX::BuildList trivlist\]"
set TeXbodies(figure) "\[TeX::Figure\]"
set TeXbodies(table) "\n\t\n\t\\caption\[short title for t.o.t.\]\{caption\}\n\t\\protect[TeX::label tab]"

set TeXbodies(tabular) "\[TeX::BuildTabular tabular\]"
set TeXbodies(tabular*) "\[TeX::BuildTabular tabular*\]"
set TeXbodies(gather) "\n\t \n\t\\label\{eq${_x}\} \\\\\n\t \n\t\\label\{eq${_x}\}"
set TeXbodies(split) "\n\t & \\\\\n\t & \\\\"
set TeXbodies(cases) "\n\t &  \\\\\n\t & "

set TeXbodies(equationarray) "\[TeX::equationarray\]"
set TeXbodies(minipage) "\[TeX::minipage\]"
set TeXbodies(matrix) "\[TeX::matrix\]"
set TeXbodies(pmatrix) "\[TeX::matrix\]"
set TeXbodies(bmatrix) "\[TeX::matrix\]"
set TeXbodies(vmatrix) "\[TeX::matrix\]"
set TeXbodies(Vmatrix) "\[TeX::matrix\]"
set TeXbodies(align) "\[TeX::align\]"
set TeXbodies(alignat) "\[TeX::alignat\]"

set TeXbodies(align) "\n\tequation 1 l.h.s. & \n\t\\label\{eq${_x}\} \\\\\n\tequation 2 l.h.s. & \n\t\\label\{eq${_x}\}"

set TeXEnvItems(enumerate) "\n\\item "
set TeXEnvItems(itemize) "\n\\item "
set TeXEnvItems(description) "\n\\item\[name\] description"
set TeXEnvItems(align) "\\\\\nnext equation l.h.s. & \n\\label\{eq${_x}\} "
set TeXEnvItems(gather) "\\\\\n \n\\label\{eq${_x}\} "
set TeXEnvItems(split) " & \\\\"
set TeXEnvItems(cases) " & \\\\"
set _texrefprefixes [list fig${_x} eq${_x} sec${_x} chap${_x} tab${_x} ]
unset _x
}

}

# call it now
TeX::adjustElectricLabels


#  environment assistors  #

## 
 # -------------------------------------------------------------------------
 #	 
 # "TeXAddItem"	--
 #	
 #  Scan the local environment and insert a new item into that environment,
 #  of the appropriate type.
 #	
 #  Currently not too sophisticated.
 # -------------------------------------------------------------------------
 ##
proc TeXAddItem {} {
    set env [lindex [split [eval getText [searchEnvironment]] "{}"] 1]
    global TeXEnvItems
    if ![catch {set item $TeXEnvItems($env)}] {
	elec::Insertion $item
    }
}

#  Template embeddable proc's  #

proc TeX::IncludeFile {} {
    # could try to ensure this file's on the search path?
    if ![regexp {\{, } [lookAt [pos::math [getPos] - 1]]] {
	append res "\{"
    }
    append res [file tail [getfile "Name of file to include:"]]
    return $res
}
    


#  Expansions  #
namespace eval TeX::Expansion {}

# proc by Tom Fetherston
proc TeX::Expansion::ExCmd { {cmd ""} {dictExt "acronyms"}} {
    if ![string length $cmd] { 
	set cmd [completion::lastWord]
	# if there's any whitespace in the command then it's no good to us
	if [containsSpace $cmd] { return 0 }
    }
    
    set m [modeALike]
    set hint [string trim [join [split $cmd \\ ]]]
    
    if { [set matches [elec::acronymListExpansions $hint ${m}${dictExt}]] == 0 } {
	return 0
    } else {
	set result [elec::expandThis $cmd $matches]
	set match [lindex  $result 0]
	catch {set keystroke [lindex $result 1]}
	if [string length $match] {
	    # we completed or cancelled, so move on
	    # WHY ISN'T THIS 'alreadyExpanding' ???????????? vmd
	    completion::already error
	    if { $match == 1 } {
		return 1
	    } else {
		set curPos [getPos]
		set retVal [completion [modeALike] Electric "${match}"]
		if {([pos::compare [getPos] == $curPos]) && [info exists keystroke]} {
		    insertText $keystroke
		} 
		return $retVal
	    }
	} else {
	    elec::alreadyExpanding Cmd
	    return 1
	}
    }
    
}

