﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsitewise/IoTSiteWiseRequest.h>
#include <aws/iotsitewise/IoTSiteWise_EXPORTS.h>
#include <aws/iotsitewise/model/DatasetSource.h>

#include <utility>

namespace Aws {
namespace IoTSiteWise {
namespace Model {

/**
 */
class UpdateDatasetRequest : public IoTSiteWiseRequest {
 public:
  AWS_IOTSITEWISE_API UpdateDatasetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateDataset"; }

  AWS_IOTSITEWISE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the dataset.</p>
   */
  inline const Aws::String& GetDatasetId() const { return m_datasetId; }
  inline bool DatasetIdHasBeenSet() const { return m_datasetIdHasBeenSet; }
  template <typename DatasetIdT = Aws::String>
  void SetDatasetId(DatasetIdT&& value) {
    m_datasetIdHasBeenSet = true;
    m_datasetId = std::forward<DatasetIdT>(value);
  }
  template <typename DatasetIdT = Aws::String>
  UpdateDatasetRequest& WithDatasetId(DatasetIdT&& value) {
    SetDatasetId(std::forward<DatasetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the dataset.</p>
   */
  inline const Aws::String& GetDatasetName() const { return m_datasetName; }
  inline bool DatasetNameHasBeenSet() const { return m_datasetNameHasBeenSet; }
  template <typename DatasetNameT = Aws::String>
  void SetDatasetName(DatasetNameT&& value) {
    m_datasetNameHasBeenSet = true;
    m_datasetName = std::forward<DatasetNameT>(value);
  }
  template <typename DatasetNameT = Aws::String>
  UpdateDatasetRequest& WithDatasetName(DatasetNameT&& value) {
    SetDatasetName(std::forward<DatasetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description about the dataset, and its functionality.</p>
   */
  inline const Aws::String& GetDatasetDescription() const { return m_datasetDescription; }
  inline bool DatasetDescriptionHasBeenSet() const { return m_datasetDescriptionHasBeenSet; }
  template <typename DatasetDescriptionT = Aws::String>
  void SetDatasetDescription(DatasetDescriptionT&& value) {
    m_datasetDescriptionHasBeenSet = true;
    m_datasetDescription = std::forward<DatasetDescriptionT>(value);
  }
  template <typename DatasetDescriptionT = Aws::String>
  UpdateDatasetRequest& WithDatasetDescription(DatasetDescriptionT&& value) {
    SetDatasetDescription(std::forward<DatasetDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data source for the dataset.</p>
   */
  inline const DatasetSource& GetDatasetSource() const { return m_datasetSource; }
  inline bool DatasetSourceHasBeenSet() const { return m_datasetSourceHasBeenSet; }
  template <typename DatasetSourceT = DatasetSource>
  void SetDatasetSource(DatasetSourceT&& value) {
    m_datasetSourceHasBeenSet = true;
    m_datasetSource = std::forward<DatasetSourceT>(value);
  }
  template <typename DatasetSourceT = DatasetSource>
  UpdateDatasetRequest& WithDatasetSource(DatasetSourceT&& value) {
    SetDatasetSource(std::forward<DatasetSourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique case-sensitive identifier that you can provide to ensure the
   * idempotency of the request. Don't reuse this client token if a new idempotent
   * request is required.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  UpdateDatasetRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_datasetId;
  bool m_datasetIdHasBeenSet = false;

  Aws::String m_datasetName;
  bool m_datasetNameHasBeenSet = false;

  Aws::String m_datasetDescription;
  bool m_datasetDescriptionHasBeenSet = false;

  DatasetSource m_datasetSource;
  bool m_datasetSourceHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace IoTSiteWise
}  // namespace Aws
