﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticbeanstalk/ElasticBeanstalkRequest.h>
#include <aws/elasticbeanstalk/ElasticBeanstalk_EXPORTS.h>
#include <aws/elasticbeanstalk/model/ConfigurationOptionSetting.h>
#include <aws/elasticbeanstalk/model/EnvironmentTier.h>
#include <aws/elasticbeanstalk/model/OptionSpecification.h>

#include <utility>

namespace Aws {
namespace ElasticBeanstalk {
namespace Model {

/**
 * <p>Request to update an environment.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticbeanstalk-2010-12-01/UpdateEnvironmentMessage">AWS
 * API Reference</a></p>
 */
class UpdateEnvironmentRequest : public ElasticBeanstalkRequest {
 public:
  AWS_ELASTICBEANSTALK_API UpdateEnvironmentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateEnvironment"; }

  AWS_ELASTICBEANSTALK_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICBEANSTALK_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the application with which the environment is associated.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  UpdateEnvironmentRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the environment to update.</p> <p>If no environment with this ID
   * exists, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code>
   * error.</p> <p>Condition: You must specify either this or an EnvironmentName, or
   * both. If you do not specify either, AWS Elastic Beanstalk returns
   * <code>MissingRequiredParameter</code> error. </p>
   */
  inline const Aws::String& GetEnvironmentId() const { return m_environmentId; }
  inline bool EnvironmentIdHasBeenSet() const { return m_environmentIdHasBeenSet; }
  template <typename EnvironmentIdT = Aws::String>
  void SetEnvironmentId(EnvironmentIdT&& value) {
    m_environmentIdHasBeenSet = true;
    m_environmentId = std::forward<EnvironmentIdT>(value);
  }
  template <typename EnvironmentIdT = Aws::String>
  UpdateEnvironmentRequest& WithEnvironmentId(EnvironmentIdT&& value) {
    SetEnvironmentId(std::forward<EnvironmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the environment to update. If no environment with this name
   * exists, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code>
   * error. </p> <p>Condition: You must specify either this or an EnvironmentId, or
   * both. If you do not specify either, AWS Elastic Beanstalk returns
   * <code>MissingRequiredParameter</code> error. </p>
   */
  inline const Aws::String& GetEnvironmentName() const { return m_environmentName; }
  inline bool EnvironmentNameHasBeenSet() const { return m_environmentNameHasBeenSet; }
  template <typename EnvironmentNameT = Aws::String>
  void SetEnvironmentName(EnvironmentNameT&& value) {
    m_environmentNameHasBeenSet = true;
    m_environmentName = std::forward<EnvironmentNameT>(value);
  }
  template <typename EnvironmentNameT = Aws::String>
  UpdateEnvironmentRequest& WithEnvironmentName(EnvironmentNameT&& value) {
    SetEnvironmentName(std::forward<EnvironmentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the group to which the target environment belongs. Specify a
   * group name only if the environment's name is specified in an environment
   * manifest and not with the environment name or environment ID parameters. See <a
   * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/environment-cfg-manifest.html">Environment
   * Manifest (env.yaml)</a> for details.</p>
   */
  inline const Aws::String& GetGroupName() const { return m_groupName; }
  inline bool GroupNameHasBeenSet() const { return m_groupNameHasBeenSet; }
  template <typename GroupNameT = Aws::String>
  void SetGroupName(GroupNameT&& value) {
    m_groupNameHasBeenSet = true;
    m_groupName = std::forward<GroupNameT>(value);
  }
  template <typename GroupNameT = Aws::String>
  UpdateEnvironmentRequest& WithGroupName(GroupNameT&& value) {
    SetGroupName(std::forward<GroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If this parameter is specified, AWS Elastic Beanstalk updates the description
   * of this environment.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateEnvironmentRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This specifies the tier to use to update the environment.</p> <p>Condition:
   * At this time, if you change the tier version, name, or type, AWS Elastic
   * Beanstalk returns <code>InvalidParameterValue</code> error. </p>
   */
  inline const EnvironmentTier& GetTier() const { return m_tier; }
  inline bool TierHasBeenSet() const { return m_tierHasBeenSet; }
  template <typename TierT = EnvironmentTier>
  void SetTier(TierT&& value) {
    m_tierHasBeenSet = true;
    m_tier = std::forward<TierT>(value);
  }
  template <typename TierT = EnvironmentTier>
  UpdateEnvironmentRequest& WithTier(TierT&& value) {
    SetTier(std::forward<TierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If this parameter is specified, AWS Elastic Beanstalk deploys the named
   * application version to the environment. If no such application version is found,
   * returns an <code>InvalidParameterValue</code> error. </p>
   */
  inline const Aws::String& GetVersionLabel() const { return m_versionLabel; }
  inline bool VersionLabelHasBeenSet() const { return m_versionLabelHasBeenSet; }
  template <typename VersionLabelT = Aws::String>
  void SetVersionLabel(VersionLabelT&& value) {
    m_versionLabelHasBeenSet = true;
    m_versionLabel = std::forward<VersionLabelT>(value);
  }
  template <typename VersionLabelT = Aws::String>
  UpdateEnvironmentRequest& WithVersionLabel(VersionLabelT&& value) {
    SetVersionLabel(std::forward<VersionLabelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If this parameter is specified, AWS Elastic Beanstalk deploys this
   * configuration template to the environment. If no such configuration template is
   * found, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code>
   * error. </p>
   */
  inline const Aws::String& GetTemplateName() const { return m_templateName; }
  inline bool TemplateNameHasBeenSet() const { return m_templateNameHasBeenSet; }
  template <typename TemplateNameT = Aws::String>
  void SetTemplateName(TemplateNameT&& value) {
    m_templateNameHasBeenSet = true;
    m_templateName = std::forward<TemplateNameT>(value);
  }
  template <typename TemplateNameT = Aws::String>
  UpdateEnvironmentRequest& WithTemplateName(TemplateNameT&& value) {
    SetTemplateName(std::forward<TemplateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This specifies the platform version that the environment will run after the
   * environment is updated.</p>
   */
  inline const Aws::String& GetSolutionStackName() const { return m_solutionStackName; }
  inline bool SolutionStackNameHasBeenSet() const { return m_solutionStackNameHasBeenSet; }
  template <typename SolutionStackNameT = Aws::String>
  void SetSolutionStackName(SolutionStackNameT&& value) {
    m_solutionStackNameHasBeenSet = true;
    m_solutionStackName = std::forward<SolutionStackNameT>(value);
  }
  template <typename SolutionStackNameT = Aws::String>
  UpdateEnvironmentRequest& WithSolutionStackName(SolutionStackNameT&& value) {
    SetSolutionStackName(std::forward<SolutionStackNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the platform, if used.</p>
   */
  inline const Aws::String& GetPlatformArn() const { return m_platformArn; }
  inline bool PlatformArnHasBeenSet() const { return m_platformArnHasBeenSet; }
  template <typename PlatformArnT = Aws::String>
  void SetPlatformArn(PlatformArnT&& value) {
    m_platformArnHasBeenSet = true;
    m_platformArn = std::forward<PlatformArnT>(value);
  }
  template <typename PlatformArnT = Aws::String>
  UpdateEnvironmentRequest& WithPlatformArn(PlatformArnT&& value) {
    SetPlatformArn(std::forward<PlatformArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If specified, AWS Elastic Beanstalk updates the configuration set associated
   * with the running environment and sets the specified configuration options to the
   * requested value.</p>
   */
  inline const Aws::Vector<ConfigurationOptionSetting>& GetOptionSettings() const { return m_optionSettings; }
  inline bool OptionSettingsHasBeenSet() const { return m_optionSettingsHasBeenSet; }
  template <typename OptionSettingsT = Aws::Vector<ConfigurationOptionSetting>>
  void SetOptionSettings(OptionSettingsT&& value) {
    m_optionSettingsHasBeenSet = true;
    m_optionSettings = std::forward<OptionSettingsT>(value);
  }
  template <typename OptionSettingsT = Aws::Vector<ConfigurationOptionSetting>>
  UpdateEnvironmentRequest& WithOptionSettings(OptionSettingsT&& value) {
    SetOptionSettings(std::forward<OptionSettingsT>(value));
    return *this;
  }
  template <typename OptionSettingsT = ConfigurationOptionSetting>
  UpdateEnvironmentRequest& AddOptionSettings(OptionSettingsT&& value) {
    m_optionSettingsHasBeenSet = true;
    m_optionSettings.emplace_back(std::forward<OptionSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of custom user-defined configuration options to remove from the
   * configuration set for this environment.</p>
   */
  inline const Aws::Vector<OptionSpecification>& GetOptionsToRemove() const { return m_optionsToRemove; }
  inline bool OptionsToRemoveHasBeenSet() const { return m_optionsToRemoveHasBeenSet; }
  template <typename OptionsToRemoveT = Aws::Vector<OptionSpecification>>
  void SetOptionsToRemove(OptionsToRemoveT&& value) {
    m_optionsToRemoveHasBeenSet = true;
    m_optionsToRemove = std::forward<OptionsToRemoveT>(value);
  }
  template <typename OptionsToRemoveT = Aws::Vector<OptionSpecification>>
  UpdateEnvironmentRequest& WithOptionsToRemove(OptionsToRemoveT&& value) {
    SetOptionsToRemove(std::forward<OptionsToRemoveT>(value));
    return *this;
  }
  template <typename OptionsToRemoveT = OptionSpecification>
  UpdateEnvironmentRequest& AddOptionsToRemove(OptionsToRemoveT&& value) {
    m_optionsToRemoveHasBeenSet = true;
    m_optionsToRemove.emplace_back(std::forward<OptionsToRemoveT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;
  bool m_applicationNameHasBeenSet = false;

  Aws::String m_environmentId;
  bool m_environmentIdHasBeenSet = false;

  Aws::String m_environmentName;
  bool m_environmentNameHasBeenSet = false;

  Aws::String m_groupName;
  bool m_groupNameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  EnvironmentTier m_tier;
  bool m_tierHasBeenSet = false;

  Aws::String m_versionLabel;
  bool m_versionLabelHasBeenSet = false;

  Aws::String m_templateName;
  bool m_templateNameHasBeenSet = false;

  Aws::String m_solutionStackName;
  bool m_solutionStackNameHasBeenSet = false;

  Aws::String m_platformArn;
  bool m_platformArnHasBeenSet = false;

  Aws::Vector<ConfigurationOptionSetting> m_optionSettings;
  bool m_optionSettingsHasBeenSet = false;

  Aws::Vector<OptionSpecification> m_optionsToRemove;
  bool m_optionsToRemoveHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticBeanstalk
}  // namespace Aws
