# frozen_string_literal: true

require 'labkit/covered_experience/current'
require 'labkit/covered_experience/error'
require 'labkit/covered_experience/experience'
require 'labkit/covered_experience/null'
require 'labkit/covered_experience/registry'
require 'labkit/logging/json_logger'

module Labkit
  # Labkit::CoveredExperience namespace module.
  #
  # This module is responsible for managing covered experiences, which are
  # specific events or activities within the application that are measured
  # and reported for performance monitoring and analysis.
  module CoveredExperience
    # Configuration class for CoveredExperience
    class Configuration
      attr_accessor :logger, :registry_path

      def initialize
        @logger = Labkit::Logging::JsonLogger.new($stdout)
        @registry_path = File.join("config", "covered_experiences")
      end
    end

    class << self
      def configuration
        @configuration ||= Configuration.new
      end

      def reset_configuration
        @configuration = nil
      end

      def configure
        yield(configuration) if block_given?
        # Reset registry when configuration changes to pick up new registry_path
        @registry = nil
      end

      def registry
        @registry ||= Registry.new(dir: configuration.registry_path)
      end

      def reset
        @registry = nil
        reset_configuration
      end

      # Retrieves a covered experience using the experience_id.
      # It retrieves from the current context when available,
      # otherwise it instantiates a new experience with the definition
      # from the registry.
      #
      # @param experience_id [String, Symbol] The ID of the experience to retrieve.
      # @return [Experience, Null] The found experience or a Null object if not found (in production/staging).
      def get(experience_id)
        find_current(experience_id) || raise_or_null(experience_id)
      end

      # Starts a covered experience using the experience_id.
      #
      # @param experience_id [String, Symbol] The ID of the experience to start.
      # @param extra [Hash] Additional data to include in the log event.
      # @return [Experience, Null] The started experience or a Null object if not found (in production/staging).
      def start(experience_id, **extra, &)
        get(experience_id).start(**extra, &)
      end

      # Resumes a covered experience using the experience_id.
      #
      # @param experience_id [String, Symbol] The ID of the experience to resume.
      # @return [Experience, Null] The started experience or a Null object if not found (in production/staging).
      def resume(experience_id, **extra, &)
        get(experience_id).resume(**extra, &)
      end

      private

      def raise_or_null(experience_id)
        return Null.instance unless %w[development test].include?(ENV['RAILS_ENV'])

        raise(NotFoundError, "Covered Experience #{experience_id} not found in the registry")
      end

      def find_current(experience_id)
        xp = Current.active_experiences[experience_id.to_s]
        return xp unless xp.nil?

        definition = registry[experience_id]
        Experience.new(definition) if definition
      end
    end
  end
end
