﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/ExportSourceType.h>
#include <aws/rds/model/ResponseMetadata.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Xml {
class XmlDocument;
}  // namespace Xml
}  // namespace Utils
namespace RDS {
namespace Model {
/**
 * <p>Contains the details of a snapshot or cluster export to Amazon S3.</p>
 * <p>This data type is used as a response element in the
 * <code>DescribeExportTasks</code> operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/ExportTask">AWS API
 * Reference</a></p>
 */
class StartExportTaskResult {
 public:
  AWS_RDS_API StartExportTaskResult() = default;
  AWS_RDS_API StartExportTaskResult(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);
  AWS_RDS_API StartExportTaskResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);

  ///@{
  /**
   * <p>A unique identifier for the snapshot or cluster export task. This ID isn't an
   * identifier for the Amazon S3 bucket where the data is exported.</p>
   */
  inline const Aws::String& GetExportTaskIdentifier() const { return m_exportTaskIdentifier; }
  template <typename ExportTaskIdentifierT = Aws::String>
  void SetExportTaskIdentifier(ExportTaskIdentifierT&& value) {
    m_exportTaskIdentifierHasBeenSet = true;
    m_exportTaskIdentifier = std::forward<ExportTaskIdentifierT>(value);
  }
  template <typename ExportTaskIdentifierT = Aws::String>
  StartExportTaskResult& WithExportTaskIdentifier(ExportTaskIdentifierT&& value) {
    SetExportTaskIdentifier(std::forward<ExportTaskIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the snapshot or cluster exported to Amazon
   * S3.</p>
   */
  inline const Aws::String& GetSourceArn() const { return m_sourceArn; }
  template <typename SourceArnT = Aws::String>
  void SetSourceArn(SourceArnT&& value) {
    m_sourceArnHasBeenSet = true;
    m_sourceArn = std::forward<SourceArnT>(value);
  }
  template <typename SourceArnT = Aws::String>
  StartExportTaskResult& WithSourceArn(SourceArnT&& value) {
    SetSourceArn(std::forward<SourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data exported from the snapshot or cluster.</p> <p>Valid Values:</p> <ul>
   * <li> <p> <code>database</code> - Export all the data from a specified
   * database.</p> </li> <li> <p> <code>database.table</code> <i>table-name</i> -
   * Export a table of the snapshot or cluster. This format is valid only for RDS for
   * MySQL, RDS for MariaDB, and Aurora MySQL.</p> </li> <li> <p>
   * <code>database.schema</code> <i>schema-name</i> - Export a database schema of
   * the snapshot or cluster. This format is valid only for RDS for PostgreSQL and
   * Aurora PostgreSQL.</p> </li> <li> <p> <code>database.schema.table</code>
   * <i>table-name</i> - Export a table of the database schema. This format is valid
   * only for RDS for PostgreSQL and Aurora PostgreSQL.</p> </li> </ul>
   */
  inline const Aws::Vector<Aws::String>& GetExportOnly() const { return m_exportOnly; }
  template <typename ExportOnlyT = Aws::Vector<Aws::String>>
  void SetExportOnly(ExportOnlyT&& value) {
    m_exportOnlyHasBeenSet = true;
    m_exportOnly = std::forward<ExportOnlyT>(value);
  }
  template <typename ExportOnlyT = Aws::Vector<Aws::String>>
  StartExportTaskResult& WithExportOnly(ExportOnlyT&& value) {
    SetExportOnly(std::forward<ExportOnlyT>(value));
    return *this;
  }
  template <typename ExportOnlyT = Aws::String>
  StartExportTaskResult& AddExportOnly(ExportOnlyT&& value) {
    m_exportOnlyHasBeenSet = true;
    m_exportOnly.emplace_back(std::forward<ExportOnlyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time when the snapshot was created.</p>
   */
  inline const Aws::Utils::DateTime& GetSnapshotTime() const { return m_snapshotTime; }
  template <typename SnapshotTimeT = Aws::Utils::DateTime>
  void SetSnapshotTime(SnapshotTimeT&& value) {
    m_snapshotTimeHasBeenSet = true;
    m_snapshotTime = std::forward<SnapshotTimeT>(value);
  }
  template <typename SnapshotTimeT = Aws::Utils::DateTime>
  StartExportTaskResult& WithSnapshotTime(SnapshotTimeT&& value) {
    SetSnapshotTime(std::forward<SnapshotTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time when the snapshot or cluster export task started.</p>
   */
  inline const Aws::Utils::DateTime& GetTaskStartTime() const { return m_taskStartTime; }
  template <typename TaskStartTimeT = Aws::Utils::DateTime>
  void SetTaskStartTime(TaskStartTimeT&& value) {
    m_taskStartTimeHasBeenSet = true;
    m_taskStartTime = std::forward<TaskStartTimeT>(value);
  }
  template <typename TaskStartTimeT = Aws::Utils::DateTime>
  StartExportTaskResult& WithTaskStartTime(TaskStartTimeT&& value) {
    SetTaskStartTime(std::forward<TaskStartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time when the snapshot or cluster export task ended.</p>
   */
  inline const Aws::Utils::DateTime& GetTaskEndTime() const { return m_taskEndTime; }
  template <typename TaskEndTimeT = Aws::Utils::DateTime>
  void SetTaskEndTime(TaskEndTimeT&& value) {
    m_taskEndTimeHasBeenSet = true;
    m_taskEndTime = std::forward<TaskEndTimeT>(value);
  }
  template <typename TaskEndTimeT = Aws::Utils::DateTime>
  StartExportTaskResult& WithTaskEndTime(TaskEndTimeT&& value) {
    SetTaskEndTime(std::forward<TaskEndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 bucket where the snapshot or cluster is exported to.</p>
   */
  inline const Aws::String& GetS3Bucket() const { return m_s3Bucket; }
  template <typename S3BucketT = Aws::String>
  void SetS3Bucket(S3BucketT&& value) {
    m_s3BucketHasBeenSet = true;
    m_s3Bucket = std::forward<S3BucketT>(value);
  }
  template <typename S3BucketT = Aws::String>
  StartExportTaskResult& WithS3Bucket(S3BucketT&& value) {
    SetS3Bucket(std::forward<S3BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 bucket prefix that is the file name and path of the exported
   * data.</p>
   */
  inline const Aws::String& GetS3Prefix() const { return m_s3Prefix; }
  template <typename S3PrefixT = Aws::String>
  void SetS3Prefix(S3PrefixT&& value) {
    m_s3PrefixHasBeenSet = true;
    m_s3Prefix = std::forward<S3PrefixT>(value);
  }
  template <typename S3PrefixT = Aws::String>
  StartExportTaskResult& WithS3Prefix(S3PrefixT&& value) {
    SetS3Prefix(std::forward<S3PrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the IAM role that is used to write to Amazon S3 when exporting a
   * snapshot or cluster.</p>
   */
  inline const Aws::String& GetIamRoleArn() const { return m_iamRoleArn; }
  template <typename IamRoleArnT = Aws::String>
  void SetIamRoleArn(IamRoleArnT&& value) {
    m_iamRoleArnHasBeenSet = true;
    m_iamRoleArn = std::forward<IamRoleArnT>(value);
  }
  template <typename IamRoleArnT = Aws::String>
  StartExportTaskResult& WithIamRoleArn(IamRoleArnT&& value) {
    SetIamRoleArn(std::forward<IamRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key identifier of the Amazon Web Services KMS key that is used to encrypt
   * the data when it's exported to Amazon S3. The KMS key identifier is its key ARN,
   * key ID, alias ARN, or alias name. The IAM role used for the export must have
   * encryption and decryption permissions to use this KMS key.</p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  StartExportTaskResult& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The progress status of the export task. The status can be one of the
   * following:</p> <ul> <li> <p> <code>CANCELED</code> </p> </li> <li> <p>
   * <code>CANCELING</code> </p> </li> <li> <p> <code>COMPLETE</code> </p> </li> <li>
   * <p> <code>FAILED</code> </p> </li> <li> <p> <code>IN_PROGRESS</code> </p> </li>
   * <li> <p> <code>STARTING</code> </p> </li> </ul>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  StartExportTaskResult& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The progress of the snapshot or cluster export task as a percentage.</p>
   */
  inline int GetPercentProgress() const { return m_percentProgress; }
  inline void SetPercentProgress(int value) {
    m_percentProgressHasBeenSet = true;
    m_percentProgress = value;
  }
  inline StartExportTaskResult& WithPercentProgress(int value) {
    SetPercentProgress(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The total amount of data exported, in gigabytes.</p>
   */
  inline int GetTotalExtractedDataInGB() const { return m_totalExtractedDataInGB; }
  inline void SetTotalExtractedDataInGB(int value) {
    m_totalExtractedDataInGBHasBeenSet = true;
    m_totalExtractedDataInGB = value;
  }
  inline StartExportTaskResult& WithTotalExtractedDataInGB(int value) {
    SetTotalExtractedDataInGB(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The reason the export failed, if it failed.</p>
   */
  inline const Aws::String& GetFailureCause() const { return m_failureCause; }
  template <typename FailureCauseT = Aws::String>
  void SetFailureCause(FailureCauseT&& value) {
    m_failureCauseHasBeenSet = true;
    m_failureCause = std::forward<FailureCauseT>(value);
  }
  template <typename FailureCauseT = Aws::String>
  StartExportTaskResult& WithFailureCause(FailureCauseT&& value) {
    SetFailureCause(std::forward<FailureCauseT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A warning about the snapshot or cluster export task.</p>
   */
  inline const Aws::String& GetWarningMessage() const { return m_warningMessage; }
  template <typename WarningMessageT = Aws::String>
  void SetWarningMessage(WarningMessageT&& value) {
    m_warningMessageHasBeenSet = true;
    m_warningMessage = std::forward<WarningMessageT>(value);
  }
  template <typename WarningMessageT = Aws::String>
  StartExportTaskResult& WithWarningMessage(WarningMessageT&& value) {
    SetWarningMessage(std::forward<WarningMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of source for the export.</p>
   */
  inline ExportSourceType GetSourceType() const { return m_sourceType; }
  inline void SetSourceType(ExportSourceType value) {
    m_sourceTypeHasBeenSet = true;
    m_sourceType = value;
  }
  inline StartExportTaskResult& WithSourceType(ExportSourceType value) {
    SetSourceType(value);
    return *this;
  }
  ///@}

  ///@{

  inline const ResponseMetadata& GetResponseMetadata() const { return m_responseMetadata; }
  template <typename ResponseMetadataT = ResponseMetadata>
  void SetResponseMetadata(ResponseMetadataT&& value) {
    m_responseMetadataHasBeenSet = true;
    m_responseMetadata = std::forward<ResponseMetadataT>(value);
  }
  template <typename ResponseMetadataT = ResponseMetadata>
  StartExportTaskResult& WithResponseMetadata(ResponseMetadataT&& value) {
    SetResponseMetadata(std::forward<ResponseMetadataT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_exportTaskIdentifier;

  Aws::String m_sourceArn;

  Aws::Vector<Aws::String> m_exportOnly;

  Aws::Utils::DateTime m_snapshotTime{};

  Aws::Utils::DateTime m_taskStartTime{};

  Aws::Utils::DateTime m_taskEndTime{};

  Aws::String m_s3Bucket;

  Aws::String m_s3Prefix;

  Aws::String m_iamRoleArn;

  Aws::String m_kmsKeyId;

  Aws::String m_status;

  int m_percentProgress{0};

  int m_totalExtractedDataInGB{0};

  Aws::String m_failureCause;

  Aws::String m_warningMessage;

  ExportSourceType m_sourceType{ExportSourceType::NOT_SET};

  ResponseMetadata m_responseMetadata;
  bool m_exportTaskIdentifierHasBeenSet = false;
  bool m_sourceArnHasBeenSet = false;
  bool m_exportOnlyHasBeenSet = false;
  bool m_snapshotTimeHasBeenSet = false;
  bool m_taskStartTimeHasBeenSet = false;
  bool m_taskEndTimeHasBeenSet = false;
  bool m_s3BucketHasBeenSet = false;
  bool m_s3PrefixHasBeenSet = false;
  bool m_iamRoleArnHasBeenSet = false;
  bool m_kmsKeyIdHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_percentProgressHasBeenSet = false;
  bool m_totalExtractedDataInGBHasBeenSet = false;
  bool m_failureCauseHasBeenSet = false;
  bool m_warningMessageHasBeenSet = false;
  bool m_sourceTypeHasBeenSet = false;
  bool m_responseMetadataHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
