import asyncio
import datetime
import logging
import os
import subprocess
from copy import deepcopy

import botocore.compat
import dateutil.parser
from botocore import UNSIGNED
from botocore.compat import (
    EC,
    compat_shell_split,
    total_seconds,
)
from botocore.credentials import (
    _DEFAULT_ADVISORY_REFRESH_TIMEOUT,
    AssumeRoleCredentialFetcher,
    AssumeRoleProvider,
    AssumeRoleWithWebIdentityProvider,
    BaseAssumeRoleCredentialFetcher,
    BotoProvider,
    CachedCredentialFetcher,
    CanonicalNameCredentialSourcer,
    ConfigNotFound,
    ConfigProvider,
    ContainerMetadataFetcher,
    ContainerProvider,
    CredentialResolver,
    CredentialRetrievalError,
    Credentials,
    DeferredRefreshableCredentials,
    EnvProvider,
    InstanceMetadataProvider,
    InvalidConfigError,
    LoginCredentialFetcher,
    LoginError,
    LoginInsufficientPermissions,
    LoginProvider,
    LoginRefreshRequired,
    LoginTokenLoader,
    LoginTokenLoadError,
    MetadataRetrievalError,
    MissingDependencyException,
    OriginalEC2Provider,
    PartialCredentialsError,
    ProcessProvider,
    ProfileProviderBuilder,
    ReadOnlyCredentials,
    RefreshableCredentials,
    RefreshWithMFAUnsupportedError,
    SharedCredentialProvider,
    SSOCredentialFetcher,
    SSOProvider,
    SSOTokenLoader,
    UnauthorizedSSOTokenError,
    UnknownCredentialError,
    _build_add_dpop_header_handler,
    _local_now,
    _parse_if_needed,
    _serialize_if_needed,
    parse,
    resolve_imds_endpoint_mode,
)
from botocore.useragent import register_feature_id, register_feature_ids
from dateutil.tz import tzutc

from aiobotocore._helpers import resolve_awaitable
from aiobotocore.config import AioConfig
from aiobotocore.tokens import AioSSOTokenProvider
from aiobotocore.utils import (
    AioContainerMetadataFetcher,
    AioInstanceMetadataFetcher,
    create_nested_client,
)

logger = logging.getLogger(__name__)


def create_credential_resolver(session, cache=None, region_name=None):
    """Create a default credential resolver.
    This creates a pre-configured credential resolver
    that includes the default lookup chain for
    credentials.
    """
    profile_name = session.get_config_variable('profile') or 'default'
    metadata_timeout = session.get_config_variable('metadata_service_timeout')
    num_attempts = session.get_config_variable('metadata_service_num_attempts')
    disable_env_vars = session.instance_variables().get('profile') is not None

    imds_config = {
        'ec2_metadata_service_endpoint': session.get_config_variable(
            'ec2_metadata_service_endpoint'
        ),
        'ec2_metadata_service_endpoint_mode': resolve_imds_endpoint_mode(
            session
        ),
        'ec2_credential_refresh_window': _DEFAULT_ADVISORY_REFRESH_TIMEOUT,
        'ec2_metadata_v1_disabled': session.get_config_variable(
            'ec2_metadata_v1_disabled'
        ),
    }

    if cache is None:
        cache = {}

    env_provider = AioEnvProvider()
    container_provider = AioContainerProvider()
    instance_metadata_provider = AioInstanceMetadataProvider(
        iam_role_fetcher=AioInstanceMetadataFetcher(
            timeout=metadata_timeout,
            num_attempts=num_attempts,
            user_agent=session.user_agent(),
            config=imds_config,
        )
    )

    profile_provider_builder = AioProfileProviderBuilder(
        session, cache=cache, region_name=region_name
    )
    assume_role_provider = AioAssumeRoleProvider(
        load_config=lambda: session.full_config,
        client_creator=_get_client_creator(session, region_name),
        cache=cache,
        profile_name=profile_name,
        credential_sourcer=AioCanonicalNameCredentialSourcer(
            [env_provider, container_provider, instance_metadata_provider]
        ),
        profile_provider_builder=profile_provider_builder,
    )

    pre_profile = [
        env_provider,
        assume_role_provider,
    ]
    profile_providers = profile_provider_builder.providers(
        profile_name=profile_name,
        disable_env_vars=disable_env_vars,
    )
    post_profile = [
        AioOriginalEC2Provider(),
        AioBotoProvider(),
        container_provider,
        instance_metadata_provider,
    ]
    providers = pre_profile + profile_providers + post_profile

    if disable_env_vars:
        # An explicitly provided profile will negate an EnvProvider.
        # We will defer to providers that understand the "profile"
        # concept to retrieve credentials.
        # The one edge case if is all three values are provided via
        # env vars:
        # export AWS_ACCESS_KEY_ID=foo
        # export AWS_SECRET_ACCESS_KEY=bar
        # export AWS_PROFILE=baz
        # Then, just like our client() calls, the explicit credentials
        # will take precedence.
        #
        # This precedence is enforced by leaving the EnvProvider in the chain.
        # This means that the only way a "profile" would win is if the
        # EnvProvider does not return credentials, which is what we want
        # in this scenario.
        providers.remove(env_provider)
        logger.debug(
            'Skipping environment variable credential check'
            ' because profile name was explicitly set.'
        )

    resolver = AioCredentialResolver(providers=providers)
    return resolver


class AioProfileProviderBuilder(ProfileProviderBuilder):
    def _create_process_provider(self, profile_name):
        return AioProcessProvider(
            profile_name=profile_name,
            load_config=lambda: self._session.full_config,
        )

    def _create_shared_credential_provider(self, profile_name):
        credential_file = self._session.get_config_variable('credentials_file')
        return AioSharedCredentialProvider(
            profile_name=profile_name,
            creds_filename=credential_file,
        )

    def _create_config_provider(self, profile_name):
        config_file = self._session.get_config_variable('config_file')
        return AioConfigProvider(
            profile_name=profile_name,
            config_filename=config_file,
        )

    def _create_web_identity_provider(self, profile_name, disable_env_vars):
        return AioAssumeRoleWithWebIdentityProvider(
            load_config=lambda: self._session.full_config,
            client_creator=_get_client_creator(
                self._session, self._region_name
            ),
            cache=self._cache,
            profile_name=profile_name,
            disable_env_vars=disable_env_vars,
        )

    def _create_sso_provider(self, profile_name):
        return AioSSOProvider(
            load_config=lambda: self._session.full_config,
            client_creator=self._session.create_client,
            profile_name=profile_name,
            cache=self._cache,
            token_cache=self._sso_token_cache,
            token_provider=AioSSOTokenProvider(
                self._session,
                cache=self._sso_token_cache,
                profile_name=profile_name,
            ),
        )

    def _create_login_provider(self, profile_name):
        return AioLoginProvider(
            load_config=lambda: self._session.full_config,
            client_creator=self._session.create_client,
            profile_name=profile_name,
            token_cache=self._login_token_cache,
        )


async def get_credentials(session):
    resolver = create_credential_resolver(session)
    return await resolver.load_credentials()


def create_assume_role_refresher(client, params):
    async def refresh():
        async with client as sts:
            response = await sts.assume_role(**params)
        credentials = response['Credentials']
        # We need to normalize the credential names to
        # the values expected by the refresh creds.
        return {
            'access_key': credentials['AccessKeyId'],
            'secret_key': credentials['SecretAccessKey'],
            'token': credentials['SessionToken'],
            'expiry_time': _serialize_if_needed(credentials['Expiration']),
        }

    return refresh


def create_mfa_serial_refresher(actual_refresh):
    class _Refresher:
        def __init__(self, refresh):
            self._refresh = refresh
            self._has_been_called = False

        async def call(self):
            if self._has_been_called:
                # We can explore an option in the future to support
                # reprompting for MFA, but for now we just error out
                # when the temp creds expire.
                raise RefreshWithMFAUnsupportedError()
            self._has_been_called = True
            return await self._refresh()

    return _Refresher(actual_refresh).call


# TODO: deprecate
create_aio_mfa_serial_refresher = create_mfa_serial_refresher


class AioCredentials(Credentials):
    # Overrides for property accessors
    def get_account_id(self):
        return self.account_id

    def get_access_key(self):
        return self.access_key

    def get_secret_key(self):
        return self.secret_key

    def get_token(self):
        return self.token

    async def get_frozen_credentials(self):
        return ReadOnlyCredentials(
            self.access_key, self.secret_key, self.token, self.account_id
        )


class AioRefreshableCredentials(RefreshableCredentials):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._refresh_lock = asyncio.Lock()

    async def get_account_id(self):
        await self._refresh()
        return self._account_id

    async def get_access_key(self):
        await self._refresh()
        return self._access_key

    async def get_secret_key(self):
        await self._refresh()
        return self._secret_key

    async def get_token(self):
        await self._refresh()
        return self._token

    # Redeclaring the properties so it doesn't call refresh
    # Have to redeclare setter as we're overriding the getter
    @property
    def access_key(self):
        # TODO: this needs to be resolved
        raise NotImplementedError(
            "missing call to self._refresh. "
            "Use get_frozen_credentials or get_access_key"
        )
        return self._access_key

    @access_key.setter
    def access_key(self, value):
        self._access_key = value

    @property
    def secret_key(self):
        # TODO: this needs to be resolved
        raise NotImplementedError(
            "missing call to self._refresh. "
            "Use get_frozen_credentials or get_secret_key instead"
        )
        return self._secret_key

    @secret_key.setter
    def secret_key(self, value):
        self._secret_key = value

    @property
    def token(self):
        # TODO: this needs to be resolved
        raise NotImplementedError(
            "missing call to self._refresh. "
            "Use get_frozen_credentials or get_token instead"
        )
        return self._token

    @token.setter
    def token(self, value):
        self._token = value

    @property
    def account_id(self):
        # TODO: this needs to be resolved
        raise NotImplementedError(
            "missing call to self._refresh. "
            "Use get_frozen_credentials or get_account_id instead"
        )
        return self._account_id

    @account_id.setter
    def account_id(self, value):
        self._account_id = value

    async def _refresh(self):
        if not self.refresh_needed(self._advisory_refresh_timeout):
            return

        # By this point we need a refresh but its not critical
        if not self._refresh_lock.locked():
            async with self._refresh_lock:
                if not self.refresh_needed(self._advisory_refresh_timeout):
                    return
                is_mandatory_refresh = self.refresh_needed(
                    self._mandatory_refresh_timeout
                )
                await self._protected_refresh(
                    is_mandatory=is_mandatory_refresh
                )
                return
        elif self.refresh_needed(self._mandatory_refresh_timeout):
            # If we're here, we absolutely need a refresh and the
            # lock is held so wait for it
            async with self._refresh_lock:
                # Might have refreshed by now
                if not self.refresh_needed(self._mandatory_refresh_timeout):
                    return
                await self._protected_refresh(is_mandatory=True)

    async def _protected_refresh(self, is_mandatory):
        try:
            # AioEnvProvider._create_credentials_fetcher is not and does not need async
            metadata = await resolve_awaitable(self._refresh_using())
        except Exception:
            period_name = 'mandatory' if is_mandatory else 'advisory'
            logger.warning(
                "Refreshing temporary credentials failed "
                "during %s refresh period.",
                period_name,
                exc_info=True,
            )
            if is_mandatory:
                # If this is a mandatory refresh, then
                # all errors that occur when we attempt to refresh
                # credentials are propagated back to the user.
                raise
            # Otherwise we'll just return.
            # The end result will be that we'll use the current
            # set of temporary credentials we have.
            return
        self._set_from_data(metadata)
        self._frozen_credentials = ReadOnlyCredentials(
            self._access_key, self._secret_key, self._token, self._account_id
        )
        if self._is_expired():
            msg = (
                "Credentials were refreshed, but the "
                "refreshed credentials are still expired."
            )
            logger.warning(msg)
            raise RuntimeError(msg)

    async def get_frozen_credentials(self):
        await self._refresh()
        return self._frozen_credentials


class AioDeferredRefreshableCredentials(
    DeferredRefreshableCredentials, AioRefreshableCredentials
):
    def __init__(self, refresh_using, method, time_fetcher=_local_now):
        self._refresh_using = refresh_using
        self._access_key = None
        self._secret_key = None
        self._token = None
        self._account_id = None
        self._expiry_time = None
        self._time_fetcher = time_fetcher
        self._refresh_lock = asyncio.Lock()
        self.method = method
        self._frozen_credentials = None


class AioCachedCredentialFetcher(CachedCredentialFetcher):
    async def _get_credentials(self):
        raise NotImplementedError('_get_credentials()')

    async def fetch_credentials(self):
        return await self._get_cached_credentials()

    async def _get_cached_credentials(self):
        """Get up-to-date credentials.

        This will check the cache for up-to-date credentials, calling assume
        role if none are available.
        """
        response = self._load_from_cache()
        if response is None:
            response = await self._get_credentials()
            self._write_to_cache(response)
        else:
            logger.debug("Credentials for role retrieved from cache.")

        creds = response['Credentials']
        expiration = _serialize_if_needed(creds['Expiration'], iso=True)
        credentials = {
            'access_key': creds['AccessKeyId'],
            'secret_key': creds['SecretAccessKey'],
            'token': creds['SessionToken'],
            'expiry_time': expiration,
            'account_id': creds.get('AccountId'),
        }

        return credentials


class AioBaseAssumeRoleCredentialFetcher(
    BaseAssumeRoleCredentialFetcher, AioCachedCredentialFetcher
):
    pass


class AioAssumeRoleCredentialFetcher(
    AssumeRoleCredentialFetcher, AioBaseAssumeRoleCredentialFetcher
):
    async def _get_credentials(self):
        """Get credentials by calling assume role."""
        register_feature_ids(self.feature_ids)
        kwargs = self._assume_role_kwargs()
        client = await self._create_client()
        async with client as sts:
            response = await sts.assume_role(**kwargs)
            self._add_account_id_to_response(response)
            return response

    async def _create_client(self):
        """Create an STS client using the source credentials."""
        frozen_credentials = (
            await self._source_credentials.get_frozen_credentials()
        )
        return self._client_creator(
            'sts',
            aws_access_key_id=frozen_credentials.access_key,
            aws_secret_access_key=frozen_credentials.secret_key,
            aws_session_token=frozen_credentials.token,
        )


class AioAssumeRoleWithWebIdentityCredentialFetcher(
    AioBaseAssumeRoleCredentialFetcher
):
    def __init__(
        self,
        client_creator,
        web_identity_token_loader,
        role_arn,
        extra_args=None,
        cache=None,
        expiry_window_seconds=None,
    ):
        self._web_identity_token_loader = web_identity_token_loader

        super().__init__(
            client_creator,
            role_arn,
            extra_args=extra_args,
            cache=cache,
            expiry_window_seconds=expiry_window_seconds,
        )

    async def _get_credentials(self):
        """Get credentials by calling assume role."""
        register_feature_ids(self.feature_ids)
        kwargs = self._assume_role_kwargs()
        # Assume role with web identity does not require credentials other than
        # the token, explicitly configure the client to not sign requests.
        config = AioConfig(signature_version=UNSIGNED)
        async with self._client_creator('sts', config=config) as client:
            response = await client.assume_role_with_web_identity(**kwargs)
            self._add_account_id_to_response(response)
            return response

    def _assume_role_kwargs(self):
        """Get the arguments for assume role based on current configuration."""
        assume_role_kwargs = deepcopy(self._assume_kwargs)
        identity_token = self._web_identity_token_loader()
        assume_role_kwargs['WebIdentityToken'] = identity_token

        return assume_role_kwargs


class AioProcessProvider(ProcessProvider):
    def __init__(self, *args, popen=asyncio.create_subprocess_exec, **kwargs):
        super().__init__(*args, **kwargs, popen=popen)

    async def load(self):
        credential_process = self._credential_process
        if credential_process is None:
            return

        register_feature_id('CREDENTIALS_PROFILE_PROCESS')
        creds_dict = await self._retrieve_credentials_using(credential_process)
        register_feature_id('CREDENTIALS_PROCESS')
        if creds_dict.get('expiry_time') is not None:
            return AioRefreshableCredentials.create_from_metadata(
                creds_dict,
                lambda: self._retrieve_credentials_using(credential_process),
                self.METHOD,
            )

        return AioCredentials(
            access_key=creds_dict['access_key'],
            secret_key=creds_dict['secret_key'],
            token=creds_dict.get('token'),
            method=self.METHOD,
            account_id=creds_dict.get('account_id'),
        )

    async def _retrieve_credentials_using(self, credential_process):
        # We're not using shell=True, so we need to pass the
        # command and all arguments as a list.
        process_list = compat_shell_split(credential_process)
        p = await self._popen(
            *process_list, stdout=subprocess.PIPE, stderr=subprocess.PIPE
        )
        stdout, stderr = await p.communicate()
        if p.returncode != 0:
            raise CredentialRetrievalError(
                provider=self.METHOD, error_msg=stderr.decode('utf-8')
            )
        parsed = botocore.compat.json.loads(stdout.decode('utf-8'))
        version = parsed.get('Version', '<Version key not provided>')
        if version != 1:
            raise CredentialRetrievalError(
                provider=self.METHOD,
                error_msg=(
                    f"Unsupported version '{version}' for credential process "
                    f"provider, supported versions: 1"
                ),
            )
        try:
            return {
                'access_key': parsed['AccessKeyId'],
                'secret_key': parsed['SecretAccessKey'],
                'token': parsed.get('SessionToken'),
                'expiry_time': parsed.get('Expiration'),
                'account_id': self._get_account_id(parsed),
            }
        except KeyError as e:
            raise CredentialRetrievalError(
                provider=self.METHOD,
                error_msg=f"Missing required key in response: {e}",
            )


class AioInstanceMetadataProvider(InstanceMetadataProvider):
    async def load(self):
        fetcher = self._role_fetcher
        metadata = await fetcher.retrieve_iam_role_credentials()
        if not metadata:
            return None
        register_feature_id('CREDENTIALS_IMDS')
        logger.info(
            'Found credentials from IAM Role: %s', metadata['role_name']
        )

        creds = AioRefreshableCredentials.create_from_metadata(
            metadata,
            method=self.METHOD,
            refresh_using=fetcher.retrieve_iam_role_credentials,
        )
        return creds


class AioEnvProvider(EnvProvider):
    async def load(self):
        access_key = self.environ.get(self._mapping['access_key'], '')

        if access_key:
            logger.info('Found credentials in environment variables.')
            fetcher = self._create_credentials_fetcher()
            credentials = fetcher(require_expiry=False)
            register_feature_id('CREDENTIALS_ENV_VARS')

            expiry_time = credentials['expiry_time']
            if expiry_time is not None:
                expiry_time = parse(expiry_time)
                return AioRefreshableCredentials(
                    credentials['access_key'],
                    credentials['secret_key'],
                    credentials['token'],
                    expiry_time,
                    refresh_using=fetcher,
                    method=self.METHOD,
                    account_id=credentials['account_id'],
                )

            return AioCredentials(
                credentials['access_key'],
                credentials['secret_key'],
                credentials['token'],
                method=self.METHOD,
                account_id=credentials['account_id'],
            )
        else:
            return None


class AioOriginalEC2Provider(OriginalEC2Provider):
    async def load(self):
        if 'AWS_CREDENTIAL_FILE' in self._environ:
            full_path = os.path.expanduser(
                self._environ['AWS_CREDENTIAL_FILE']
            )
            creds = self._parser(full_path)
            if self.ACCESS_KEY in creds:
                logger.info('Found credentials in AWS_CREDENTIAL_FILE.')
                access_key = creds[self.ACCESS_KEY]
                secret_key = creds[self.SECRET_KEY]
                # EC2 creds file doesn't support session tokens.
                return AioCredentials(
                    access_key, secret_key, method=self.METHOD
                )
        else:
            return None


class AioSharedCredentialProvider(SharedCredentialProvider):
    async def load(self):
        try:
            available_creds = self._ini_parser(self._creds_filename)
        except ConfigNotFound:
            return None
        if self._profile_name in available_creds:
            config = available_creds[self._profile_name]
            if self.ACCESS_KEY in config:
                logger.info(
                    "Found credentials in shared credentials file: %s",
                    self._creds_filename,
                )
                access_key, secret_key = self._extract_creds_from_mapping(
                    config, self.ACCESS_KEY, self.SECRET_KEY
                )
                token = self._get_session_token(config)
                account_id = self._get_account_id(config)
                register_feature_id('CREDENTIALS_PROFILE')
                return AioCredentials(
                    access_key,
                    secret_key,
                    token,
                    method=self.METHOD,
                    account_id=account_id,
                )


class AioConfigProvider(ConfigProvider):
    async def load(self):
        try:
            full_config = self._config_parser(self._config_filename)
        except ConfigNotFound:
            return None
        if self._profile_name in full_config['profiles']:
            profile_config = full_config['profiles'][self._profile_name]
            if self.ACCESS_KEY in profile_config:
                logger.info(
                    "Credentials found in config file: %s",
                    self._config_filename,
                )
                access_key, secret_key = self._extract_creds_from_mapping(
                    profile_config, self.ACCESS_KEY, self.SECRET_KEY
                )
                token = self._get_session_token(profile_config)
                account_id = self._get_account_id(profile_config)
                register_feature_id('CREDENTIALS_PROFILE')
                return AioCredentials(
                    access_key,
                    secret_key,
                    token,
                    method=self.METHOD,
                    account_id=account_id,
                )
        else:
            return None


class AioBotoProvider(BotoProvider):
    async def load(self):
        if self.BOTO_CONFIG_ENV in self._environ:
            potential_locations = [self._environ[self.BOTO_CONFIG_ENV]]
        else:
            potential_locations = self.DEFAULT_CONFIG_FILENAMES
        for filename in potential_locations:
            try:
                config = self._ini_parser(filename)
            except ConfigNotFound:
                # Move on to the next potential config file name.
                continue
            if 'Credentials' in config:
                credentials = config['Credentials']
                if self.ACCESS_KEY in credentials:
                    logger.info(
                        "Found credentials in boto config file: %s", filename
                    )
                    access_key, secret_key = self._extract_creds_from_mapping(
                        credentials, self.ACCESS_KEY, self.SECRET_KEY
                    )
                    register_feature_id('CREDENTIALS_BOTO2_CONFIG_FILE')
                    return AioCredentials(
                        access_key, secret_key, method=self.METHOD
                    )


class AioAssumeRoleProvider(AssumeRoleProvider):
    async def load(self):
        self._loaded_config = self._load_config()
        profiles = self._loaded_config.get('profiles', {})
        profile = profiles.get(self._profile_name, {})
        if self._has_assume_role_config_vars(profile):
            return await self._load_creds_via_assume_role(self._profile_name)

    async def _load_creds_via_assume_role(self, profile_name):
        role_config = self._get_role_config(profile_name)
        source_credentials = await self._resolve_source_credentials(
            role_config, profile_name
        )

        extra_args = {}
        role_session_name = role_config.get('role_session_name')
        if role_session_name is not None:
            extra_args['RoleSessionName'] = role_session_name

        external_id = role_config.get('external_id')
        if external_id is not None:
            extra_args['ExternalId'] = external_id

        mfa_serial = role_config.get('mfa_serial')
        if mfa_serial is not None:
            extra_args['SerialNumber'] = mfa_serial

        duration_seconds = role_config.get('duration_seconds')
        if duration_seconds is not None:
            extra_args['DurationSeconds'] = duration_seconds

        fetcher = AioAssumeRoleCredentialFetcher(
            client_creator=self._client_creator,
            source_credentials=source_credentials,
            role_arn=role_config['role_arn'],
            extra_args=extra_args,
            mfa_prompter=self._prompter,
            cache=self.cache,
        )
        fetcher.feature_ids = self._feature_ids.copy()
        refresher = fetcher.fetch_credentials
        if mfa_serial is not None:
            refresher = create_mfa_serial_refresher(refresher)

        self._feature_ids.add('CREDENTIALS_STS_ASSUME_ROLE')
        register_feature_ids(self._feature_ids)
        # The initial credentials are empty and the expiration time is set
        # to now so that we can delay the call to assume role until it is
        # strictly needed.
        return AioDeferredRefreshableCredentials(
            method=self.METHOD,
            refresh_using=refresher,
            time_fetcher=_local_now,
        )

    async def _resolve_source_credentials(self, role_config, profile_name):
        credential_source = role_config.get('credential_source')
        if credential_source is not None:
            self._feature_ids.add('CREDENTIALS_PROFILE_NAMED_PROVIDER')
            return await self._resolve_credentials_from_source(
                credential_source, profile_name
            )

        source_profile = role_config['source_profile']
        self._visited_profiles.append(source_profile)
        self._feature_ids.add('CREDENTIALS_PROFILE_SOURCE_PROFILE')
        return await self._resolve_credentials_from_profile(source_profile)

    async def _resolve_credentials_from_profile(self, profile_name):
        profiles = self._loaded_config.get('profiles', {})
        profile = profiles[profile_name]
        self._feature_ids.add('CREDENTIALS_PROFILE')
        if (
            self._has_static_credentials(profile)
            and not self._profile_provider_builder
        ):
            # This is only here for backwards compatibility. If this provider
            # isn't given a profile provider builder we still want to be able
            # to handle the basic static credential case as we would before the
            # profile provider builder parameter was added.
            return self._resolve_static_credentials_from_profile(profile)
        elif self._has_static_credentials(
            profile
        ) or not self._has_assume_role_config_vars(profile):
            profile_providers = self._profile_provider_builder.providers(
                profile_name=profile_name,
                disable_env_vars=True,
            )
            profile_chain = AioCredentialResolver(profile_providers)
            credentials = await profile_chain.load_credentials()
            if credentials is None:
                error_message = (
                    'The source profile "%s" must have credentials.'
                )
                raise InvalidConfigError(
                    error_msg=error_message % profile_name,
                )
            return credentials
        return await self._load_creds_via_assume_role(profile_name)

    def _resolve_static_credentials_from_profile(self, profile):
        try:
            return AioCredentials(
                access_key=profile['aws_access_key_id'],
                secret_key=profile['aws_secret_access_key'],
                token=profile.get('aws_session_token'),
            )
        except KeyError as e:
            raise PartialCredentialsError(
                provider=self.METHOD, cred_var=str(e)
            )

    async def _resolve_credentials_from_source(
        self, credential_source, profile_name
    ):
        credentials = await self._credential_sourcer.source_credentials(
            credential_source
        )
        if credentials is None:
            raise CredentialRetrievalError(
                provider=credential_source,
                error_msg=(
                    'No credentials found in credential_source referenced '
                    f'in profile {profile_name}'
                ),
            )
        named_provider_feature_id = self.NAMED_PROVIDER_FEATURE_MAP.get(
            credential_source
        )
        if named_provider_feature_id:
            self._feature_ids.add(named_provider_feature_id)
        return credentials


class AioAssumeRoleWithWebIdentityProvider(AssumeRoleWithWebIdentityProvider):
    async def load(self):
        return await self._assume_role_with_web_identity()

    async def _assume_role_with_web_identity(self):
        token_path = self._get_config('web_identity_token_file')
        if not token_path:
            return None
        token_loader = self._token_loader_cls(token_path)

        role_arn = self._get_config('role_arn')
        if not role_arn:
            error_msg = (
                'The provided profile or the current environment is '
                'configured to assume role with web identity but has no '
                'role ARN configured. Ensure that the profile has the role_arn'
                'configuration set or the AWS_ROLE_ARN env var is set.'
            )
            raise InvalidConfigError(error_msg=error_msg)

        extra_args = {}
        role_session_name = self._get_config('role_session_name')
        if role_session_name is not None:
            extra_args['RoleSessionName'] = role_session_name

        fetcher = AioAssumeRoleWithWebIdentityCredentialFetcher(
            client_creator=self._client_creator,
            web_identity_token_loader=token_loader,
            role_arn=role_arn,
            extra_args=extra_args,
            cache=self.cache,
        )
        fetcher.feature_ids = self._feature_ids.copy()

        self._feature_ids.add('CREDENTIALS_STS_ASSUME_ROLE_WEB_ID')
        register_feature_ids(self._feature_ids)
        # The initial credentials are empty and the expiration time is set
        # to now so that we can delay the call to assume role until it is
        # strictly needed.
        return AioDeferredRefreshableCredentials(
            method=self.METHOD,
            refresh_using=fetcher.fetch_credentials,
        )


class AioCanonicalNameCredentialSourcer(CanonicalNameCredentialSourcer):
    async def source_credentials(self, source_name):
        """Loads source credentials based on the provided configuration.

        :type source_name: str
        :param source_name: The value of credential_source in the config
            file. This is the canonical name of the credential provider.

        :rtype: Credentials
        """
        source = self._get_provider(source_name)
        if isinstance(source, AioCredentialResolver):
            return await source.load_credentials()
        return await source.load()

    def _get_provider(self, canonical_name):
        """Return a credential provider by its canonical name.

        :type canonical_name: str
        :param canonical_name: The canonical name of the provider.

        :raises UnknownCredentialError: Raised if no
            credential provider by the provided name
            is found.
        """
        provider = self._get_provider_by_canonical_name(canonical_name)

        # The AssumeRole provider should really be part of the SharedConfig
        # provider rather than being its own thing, but it is not. It is
        # effectively part of both the SharedConfig provider and the
        # SharedCredentials provider now due to the way it behaves.
        # Therefore if we want either of those providers we should return
        # the AssumeRole provider with it.
        if canonical_name.lower() in ['sharedconfig', 'sharedcredentials']:
            assume_role_provider = self._get_provider_by_method('assume-role')
            if assume_role_provider is not None:
                # The SharedConfig or SharedCredentials provider may not be
                # present if it was removed for some reason, but the
                # AssumeRole provider could still be present. In that case,
                # return the assume role provider by itself.
                if provider is None:
                    return assume_role_provider

                # If both are present, return them both as a
                # CredentialResolver so that calling code can treat them as
                # a single entity.
                return AioCredentialResolver([assume_role_provider, provider])

        if provider is None:
            raise UnknownCredentialError(name=canonical_name)

        return provider


class AioContainerProvider(ContainerProvider):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        # This will always run if no fetcher arg is provided
        if isinstance(self._fetcher, ContainerMetadataFetcher):
            self._fetcher = AioContainerMetadataFetcher()

    async def load(self):
        if self.ENV_VAR in self._environ or self.ENV_VAR_FULL in self._environ:
            return await self._retrieve_or_fail()

    async def _retrieve_or_fail(self):
        if self._provided_relative_uri():
            full_uri = self._fetcher.full_url(self._environ[self.ENV_VAR])
        else:
            full_uri = self._environ[self.ENV_VAR_FULL]
        fetcher = self._create_fetcher(full_uri)
        creds = await fetcher()
        return AioRefreshableCredentials(
            access_key=creds['access_key'],
            secret_key=creds['secret_key'],
            token=creds['token'],
            method=self.METHOD,
            expiry_time=_parse_if_needed(creds['expiry_time']),
            refresh_using=fetcher,
            account_id=creds.get('account_id'),
        )

    def _create_fetcher(self, full_uri, *args, **kwargs):
        async def fetch_creds():
            try:
                headers = self._build_headers()
                response = await self._fetcher.retrieve_full_uri(
                    full_uri, headers=headers
                )
                register_feature_id('CREDENTIALS_HTTP')
            except MetadataRetrievalError as e:
                logger.debug(
                    "Error retrieving container metadata: %s", e, exc_info=True
                )
                raise CredentialRetrievalError(
                    provider=self.METHOD, error_msg=str(e)
                )
            return {
                'access_key': response['AccessKeyId'],
                'secret_key': response['SecretAccessKey'],
                'token': response['Token'],
                'expiry_time': response['Expiration'],
                'account_id': response.get('AccountId'),
            }

        return fetch_creds


class AioCredentialResolver(CredentialResolver):
    async def load_credentials(self):
        """
        Goes through the credentials chain, returning the first ``Credentials``
        that could be loaded.
        """
        # First provider to return a non-None response wins.
        for provider in self.providers:
            logger.debug("Looking for credentials via: %s", provider.METHOD)
            creds = await provider.load()
            if creds is not None:
                return creds

        # If we got here, no credentials could be found.
        # This feels like it should be an exception, but historically, ``None``
        # is returned.
        #
        # +1
        # -js
        return None


class AioSSOCredentialFetcher(
    SSOCredentialFetcher, AioCachedCredentialFetcher
):
    async def _get_credentials(self):
        """Get credentials by calling SSO get role credentials."""
        config = AioConfig(
            signature_version=UNSIGNED,
            region_name=self._sso_region,
        )
        async with self._client_creator('sso', config=config) as client:
            if self._token_provider:
                initial_token_data = self._token_provider.load_token()
                token = (await initial_token_data.get_frozen_token()).token
            else:
                token_dict = self._token_loader(self._start_url)
                token = token_dict['accessToken']

                # raise an UnauthorizedSSOTokenError if the loaded legacy token
                # is expired to save a call to GetRoleCredentials with an
                # expired token.
                expiration = dateutil.parser.parse(token_dict['expiresAt'])
                remaining = total_seconds(expiration - self._time_fetcher())
                if remaining <= 0:
                    raise UnauthorizedSSOTokenError()

            kwargs = {
                'roleName': self._role_name,
                'accountId': self._account_id,
                'accessToken': token,
            }
            try:
                register_feature_ids(self.feature_ids)
                response = await client.get_role_credentials(**kwargs)
            except client.exceptions.UnauthorizedException:
                raise UnauthorizedSSOTokenError()
            credentials = response['roleCredentials']

            credentials = {
                'ProviderType': 'sso',
                'Credentials': {
                    'AccessKeyId': credentials['accessKeyId'],
                    'SecretAccessKey': credentials['secretAccessKey'],
                    'SessionToken': credentials['sessionToken'],
                    'Expiration': self._parse_timestamp(
                        credentials['expiration']
                    ),
                    'AccountId': self._account_id,
                },
            }
            return credentials


class AioSSOProvider(SSOProvider):
    async def load(self):
        sso_config = self._load_sso_config()
        if not sso_config:
            return None

        fetcher_kwargs = {
            'start_url': sso_config['sso_start_url'],
            'sso_region': sso_config['sso_region'],
            'role_name': sso_config['sso_role_name'],
            'account_id': sso_config['sso_account_id'],
            'client_creator': self._client_creator,
            'token_loader': SSOTokenLoader(cache=self._token_cache),
            'cache': self.cache,
        }
        sso_session_in_config = 'sso_session' in sso_config
        if sso_session_in_config:
            fetcher_kwargs['sso_session_name'] = sso_config['sso_session']
            fetcher_kwargs['token_provider'] = self._token_provider
            self._feature_ids.add('CREDENTIALS_PROFILE_SSO')
        else:
            self._feature_ids.add('CREDENTIALS_PROFILE_SSO_LEGACY')

        sso_fetcher = AioSSOCredentialFetcher(**fetcher_kwargs)
        sso_fetcher.feature_ids = self._feature_ids.copy()

        if sso_session_in_config:
            self._feature_ids.add('CREDENTIALS_SSO')
        else:
            self._feature_ids.add('CREDENTIALS_SSO_LEGACY')

        register_feature_ids(self._feature_ids)
        return AioDeferredRefreshableCredentials(
            method=self.METHOD,
            refresh_using=sso_fetcher.fetch_credentials,
        )


class AioLoginCredentialFetcher(LoginCredentialFetcher):
    async def refresh_credentials(self):
        """Refreshes login credentials, including saving them to the cache."""
        if self.feature_ids:
            register_feature_ids(self.feature_ids)
        # Reload the token from disk, we need the refresh info
        token = self._token_loader.load_token(self._session_name)
        private_key = self._load_private_key(token)

        # Check if token has already been refreshed and is still valid
        if (
            token
            and 'accessToken' in token
            and 'expiresAt' in token['accessToken']
        ):
            expiry_time = _parse_if_needed(token['accessToken']['expiresAt'])
            remaining_time = total_seconds(expiry_time - self._time_fetcher())
            if remaining_time > self._REFRESH_THRESHOLD:
                return self._token_to_credentials(token)

        config = AioConfig(
            signature_version=botocore.UNSIGNED,
        )
        async with self._client_creator(
            'signin',
            config=config,
        ) as client:
            client.meta.events.register(
                'before-call.signin.CreateOAuth2Token',
                _build_add_dpop_header_handler(private_key),
            )

            try:
                response = await client.create_o_auth2_token(
                    tokenInput={
                        'clientId': token['clientId'],
                        'refreshToken': token['refreshToken'],
                        'grantType': 'refresh_token',
                    },
                )
            except client.exceptions.AccessDeniedException as e:
                error_type = e.response.get('error', '')
                if error_type in ('TOKEN_EXPIRED', 'USER_CREDENTIALS_CHANGED'):
                    raise LoginRefreshRequired() from e
                elif error_type == 'INSUFFICIENT_PERMISSIONS':
                    raise LoginInsufficientPermissions() from e
                raise LoginError() from e

            if response is None or 'tokenOutput' not in response:
                raise LoginTokenLoadError(
                    error_msg=(
                        "Unable to refresh access token due to an invalid service response. "
                        "Please try running 'aws login' again. If the issue persists, there "
                        "may be a temporary signin service problem."
                    )
                )

        output = response.get('tokenOutput')

        expires_timestamp = self._time_fetcher().astimezone(
            tzutc()
        ) + datetime.timedelta(seconds=output['expiresIn'])

        # Overwrite token with refreshed fields
        token.update(
            {
                'accessToken': {
                    'accessKeyId': output['accessToken']['accessKeyId'],
                    'secretAccessKey': output['accessToken'][
                        'secretAccessKey'
                    ],
                    'sessionToken': output['accessToken']['sessionToken'],
                    'accountId': token['accessToken']['accountId'],
                    'expiresAt': expires_timestamp.strftime(
                        '%Y-%m-%dT%H:%M:%SZ'
                    ),
                },
                'refreshToken': output['refreshToken'],
            }
        )
        self._token_loader.save_token(self._session_name, token)

        return self._token_to_credentials(token)


class AioLoginProvider(LoginProvider):
    async def load(self):
        loaded_config = self._load_config()
        profiles = loaded_config.get('profiles', {})
        profile_config = profiles.get(self._profile_name, {})

        if 'login_session' not in profile_config:
            return None

        if EC is None:
            raise MissingDependencyException(
                msg=(
                    "Using the login credential provider requires an "
                    "additional dependency. You will need to pip install "
                    "\"botocore[crt]\" before proceeding."
                )
            )

        fetcher = AioLoginCredentialFetcher(
            session_name=profile_config['login_session'],
            token_loader=LoginTokenLoader(self._token_cache),
            client_creator=self._client_creator,
            time_fetcher=_local_now,
            feature_ids=self._feature_ids,
        )

        register_feature_ids(self._feature_ids)

        # Return the current cached credentials initially,
        # regardless if they're expired
        cached_credentials = fetcher.load_cached_credentials()

        return AioRefreshableCredentials(
            access_key=cached_credentials['access_key'],
            secret_key=cached_credentials['secret_key'],
            token=cached_credentials['token'],
            expiry_time=_parse_if_needed(cached_credentials['expiry_time']),
            account_id=cached_credentials['account_id'],
            method=self.METHOD,
            refresh_using=fetcher.refresh_credentials,
            time_fetcher=_local_now,
        )


def _get_client_creator(session, region_name):
    """Create a client creator function for use in credential providers.

    This is the async version of botocore.credentials._get_client_creator that
    uses aiobotocore's create_nested_client.
    """

    def client_creator(service_name, **kwargs):
        create_client_kwargs = {'region_name': region_name}
        create_client_kwargs.update(**kwargs)
        return create_nested_client(
            session, service_name, **create_client_kwargs
        )

    return client_creator
