﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime/ChimeRequest.h>
#include <aws/chime/Chime_EXPORTS.h>
#include <aws/chime/model/UserType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace Chime {
namespace Model {

/**
 */
class ListUsersRequest : public ChimeRequest {
 public:
  AWS_CHIME_API ListUsersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListUsers"; }

  AWS_CHIME_API Aws::String SerializePayload() const override;

  AWS_CHIME_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The Amazon Chime account ID.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  ListUsersRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional. The user email address used to filter results. Maximum 1.</p>
   */
  inline const Aws::String& GetUserEmail() const { return m_userEmail; }
  inline bool UserEmailHasBeenSet() const { return m_userEmailHasBeenSet; }
  template <typename UserEmailT = Aws::String>
  void SetUserEmail(UserEmailT&& value) {
    m_userEmailHasBeenSet = true;
    m_userEmail = std::forward<UserEmailT>(value);
  }
  template <typename UserEmailT = Aws::String>
  ListUsersRequest& WithUserEmail(UserEmailT&& value) {
    SetUserEmail(std::forward<UserEmailT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user type.</p>
   */
  inline UserType GetUserType() const { return m_userType; }
  inline bool UserTypeHasBeenSet() const { return m_userTypeHasBeenSet; }
  inline void SetUserType(UserType value) {
    m_userTypeHasBeenSet = true;
    m_userType = value;
  }
  inline ListUsersRequest& WithUserType(UserType value) {
    SetUserType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return in a single call. Defaults to
   * 100.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListUsersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to use to retrieve the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListUsersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;

  Aws::String m_userEmail;

  UserType m_userType{UserType::NOT_SET};

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_accountIdHasBeenSet = false;
  bool m_userEmailHasBeenSet = false;
  bool m_userTypeHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace Chime
}  // namespace Aws
