/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*!
 * \file tvm/ir/global_var_supply.h
 * \brief GlobalVarSupply that can be used to generate unique \class GlobalVar.
 */
#ifndef TVM_IR_GLOBAL_VAR_SUPPLY_H_
#define TVM_IR_GLOBAL_VAR_SUPPLY_H_

#include <string>
#include <unordered_map>

#include "tvm/ir/expr.h"
#include "tvm/ir/module.h"
#include "tvm/ir/name_supply.h"

namespace tvm {

/*!
 * \brief GlobalVarSupply can be used to generate unique GlobalVars.
 */
class GlobalVarSupplyNode : public Object {
 public:
  /*!
   * \brief Empty constructor. Will use an empty NameSupply.
   */
  GlobalVarSupplyNode() : GlobalVarSupplyNode(NameSupply("")) {}

  /*!
   * \brief Constructor.
   * \param name_supply The NameSupply to use for generating the names of fresh GlobalVars.
   * \param name_to_var_map An optional map.
   */
  explicit GlobalVarSupplyNode(NameSupply name_supply,
                               std::unordered_map<std::string, GlobalVar> name_to_var_map = {});

  /*!
   * \brief Generates a unique GlobalVar from this supply.
   * \param name The name from which the name of the GlobalVar is derived.
   * \param add_prefix If set to true, then the prefix of the contained NameSupply will be prepended
   * to the name. \return A unique GlobalVar.
   */
  GlobalVar FreshGlobal(String name, bool add_prefix = true);

  /*!
   * \brief Looks up for a GlobalVar with the given name in this supply.
   * If no entry is found, creates one, places it in the cache and returns it.
   * \param name The name of the GlobalVar to search for.
   * \param add_prefix If set to true, the prefix of the contained NameSupply will be prepended to
   * the name before performing the search. \return A cached GlobalVar.
   */
  GlobalVar UniqueGlobalFor(const String& name, bool add_prefix = true);

  /*!
   * \brief Reserves an existing GlobalVar with this supply.
   * \param var The GlobalVar to be registered.
   * \param allow_conflict Allow conflict with other GlobalVars that have the same name.
   */
  void ReserveGlobalVar(const GlobalVar& var, bool allow_conflict = false);

  void VisitAttrs(AttrVisitor* v) {}

  /*! \brief The NameSupply used to generate unique name hints to GlobalVars. */
  NameSupply name_supply_;

  static constexpr const char* _type_key = "GlobalVarSupply";
  static constexpr const bool _type_has_method_sequal_reduce = false;
  static constexpr const bool _type_has_method_shash_reduce = false;
  TVM_DECLARE_FINAL_OBJECT_INFO(GlobalVarSupplyNode, Object);

 private:
  std::unordered_map<std::string, GlobalVar> name_to_var_map_;
};

/*!
 * \brief Managed reference class to GlobalVarSupplyNode.
 * \sa GlobalVarSupplyNode
 */
class GlobalVarSupply : public ObjectRef {
 public:
  /*!
   * \brief Constructor.
   * \param name_supply The NameSupply to be used when generating new GlobalVars.
   * \param name_to_var_map An optional map.
   */
  TVM_DLL explicit GlobalVarSupply(const NameSupply& name_supply,
                                   std::unordered_map<std::string, GlobalVar> name_to_var_map = {});

  /*!
   * \brief Constructs a supply from an array of IRModules. GlobalVars generated by this supply are
   * guaranteed not to conflict with any GlobalVars that belong to the modules. \param modules Array
   * of IRModules.
   */
  TVM_DLL explicit GlobalVarSupply(const Array<IRModule>& modules);

  /*!
   * \brief Constructs a GlobalVarSupply from an IRModule. GlobalVars generated by this supply are
   * guaranteed not to conflict with GlobalVars that belong to the modules. \param module The
   * IRModule.
   */
  TVM_DLL explicit GlobalVarSupply(const IRModule module);

  TVM_DEFINE_MUTABLE_OBJECT_REF_METHODS(GlobalVarSupply, ObjectRef, GlobalVarSupplyNode);
};

}  // namespace tvm

#endif  // TVM_IR_GLOBAL_VAR_SUPPLY_H_
