/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.parse.repl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ChecksumException;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.InvalidPathException;
import org.apache.hadoop.fs.InvalidRequestException;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIOException;
import org.apache.hadoop.fs.UnsupportedFileSystemException;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hive.common.DataCopyStatistics;
import org.apache.hadoop.hive.common.FileUtils;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.ReplChangeManager;
import org.apache.hadoop.hive.ql.ErrorMsg;
import org.apache.hadoop.hive.ql.exec.util.Retryable;
import org.apache.hadoop.hive.ql.io.AcidUtils;
import org.apache.hadoop.hive.ql.metadata.HiveFatalException;
import org.apache.hadoop.hive.shims.HadoopShims;
import org.apache.hadoop.hive.shims.ShimLoader;
import org.apache.hadoop.hive.shims.Utils;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CopyUtils {
    private static final Logger LOG = LoggerFactory.getLogger(CopyUtils.class);
    public static final String RAW_RESERVED_VIRTUAL_PATH = "/.reserved/raw/";
    private static final int MAX_IO_RETRY = 5;
    private final HiveConf hiveConf;
    private final long maxCopyFileSize;
    private final long maxNumberOfFiles;
    private final boolean hiveInReplTest;
    private final String copyAsUser;
    private FileSystem destinationFs;
    private final int maxParallelCopyTask;
    private AtomicLong totalBytesCopied = new AtomicLong(0L);
    @VisibleForTesting
    public static Callable<Boolean> testCallable;
    private List<Class<? extends Exception>> failOnParentExceptionList = Arrays.asList(PathIOException.class, UnsupportedFileSystemException.class, InvalidPathException.class, InvalidRequestException.class, FileAlreadyExistsException.class, ChecksumException.class, ParentNotDirectoryException.class, QuotaExceededException.class, FileNotFoundException.class);

    public CopyUtils(String distCpDoAsUser, HiveConf hiveConf, FileSystem destinationFs) {
        this.hiveConf = hiveConf;
        this.maxNumberOfFiles = hiveConf.getLongVar(HiveConf.ConfVars.HIVE_EXEC_COPYFILE_MAXNUMFILES);
        this.maxCopyFileSize = hiveConf.getLongVar(HiveConf.ConfVars.HIVE_EXEC_COPYFILE_MAXSIZE);
        this.hiveInReplTest = hiveConf.getBoolVar(HiveConf.ConfVars.HIVE_IN_TEST_REPL);
        this.maxParallelCopyTask = hiveConf.getIntVar(HiveConf.ConfVars.REPL_PARALLEL_COPY_TASKS);
        this.copyAsUser = distCpDoAsUser;
        this.destinationFs = destinationFs;
    }

    private void incrementTotalBytesCopied(long bytesCopied) {
        this.totalBytesCopied.addAndGet(bytesCopied);
    }

    public long getTotalBytesCopied() {
        return this.totalBytesCopied.get();
    }

    private <T> T retryableFxn(Callable<T> callable) throws IOException {
        Retryable retryable = Retryable.builder().withHiveConf(this.hiveConf).withRetryOnException(IOException.class).withFailOnParentExceptionList(this.failOnParentExceptionList).build();
        try {
            return (T)retryable.executeCallable(() -> callable.call());
        }
        catch (Exception e) {
            if (this.failOnParentExceptionList.stream().anyMatch(k -> k.isAssignableFrom(e.getClass()))) {
                throw new IOException(e);
            }
            throw new IOException(ErrorMsg.REPL_FILE_SYSTEM_OPERATION_RETRY.getMsg(), e);
        }
    }

    @VisibleForTesting
    String checkSumFor(Path srcFile, FileSystem fs) throws IOException {
        return this.retryableFxn(() -> ReplChangeManager.checksumFor((Path)srcFile, (FileSystem)fs));
    }

    @VisibleForTesting
    void copyFilesBetweenFS(FileSystem srcFS, Path[] paths, FileSystem dstFS, Path dst, boolean deleteSource, boolean overwrite, DataCopyStatistics copyStatistics) throws IOException {
        this.retryableFxn(() -> {
            boolean preserveXAttrs = FileUtils.shouldPreserveXAttrs((HiveConf)this.hiveConf, (FileSystem)srcFS, (FileSystem)dstFS, (Path)paths[0]);
            FileUtils.copy((FileSystem)srcFS, (Path[])paths, (FileSystem)dstFS, (Path)dst, (boolean)deleteSource, (boolean)overwrite, (boolean)preserveXAttrs, (Configuration)this.hiveConf, (DataCopyStatistics)copyStatistics);
            return null;
        });
    }

    @VisibleForTesting
    boolean exists(FileSystem fs, Path path) throws IOException {
        return this.retryableFxn(() -> fs.exists(path));
    }

    @VisibleForTesting
    boolean delete(FileSystem fs, Path path, boolean recursive) throws IOException {
        return this.retryableFxn(() -> fs.delete(path, recursive));
    }

    @VisibleForTesting
    boolean mkdirs(FileSystem fs, Path path) throws IOException {
        return this.retryableFxn(() -> fs.mkdirs(path));
    }

    @VisibleForTesting
    boolean rename(FileSystem fs, Path srcPath, Path dstPath) throws IOException {
        return this.retryableFxn(() -> fs.rename(srcPath, dstPath));
    }

    @VisibleForTesting
    ContentSummary getContentSummary(FileSystem fs, Path f) throws IOException {
        return this.retryableFxn(() -> fs.getContentSummary(f));
    }

    public void copyAndVerify(Path destRoot, List<ReplChangeManager.FileInfo> srcFiles, Path origSrcPath, boolean readSrcAsFilesList, boolean overwrite) throws IOException, HiveFatalException {
        UserGroupInformation proxyUser = this.getProxyUser();
        if (CollectionUtils.isEmpty(srcFiles)) {
            throw new IOException(ErrorMsg.REPL_INVALID_ARGUMENTS.format("SrcFiles can not be empty during copy operation."));
        }
        FileSystem sourceFs = srcFiles.get(0).getSrcFs();
        boolean useRegularCopy = this.regularCopy(sourceFs, srcFiles);
        ExecutorService executorService = null;
        try {
            if (useRegularCopy || readSrcAsFilesList) {
                Map<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>> map = this.fsToFileMap(srcFiles, destRoot);
                for (Map.Entry<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>> entry : map.entrySet()) {
                    Map<Path, List<ReplChangeManager.FileInfo>> destMap = entry.getValue();
                    if (destMap.size() > 1) {
                        if (executorService == null) {
                            executorService = this.getExecutorService();
                        }
                        ArrayList<Callable<Void>> copyList = new ArrayList<Callable<Void>>();
                        for (Map.Entry<Path, List<ReplChangeManager.FileInfo>> destMapEntry : destMap.entrySet()) {
                            copyList.add(() -> {
                                DataCopyStatistics copyStatistics = new DataCopyStatistics();
                                this.doCopy(destMapEntry, proxyUser, this.regularCopy(sourceFs, (List)destMapEntry.getValue()), overwrite, copyStatistics);
                                this.incrementTotalBytesCopied(copyStatistics.getBytesCopied());
                                return null;
                            });
                        }
                        executorService.invokeAll(copyList);
                        continue;
                    }
                    for (Map.Entry<Path, List<ReplChangeManager.FileInfo>> destMapEntry : destMap.entrySet()) {
                        DataCopyStatistics copyStatistics = new DataCopyStatistics();
                        this.doCopy(destMapEntry, proxyUser, this.regularCopy(sourceFs, destMapEntry.getValue()), overwrite, copyStatistics);
                        this.incrementTotalBytesCopied(copyStatistics.getBytesCopied());
                    }
                }
            } else {
                srcFiles.clear();
                srcFiles.add(new ReplChangeManager.FileInfo(sourceFs, origSrcPath, null));
                DataCopyStatistics copyStatistics = new DataCopyStatistics();
                this.doCopyRetry(sourceFs, srcFiles, destRoot, proxyUser, useRegularCopy, overwrite, copyStatistics);
                this.incrementTotalBytesCopied(copyStatistics.getBytesCopied());
            }
        }
        catch (InterruptedException e) {
            LOG.error("Failed to copy ", (Throwable)e);
            throw new IOException(ErrorMsg.REPL_FILE_SYSTEM_OPERATION_RETRY.getMsg());
        }
        finally {
            if (executorService != null) {
                executorService.shutdown();
            }
        }
    }

    @VisibleForTesting
    ExecutorService getExecutorService() {
        return Executors.newFixedThreadPool(this.maxParallelCopyTask);
    }

    @VisibleForTesting
    void doCopy(Map.Entry<Path, List<ReplChangeManager.FileInfo>> destMapEntry, UserGroupInformation proxyUser, boolean useRegularCopy, boolean overwrite, DataCopyStatistics copyStatistics) throws IOException, HiveFatalException {
        Path destination = destMapEntry.getKey();
        List<ReplChangeManager.FileInfo> fileInfoList = destMapEntry.getValue();
        FileSystem sourceFsOfFileInfo = fileInfoList.get(0).getSourcePath().getFileSystem((Configuration)this.hiveConf);
        if (!this.exists(this.destinationFs, destination) && !this.mkdirs(this.destinationFs, destination)) {
            LOG.error("Failed to create destination directory: " + String.valueOf(destination));
            throw new IOException("Destination directory creation failed");
        }
        this.doCopyRetry(sourceFsOfFileInfo, fileInfoList, destination, proxyUser, useRegularCopy, overwrite, copyStatistics);
    }

    private void doCopyRetry(FileSystem sourceFs, List<ReplChangeManager.FileInfo> srcFileList, Path destination, UserGroupInformation proxyUser, boolean useRegularCopy, boolean overwrite, DataCopyStatistics copyStatistics) throws IOException, HiveFatalException {
        boolean isCopyError = false;
        List<Path> pathList = Lists.transform(srcFileList, ReplChangeManager.FileInfo::getEffectivePath);
        for (int repeat = 0; !pathList.isEmpty() && repeat < 5; ++repeat) {
            try {
                if (repeat > 0 && (pathList = this.getFilesToRetry(sourceFs, srcFileList, destination, isCopyError)).isEmpty()) break;
                LOG.info("Attempt: " + (repeat + 1) + ". Copying files: " + String.valueOf(pathList));
                isCopyError = true;
                this.doCopyOnce(sourceFs, pathList, destination, useRegularCopy, proxyUser, overwrite, copyStatistics);
                isCopyError = false;
                continue;
            }
            catch (IOException e) {
                LOG.info("file operation failed", (Throwable)e);
                if (repeat >= 4 || this.failOnParentExceptionList.stream().anyMatch(k -> k.isAssignableFrom(e.getClass())) || ErrorMsg.REPL_FILE_SYSTEM_OPERATION_RETRY.getMsg().equals(e.getMessage())) break;
                if (e instanceof FileNotFoundException) continue;
                int sleepTime = FileUtils.getSleepTime((int)repeat);
                LOG.info("Sleep for " + sleepTime + " milliseconds before retry " + (repeat + 1));
                try {
                    Thread.sleep(sleepTime);
                }
                catch (InterruptedException timerEx) {
                    LOG.info("sleep interrupted", (Object)timerEx.getMessage());
                }
                sourceFs = pathList.get(0).getFileSystem((Configuration)this.hiveConf);
                this.destinationFs = destination.getFileSystem((Configuration)this.hiveConf);
            }
        }
        if (!pathList.isEmpty()) {
            LOG.error("File copy failed even after several attempts. Files list: " + String.valueOf(pathList));
            throw new IOException(ErrorMsg.REPL_FILE_SYSTEM_OPERATION_RETRY.getMsg());
        }
    }

    private List<Path> getFilesToRetry(FileSystem sourceFs, List<ReplChangeManager.FileInfo> srcFileList, Path destination, boolean isCopyError) throws IOException, HiveFatalException {
        ArrayList<Path> pathList = new ArrayList<Path>();
        for (ReplChangeManager.FileInfo srcFile : srcFileList) {
            if (srcFile.isCopyDone()) continue;
            Path srcPath = srcFile.getEffectivePath();
            Path destPath = new Path(destination, srcPath.getName());
            if (this.exists(this.destinationFs, destination)) {
                if (this.isSourceFileMismatch(sourceFs, srcFile)) {
                    this.delete(this.destinationFs, destPath, true);
                    srcFile.setIsUseSourcePath(false);
                } else if (!isCopyError) {
                    srcFile.setCopyDone(true);
                    continue;
                }
            } else if (this.isSourceFileMismatch(sourceFs, srcFile)) {
                srcFile.setIsUseSourcePath(false);
            }
            srcPath = srcFile.getEffectivePath();
            if (null == srcPath) {
                LOG.error("File copy failed and likely source file is deleted or modified.Source File: " + String.valueOf(srcFile.getSourcePath()));
                throw new HiveFatalException(ErrorMsg.REPL_FILE_MISSING_FROM_SRC_AND_CM_PATH.getMsg());
            }
            if (!srcFile.isUseSourcePath() && !this.exists(sourceFs, srcFile.getCmPath())) {
                LOG.error("File Copy Failed. Both source and CM files are missing from source. Missing Source File: " + String.valueOf(srcFile.getSourcePath()) + ", CM File: " + String.valueOf(srcFile.getCmPath()) + ". Try setting higher value for hive.repl.cm.retain in source warehouse. Also, bootstrap the system again to get back the consistent replicated state.");
                throw new HiveFatalException(ErrorMsg.REPL_FILE_MISSING_FROM_SRC_AND_CM_PATH.getMsg());
            }
            pathList.add(srcPath);
        }
        return pathList;
    }

    public void renameFileCopiedFromCmPath(Path toPath, FileSystem dstFs, List<ReplChangeManager.FileInfo> srcFiles) throws IOException {
        for (ReplChangeManager.FileInfo srcFile : srcFiles) {
            boolean result;
            if (srcFile.isUseSourcePath()) continue;
            String destFileName = srcFile.getCmPath().getName();
            Path destRoot = CopyUtils.getCopyDestination(srcFile, toPath);
            Path destFile = new Path(destRoot, destFileName);
            if (!this.exists(dstFs, destFile)) continue;
            String destFileWithSourceName = srcFile.getSourcePath().getName();
            Path newDestFile = new Path(destRoot, destFileWithSourceName);
            try {
                this.delete(dstFs, newDestFile, true);
                LOG.debug(" file " + String.valueOf(newDestFile) + " is deleted before renaming");
            }
            catch (FileNotFoundException fileNotFoundException) {
                // empty catch block
            }
            if (result = this.rename(dstFs, destFile, newDestFile)) continue;
            throw new IllegalStateException("could not rename " + destFile.getName() + " to " + newDestFile.getName());
        }
    }

    private boolean isSourceFileMismatch(FileSystem sourceFs, ReplChangeManager.FileInfo srcFile) throws IOException {
        String sourceChecksumString;
        this.runTestOnlyExecutions();
        if (srcFile.isUseSourcePath() && (sourceChecksumString = srcFile.getCheckSum()) != null) {
            String verifySourceChecksumString;
            try {
                verifySourceChecksumString = this.checkSumFor(srcFile.getSourcePath(), sourceFs);
            }
            catch (IOException e) {
                LOG.info("Unable to calculate checksum for source file: " + String.valueOf(srcFile.getSourcePath()), (Throwable)e);
                if (!this.exists(sourceFs, srcFile.getSourcePath())) {
                    return true;
                }
                throw e;
            }
            if (!sourceChecksumString.equals(verifySourceChecksumString)) {
                return true;
            }
        }
        return false;
    }

    @VisibleForTesting
    private void runTestOnlyExecutions() throws IOException {
        if (testCallable != null) {
            try {
                testCallable.call();
            }
            catch (Exception e) {
                throw new IOException(e);
            }
        }
    }

    private UserGroupInformation getProxyUser() throws IOException {
        if (this.copyAsUser == null) {
            return null;
        }
        return this.retryableFxn(() -> {
            String currentUser = Utils.getUGI().getShortUserName();
            if (!currentUser.equals(this.copyAsUser)) {
                return UserGroupInformation.createProxyUser((String)this.copyAsUser, (UserGroupInformation)UserGroupInformation.getLoginUser());
            }
            return null;
        });
    }

    private void doCopyOnce(FileSystem sourceFs, List<Path> srcList, Path destination, boolean useRegularCopy, UserGroupInformation proxyUser, boolean overwrite, DataCopyStatistics copyStatistics) throws IOException {
        if (useRegularCopy) {
            this.doRegularCopyOnce(sourceFs, srcList, destination, proxyUser, overwrite, copyStatistics);
        } else {
            this.doDistCpCopyOnce(sourceFs, srcList, destination, proxyUser, copyStatistics);
        }
    }

    private void doDistCpCopyOnce(FileSystem sourceFs, List<Path> srcList, Path destination, UserGroupInformation proxyUser, DataCopyStatistics copyStatistics) throws IOException {
        if (this.hiveConf.getBoolVar(HiveConf.ConfVars.REPL_ADD_RAW_RESERVED_NAMESPACE)) {
            srcList = srcList.stream().map(path -> {
                URI uri = path.toUri();
                return new Path(uri.getScheme(), uri.getAuthority(), RAW_RESERVED_VIRTUAL_PATH + uri.getPath());
            }).collect(Collectors.toList());
            URI destinationUri = destination.toUri();
            destination = new Path(destinationUri.getScheme(), destinationUri.getAuthority(), RAW_RESERVED_VIRTUAL_PATH + destinationUri.getPath());
        }
        if (!FileUtils.distCp((FileSystem)sourceFs, srcList, (Path)destination, (boolean)false, (UserGroupInformation)proxyUser, (HiveConf)this.hiveConf, (HadoopShims)ShimLoader.getHadoopShims())) {
            LOG.error("Distcp failed to copy files: " + String.valueOf(srcList) + " to destination: " + String.valueOf(destination));
            throw new IOException("Distcp operation failed.");
        }
        if (sourceFs.getUri().getScheme().equals("hdfs")) {
            for (Path path2 : srcList) {
                ContentSummary srcContentSummary = sourceFs.getContentSummary(path2);
                copyStatistics.incrementBytesCopiedCounter(srcContentSummary.getLength());
            }
        }
        LOG.info("CopyUtils copied {} number of bytes by using distcp", (Object)copyStatistics.getBytesCopied());
    }

    private void doRegularCopyOnce(FileSystem sourceFs, List<Path> srcList, Path destination, UserGroupInformation proxyUser, boolean overWrite, DataCopyStatistics copyStatistics) throws IOException {
        Path[] paths = srcList.toArray(new Path[0]);
        if (proxyUser != null) {
            Path finalDestination = destination;
            try {
                proxyUser.doAs(() -> {
                    if (overWrite) {
                        this.deleteSubDirs(this.destinationFs, destination);
                    }
                    this.copyFilesBetweenFS(sourceFs, paths, this.destinationFs, finalDestination, false, true, copyStatistics);
                    return true;
                });
            }
            catch (InterruptedException e) {
                throw new IOException(e);
            }
        } else {
            if (overWrite) {
                this.deleteSubDirs(this.destinationFs, destination);
            }
            this.copyFilesBetweenFS(sourceFs, paths, this.destinationFs, destination, false, true, copyStatistics);
        }
    }

    private void deleteSubDirs(FileSystem fs, Path path) throws IOException {
        this.delete(fs, path, true);
        this.mkdirs(fs, path);
    }

    public void doCopy(Path destination, List<Path> srcPaths) throws IOException {
        Map<FileSystem, List<Path>> map = this.fsToPathMap(srcPaths);
        UserGroupInformation proxyUser = this.getProxyUser();
        for (Map.Entry<FileSystem, List<Path>> entry : map.entrySet()) {
            FileSystem sourceFs = entry.getKey();
            List fileList = Lists.transform(entry.getValue(), path -> new ReplChangeManager.FileInfo(sourceFs, path, null));
            this.doCopyOnce(sourceFs, entry.getValue(), destination, this.regularCopy(sourceFs, fileList), proxyUser, false, new DataCopyStatistics());
        }
    }

    boolean regularCopy(FileSystem sourceFs, List<ReplChangeManager.FileInfo> fileList) throws IOException {
        if (this.hiveInReplTest) {
            return true;
        }
        if (this.isLocal(sourceFs) || this.isLocal(this.destinationFs)) {
            return true;
        }
        long size = 0L;
        long numberOfFiles = 0L;
        for (ReplChangeManager.FileInfo fileInfo : fileList) {
            ContentSummary contentSummary;
            block5: {
                contentSummary = null;
                try {
                    contentSummary = this.getContentSummary(sourceFs, fileInfo.getEffectivePath());
                }
                catch (IOException e) {
                    if (!fileInfo.isUseSourcePath() || fileInfo.getCmPath() == null) break block5;
                    contentSummary = this.getContentSummary(sourceFs, fileInfo.getCmPath());
                    fileInfo.setIsUseSourcePath(false);
                }
            }
            if (contentSummary == null || !this.limitReachedForLocalCopy(size += contentSummary.getLength(), numberOfFiles += contentSummary.getFileCount())) continue;
            return false;
        }
        return true;
    }

    boolean limitReachedForLocalCopy(long size, long numberOfFiles) {
        boolean result;
        boolean bl = result = size > this.maxCopyFileSize && numberOfFiles > this.maxNumberOfFiles;
        if (result) {
            LOG.info("Source is {} bytes. (MAX: {})", (Object)size, (Object)this.maxCopyFileSize);
            LOG.info("Source is {} files. (MAX: {})", (Object)numberOfFiles, (Object)this.maxNumberOfFiles);
            LOG.info("going to launch distributed copy (distcp) job.");
        }
        return result;
    }

    private boolean isLocal(FileSystem fs) {
        return fs.getScheme().equals("file");
    }

    private Map<FileSystem, List<Path>> fsToPathMap(List<Path> srcPaths) throws IOException {
        HashMap<FileSystem, List<Path>> result = new HashMap<FileSystem, List<Path>>();
        for (Path path : srcPaths) {
            FileSystem fileSystem = path.getFileSystem((Configuration)this.hiveConf);
            if (!result.containsKey(fileSystem)) {
                result.put(fileSystem, new ArrayList());
            }
            ((List)result.get(fileSystem)).add(path);
        }
        return result;
    }

    private Map<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>> fsToFileMap(List<ReplChangeManager.FileInfo> srcFiles, Path destRoot) throws IOException {
        HashMap<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>> result = new HashMap<FileSystem, Map<Path, List<ReplChangeManager.FileInfo>>>();
        for (ReplChangeManager.FileInfo file : srcFiles) {
            FileSystem fileSystem = file.getSrcFs();
            if (!result.containsKey(fileSystem)) {
                result.put(fileSystem, new HashMap());
            }
            Path destination = CopyUtils.getCopyDestination(file, destRoot);
            if (!((Map)result.get(fileSystem)).containsKey(destination)) {
                ((Map)result.get(fileSystem)).put(destination, new ArrayList());
            }
            ((List)((Map)result.get(fileSystem)).get(destination)).add(file);
        }
        return result;
    }

    public static Path getCopyDestination(ReplChangeManager.FileInfo fileInfo, Path destRoot) {
        if (fileInfo.getSubDir() == null) {
            return destRoot;
        }
        String[] subDirs = fileInfo.getSubDir().split("/");
        Path destination = destRoot;
        for (String subDir : subDirs) {
            if (subDir.startsWith("base_")) {
                AcidUtils.ParsedBaseLight pb = AcidUtils.ParsedBase.parseBase(new Path(subDir));
                subDir = pb.getVisibilityTxnId() > 0L ? AcidUtils.baseDir(pb.getWriteId()) : subDir;
            } else if (subDir.startsWith("delta_")) {
                pdl = AcidUtils.ParsedDeltaLight.parse(new Path(subDir));
                subDir = pdl.getVisibilityTxnId() > 0L ? AcidUtils.deltaSubdir(pdl.getMinWriteId(), pdl.getMaxWriteId()) : subDir;
            } else if (subDir.startsWith("delete_delta_")) {
                pdl = AcidUtils.ParsedDeltaLight.parse(new Path(subDir));
                subDir = pdl.getVisibilityTxnId() > 0L ? AcidUtils.deleteDeltaSubdir(pdl.getMinWriteId(), pdl.getMaxWriteId()) : subDir;
            }
            destination = new Path(destination, subDir);
        }
        return destination;
    }
}

