/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.controller.rebalancer.waged.constraints;

import com.google.common.collect.Maps;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.apache.helix.HelixRebalanceException;
import org.apache.helix.controller.rebalancer.waged.RebalanceAlgorithm;
import org.apache.helix.controller.rebalancer.waged.constraints.HardConstraint;
import org.apache.helix.controller.rebalancer.waged.constraints.SoftConstraint;
import org.apache.helix.controller.rebalancer.waged.model.AssignableNode;
import org.apache.helix.controller.rebalancer.waged.model.AssignableReplica;
import org.apache.helix.controller.rebalancer.waged.model.ClusterContext;
import org.apache.helix.controller.rebalancer.waged.model.ClusterModel;
import org.apache.helix.controller.rebalancer.waged.model.OptimalAssignment;
import org.apache.helix.model.ResourceAssignment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ConstraintBasedAlgorithm
implements RebalanceAlgorithm {
    private static final float DIV_GUARD = 0.01f;
    private static final Logger LOG = LoggerFactory.getLogger(ConstraintBasedAlgorithm.class);
    private final List<HardConstraint> _hardConstraints;
    private final Map<SoftConstraint, Float> _softConstraints;

    ConstraintBasedAlgorithm(List<HardConstraint> hardConstraints, Map<SoftConstraint, Float> softConstraints) {
        this._hardConstraints = hardConstraints;
        this._softConstraints = softConstraints;
    }

    @Override
    public OptimalAssignment calculate(ClusterModel clusterModel) throws HelixRebalanceException {
        OptimalAssignment optimalAssignment = new OptimalAssignment();
        ArrayList<AssignableNode> nodes = new ArrayList<AssignableNode>(clusterModel.getAssignableNodes().values());
        Set<String> busyInstances = this.getBusyInstances(clusterModel.getContext().getBestPossibleAssignment().values());
        HashMap<String, Float> positiveEstimateClusterRemainCap = new HashMap<String, Float>();
        for (Map.Entry<String, Integer> clusterRemainingCap : clusterModel.getContext().getEstimateUtilizationMap().entrySet()) {
            String capacityKey = clusterRemainingCap.getKey();
            if (clusterRemainingCap.getValue() < 0) {
                throw new HelixRebalanceException(String.format("The cluster does not have enough %s capacity for all partitions. ", capacityKey), HelixRebalanceException.Type.FAILED_TO_CALCULATE);
            }
            positiveEstimateClusterRemainCap.put(capacityKey, Float.valueOf((float)clusterModel.getContext().getEstimateUtilizationMap().get(capacityKey).intValue() + (float)clusterModel.getContext().getClusterCapacityMap().get(capacityKey).intValue() * 0.01f));
        }
        List toBeAssignedReplicas = clusterModel.getAssignableReplicaMap().values().stream().flatMap(Collection::stream).map(replica -> new AssignableReplicaWithScore((AssignableReplica)replica, clusterModel, (Map<String, Float>)positiveEstimateClusterRemainCap)).sorted().collect(Collectors.toList());
        for (AssignableReplicaWithScore replicaWithScore : toBeAssignedReplicas) {
            AssignableReplica replica2 = replicaWithScore.getAssignableReplica();
            Optional<AssignableNode> maybeBestNode = this.getNodeWithHighestPoints(replica2, nodes, clusterModel.getContext(), busyInstances, optimalAssignment);
            if (!maybeBestNode.isPresent() || optimalAssignment.hasAnyFailure()) {
                String errorMessage = String.format("Unable to find any available candidate node for partition %s; Fail reasons: %s", replica2.getPartitionName(), optimalAssignment.getFailures());
                throw new HelixRebalanceException(errorMessage, HelixRebalanceException.Type.FAILED_TO_CALCULATE);
            }
            AssignableNode bestNode = maybeBestNode.get();
            clusterModel.assign(replica2.getResourceName(), replica2.getPartitionName(), replica2.getReplicaState(), bestNode.getInstanceName());
        }
        optimalAssignment.updateAssignments(clusterModel);
        return optimalAssignment;
    }

    private Optional<AssignableNode> getNodeWithHighestPoints(AssignableReplica replica, List<AssignableNode> assignableNodes, ClusterContext clusterContext, Set<String> busyInstances, OptimalAssignment optimalAssignment) {
        ConcurrentHashMap hardConstraintFailures = new ConcurrentHashMap();
        List candidateNodes = assignableNodes.parallelStream().filter(candidateNode -> {
            boolean isValid = true;
            for (HardConstraint hardConstraint : this._hardConstraints) {
                if (hardConstraint.isAssignmentValid((AssignableNode)candidateNode, replica, clusterContext)) continue;
                hardConstraintFailures.computeIfAbsent(candidateNode, node -> new ArrayList()).add(hardConstraint);
                isValid = false;
            }
            return isValid;
        }).collect(Collectors.toList());
        if (candidateNodes.isEmpty()) {
            LOG.info("Found no eligible candidate nodes. Enabling hard constraint level logging for cluster: {}", (Object)clusterContext.getClusterName());
            this.enableFullLoggingForCluster();
            optimalAssignment.recordAssignmentFailure(replica, Maps.transformValues(hardConstraintFailures, this::convertFailureReasons));
            return Optional.empty();
        }
        LOG.info("Disabling hard constraint level logging for cluster: {}", (Object)clusterContext.getClusterName());
        this.removeFullLoggingForCluster();
        return candidateNodes.parallelStream().map(node -> new AbstractMap.SimpleEntry<AssignableNode, Double>((AssignableNode)node, this.getAssignmentNormalizedScore((AssignableNode)node, replica, clusterContext))).max((nodeEntry1, nodeEntry2) -> {
            int scoreCompareResult = ((Double)nodeEntry1.getValue()).compareTo((Double)nodeEntry2.getValue());
            if (scoreCompareResult == 0) {
                String logicalId1 = ((AssignableNode)nodeEntry1.getKey()).getLogicalId();
                String logicalId2 = ((AssignableNode)nodeEntry2.getKey()).getLogicalId();
                int idleScore1 = busyInstances.contains(logicalId1) ? 0 : 1;
                int idleScore2 = busyInstances.contains(logicalId2) ? 0 : 1;
                return idleScore1 != idleScore2 ? idleScore1 - idleScore2 : -((AssignableNode)nodeEntry1.getKey()).compareTo((AssignableNode)nodeEntry2.getKey());
            }
            return scoreCompareResult;
        }).map(Map.Entry::getKey);
    }

    private double getAssignmentNormalizedScore(AssignableNode node, AssignableReplica replica, ClusterContext clusterContext) {
        double sum = 0.0;
        for (Map.Entry<SoftConstraint, Float> softConstraintEntry : this._softConstraints.entrySet()) {
            SoftConstraint softConstraint = softConstraintEntry.getKey();
            float weight = softConstraintEntry.getValue().floatValue();
            if (weight == 0.0f) continue;
            sum += (double)weight * softConstraint.getAssignmentNormalizedScore(node, replica, clusterContext);
        }
        return sum;
    }

    private List<String> convertFailureReasons(List<HardConstraint> hardConstraints) {
        return hardConstraints.stream().map(HardConstraint::getDescription).collect(Collectors.toList());
    }

    private void enableFullLoggingForCluster() {
        for (HardConstraint hardConstraint : this._hardConstraints) {
            hardConstraint.setEnableLogging(true);
        }
    }

    private void removeFullLoggingForCluster() {
        for (HardConstraint hardConstraint : this._hardConstraints) {
            hardConstraint.setEnableLogging(false);
        }
    }

    private Set<String> getBusyInstances(Collection<ResourceAssignment> assignments) {
        return assignments.stream().flatMap(resourceAssignment -> resourceAssignment.getRecord().getMapFields().values().stream().flatMap(instanceStateMap -> instanceStateMap.keySet().stream()).collect(Collectors.toSet()).stream()).collect(Collectors.toSet());
    }

    private static class AssignableReplicaWithScore
    implements Comparable<AssignableReplicaWithScore> {
        private final AssignableReplica _replica;
        private float _score = 0.0f;
        private final boolean _isInBestPossibleAssignment;
        private final boolean _isInBaselineAssignment;
        private final Integer _replicaHash;

        AssignableReplicaWithScore(AssignableReplica replica, ClusterModel clusterModel, Map<String, Float> overallClusterRemainingCapacityMap) {
            this._replica = replica;
            this._isInBestPossibleAssignment = clusterModel.getContext().getBestPossibleAssignment().containsKey(replica.getResourceName());
            this._isInBaselineAssignment = clusterModel.getContext().getBaselineAssignment().containsKey(replica.getResourceName());
            this._replicaHash = Objects.hash(replica.toString(), clusterModel.getAssignableLogicalIds());
            this.computeScore(overallClusterRemainingCapacityMap);
        }

        public void computeScore(Map<String, Float> positiveEstimateClusterRemainCap) {
            float score = 0.0f;
            for (Map.Entry<String, Integer> resourceCapacity : this._replica.getCapacity().entrySet()) {
                String capacityKey = resourceCapacity.getKey();
                score += (float)resourceCapacity.getValue().intValue() / positiveEstimateClusterRemainCap.get(capacityKey).floatValue();
            }
            this._score = score;
        }

        public AssignableReplica getAssignableReplica() {
            return this._replica;
        }

        public String toString() {
            return this._replica.toString();
        }

        @Override
        public int compareTo(AssignableReplicaWithScore replica2) {
            int statePriority2;
            if (this._isInBestPossibleAssignment != replica2._isInBestPossibleAssignment) {
                return this._isInBestPossibleAssignment ? -1 : 1;
            }
            if (this._isInBaselineAssignment != replica2._isInBaselineAssignment) {
                return this._isInBaselineAssignment ? -1 : 1;
            }
            int statePriority1 = this._replica.getStatePriority();
            if (statePriority1 != (statePriority2 = replica2._replica.getStatePriority())) {
                return statePriority1 - statePriority2;
            }
            int result = Float.compare(replica2._score, this._score);
            if (result != 0) {
                return result;
            }
            if (!this._replicaHash.equals(replica2._replicaHash)) {
                return this._replicaHash.compareTo(replica2._replicaHash);
            }
            return this._replica.toString().compareTo(replica2.toString());
        }
    }
}

