--
-- Licensed to the Apache Software Foundation (ASF) under one
-- or more contributor license agreements.  See the NOTICE file--
--  distributed with this work for additional information
-- regarding copyright ownership.  The ASF licenses this file
-- to you under the Apache License, Version 2.0 (the
-- "License"). You may not use this file except in compliance
-- with the License.  You may obtain a copy of the License at
--
--  http://www.apache.org/licenses/LICENSE-2.0
--
-- Unless required by applicable law or agreed to in writing,
-- software distributed under the License is distributed on an
-- "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
-- KIND, either express or implied.  See the License for the
-- specific language governing permissions and limitations
-- under the License.
--

CREATE TABLE IF NOT EXISTS policy_meta (
    policy_id BIGINT NOT NULL,
    policy_name VARCHAR(128) NOT NULL,
    policy_type VARCHAR(64) NOT NULL,
    metalake_id BIGINT NOT NULL,
    audit_info TEXT NOT NULL,
    current_version INT NOT NULL DEFAULT 1,
    last_version INT NOT NULL DEFAULT 1,
    deleted_at BIGINT NOT NULL DEFAULT 0,
    PRIMARY KEY (policy_id),
    UNIQUE (metalake_id, policy_name, deleted_at)
);

COMMENT ON TABLE policy_meta IS 'policy metadata';
COMMENT ON COLUMN policy_meta.policy_id IS 'policy id';
COMMENT ON COLUMN policy_meta.policy_name IS 'policy name';
COMMENT ON COLUMN policy_meta.policy_type IS 'policy type';
COMMENT ON COLUMN policy_meta.metalake_id IS 'metalake id';
COMMENT ON COLUMN policy_meta.audit_info IS 'policy audit info';
COMMENT ON COLUMN policy_meta.current_version IS 'policy current version';
COMMENT ON COLUMN policy_meta.last_version IS 'policy last version';
COMMENT ON COLUMN policy_meta.deleted_at IS 'policy deleted at';


CREATE TABLE IF NOT EXISTS policy_version_info (
    id BIGINT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    metalake_id BIGINT NOT NULL,
    policy_id BIGINT NOT NULL,
    version INT NOT NULL,
    policy_comment TEXT DEFAULT NULL,
    enabled BOOLEAN DEFAULT TRUE,
    content TEXT DEFAULT NULL,
    deleted_at BIGINT NOT NULL DEFAULT 0,
    PRIMARY KEY (id),
    UNIQUE (policy_id, version, deleted_at)
);

CREATE INDEX IF NOT EXISTS policy_version_info_idx_metalake_id ON policy_version_info (metalake_id);
COMMENT ON TABLE policy_version_info IS 'policy version info';
COMMENT ON COLUMN policy_version_info.id IS 'auto increment id';
COMMENT ON COLUMN policy_version_info.metalake_id IS 'metalake id';
COMMENT ON COLUMN policy_version_info.policy_id IS 'policy id';
COMMENT ON COLUMN policy_version_info.version IS 'policy info version';
COMMENT ON COLUMN policy_version_info.policy_comment IS 'policy info comment';
COMMENT ON COLUMN policy_version_info.enabled IS 'whether the policy is enabled, 0 is disabled, 1 is enabled';
COMMENT ON COLUMN policy_version_info.content IS 'policy content';
COMMENT ON COLUMN policy_version_info.deleted_at IS 'policy deleted at';


CREATE TABLE IF NOT EXISTS policy_relation_meta (
    id BIGINT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    policy_id BIGINT NOT NULL,
    metadata_object_id BIGINT NOT NULL,
    metadata_object_type VARCHAR(64) NOT NULL,
    audit_info TEXT NOT NULL,
    current_version INT NOT NULL DEFAULT 1,
    last_version INT NOT NULL DEFAULT 1,
    deleted_at BIGINT NOT NULL DEFAULT 0,
    PRIMARY KEY (id),
    UNIQUE (policy_id, metadata_object_id, metadata_object_type, deleted_at)
);

CREATE INDEX IF NOT EXISTS policy_relation_meta_idx_policy_id ON policy_relation_meta (policy_id);
CREATE INDEX IF NOT EXISTS policy_relation_meta_idx_metadata_object_id ON policy_relation_meta (metadata_object_id);
COMMENT ON TABLE policy_relation_meta IS 'policy metadata object relation';
COMMENT ON COLUMN policy_relation_meta.id IS 'auto increment id';
COMMENT ON COLUMN policy_relation_meta.policy_id IS 'policy id';
COMMENT ON COLUMN policy_relation_meta.metadata_object_id IS 'metadata object id';
COMMENT ON COLUMN policy_relation_meta.metadata_object_type IS 'metadata object type';
COMMENT ON COLUMN policy_relation_meta.audit_info IS 'policy relation audit info';
COMMENT ON COLUMN policy_relation_meta.current_version IS 'policy relation current version';
COMMENT ON COLUMN policy_relation_meta.last_version IS 'policy relation last version';
COMMENT ON COLUMN policy_relation_meta.deleted_at IS 'policy relation deleted at';

-- using default 'unknown' to fill in the new column for compatibility
ALTER TABLE model_version_info ADD COLUMN model_version_uri_name VARCHAR(256) NOT NULL DEFAULT 'unknown';
COMMENT ON COLUMN model_version_info.model_version_uri_name IS 'model version uri name';
ALTER TABLE model_version_info DROP CONSTRAINT model_version_info_model_id_version_deleted_at_key;
ALTER TABLE model_version_info ADD CONSTRAINT uk_mid_ver_uri_del UNIQUE (model_id, version, model_version_uri_name, deleted_at);
-- remove the default value for model_version_uri_name
ALTER TABLE model_version_info ALTER COLUMN model_version_uri_name DROP DEFAULT;

CREATE TABLE IF NOT EXISTS statistic_meta (
    id BIGINT NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    statistic_id BIGINT NOT NULL,
    statistic_name VARCHAR(128) NOT NULL,
    metalake_id BIGINT NOT NULL,
    statistic_value TEXT NOT NULL,
    metadata_object_id BIGINT NOT NULL,
    metadata_object_type VARCHAR(64) NOT NULL,
    PRIMARY KEY (statistic_id),
    UNIQUE (statistic_name, metadata_object_id, deleted_at)
);

CREATE INDEX IF NOT EXISTS policy_relation_meta_idx_stid ON statistic_meta (statistic_id);
CREATE INDEX IF NOT EXISTS policy_relation_meta_idx_moid ON statistic_meta (metadata_object_id);
COMMENT ON TABLE statistic_meta IS 'statistic metadata';
COMMENT ON COLUMN statistic_meta.id IS 'auto increment id';
COMMENT ON COLUMN statistic_meta.statistic_id IS 'statistic id';
COMMENT ON COLUMN statistic_meta.statistic_name IS 'statistic name';
COMMENT ON COLUMN statistic_meta.metalake_id IS 'metalake id';
COMMENT ON COLUMN statistic_meta.statistic_value IS 'statistic value';
COMMENT ON COLUMN statistic_meta.metadata_object_id IS 'metadata object id';
COMMENT ON COLUMN statistic_meta.metadata_object_type IS 'metadata object type';
COMMENT ON COLUMN statistic_meta.audit_info IS 'statistic audit info';
COMMENT ON COLUMN statistic_meta.current_version IS 'statistic current version';
COMMENT ON COLUMN statistic_meta.last_version IS 'statistic last version';
COMMENT ON COLUMN statistic_meta.deleted_at IS 'statistic deleted at';

CREATE TABLE IF NOT EXISTS job_template_meta (
    job_template_id BIGINT NOT NULL,
    job_template_name VARCHAR(128) NOT NULL,
    metalake_id BIGINT NOT NULL,
    job_template_comment TEXT DEFAULT NULL,
    job_template_content TEXT NOT NULL,
    audit_info TEXT NOT NULL,
    current_version INT NOT NULL DEFAULT 1,
    last_version INT NOT NULL DEFAULT 1,
    deleted_at BIGINT NOT NULL DEFAULT 0,
    PRIMARY KEY (job_template_id),
    UNIQUE (metalake_id, job_template_name, deleted_at)
);

COMMENT ON TABLE job_template_meta IS 'job template metadata';
COMMENT ON COLUMN job_template_meta.job_template_id IS 'job template id';
COMMENT ON COLUMN job_template_meta.job_template_name IS 'job template name';
COMMENT ON COLUMN job_template_meta.metalake_id IS 'metalake id';
COMMENT ON COLUMN job_template_meta.job_template_comment IS 'job template comment';
COMMENT ON COLUMN job_template_meta.job_template_content IS 'job template content';
COMMENT ON COLUMN job_template_meta.audit_info IS 'job template audit info';
COMMENT ON COLUMN job_template_meta.current_version IS 'job template current version';
COMMENT ON COLUMN job_template_meta.last_version IS 'job template last version';
COMMENT ON COLUMN job_template_meta.deleted_at IS 'job template deleted at';


CREATE TABLE IF NOT EXISTS job_run_meta (
    job_run_id BIGINT NOT NULL,
    job_template_id BIGINT NOT NULL,
    metalake_id BIGINT NOT NULL,
    job_execution_id VARCHAR(256) NOT NULL,
    job_run_status VARCHAR(64) NOT NULL,
    job_finished_at BIGINT NOT NULL DEFAULT 0,
    audit_info TEXT NOT NULL,
    current_version INT NOT NULL DEFAULT 1,
    last_version INT NOT NULL DEFAULT 1,
    deleted_at BIGINT NOT NULL DEFAULT 0,
    PRIMARY KEY (job_run_id),
    UNIQUE (metalake_id, job_execution_id, deleted_at)
);

CREATE INDEX IF NOT EXISTS policy_relation_meta_idx_job_template_id ON job_run_meta (job_template_id);
CREATE INDEX IF NOT EXISTS policy_relation_meta_idx_job_execution_id ON job_run_meta (job_execution_id);
COMMENT ON TABLE job_run_meta IS 'job run metadata';
COMMENT ON COLUMN job_run_meta.job_run_id IS 'job run id';
COMMENT ON COLUMN job_run_meta.job_template_id IS 'job template id';
COMMENT ON COLUMN job_run_meta.metalake_id IS 'metalake id';
COMMENT ON COLUMN job_run_meta.job_execution_id IS 'job execution id';
COMMENT ON COLUMN job_run_meta.job_run_status IS 'job run status';
COMMENT ON COLUMN job_run_meta.job_finished_at IS 'job finished at';
COMMENT ON COLUMN job_run_meta.audit_info IS 'job run audit info';
COMMENT ON COLUMN job_run_meta.current_version IS 'job run current version';
COMMENT ON COLUMN job_run_meta.last_version IS 'job run last version';
COMMENT ON COLUMN job_run_meta.deleted_at IS 'job run deleted at';
